/*! @file
 * @brief
 * tool to validate relation files for the current factorization
 */


#include "at_startup.H"
#include <fstream>
#include <sstream>
#include <gmp.h>
#include "qsieve.H"
#include "StaticFactorbase.H"

using namespace std;

const string RecoveryFile         = "recovery.dat";
const string StaticRelationsFile  = "static_relations.dat";
const string SpecialRelationsFile = "special_relations.dat";


// for Recovery
filebuf Recovery_buffer;
ostream Recovery_to_file (&Recovery_buffer);
istream Recovery_from_file (&Recovery_buffer);


mpz_t n, // number to factorize (will be reduced during factorization)
      kN; // input for MPQS (includes a suitable multiplier)



#include "StaticRelations.H"
#include "Sieving.H"
CRelation* StaticRelations::GLS[StaticFactorbase::MaxSize] = {NULL};
int StaticRelations::Filling_GLS = 0;


// dummy definitions to satisfy static linker:
CmpqsFactor CRelation::combine(istream&)
{
  return CmpqsFactor();
}

CmpqsFactor CRelation::multi_combine_main(istream &)
{
  return CmpqsFactor();
}


#include "ConfigFile.cc"


//int main(const int argc, const char* const argv[])
int main(const int argc, const char* const *)
{

  PrintHeader("Qsieve relation validator");
  cout.setf(ios::fixed); // fixed decimal notation, not scientific notation!

  if (argc!=1) exit(1);

  mpz_init(n); // our number to factorize
  mpz_init(kN);

  Read_ConfigFile();

  Recovery_buffer.open(RecoveryFile.c_str(),ios::in|ios::out|ios::ate);
  // offenbar wird trotz ios::ate nicht (immer) ans Ende positioniert
  // deshalb wird die Testabfrage modifiziert:
  if (Recovery_from_file) Recovery_from_file.seekg(0,ios::end);
#ifdef STL_STREAM_workaround
  if ( (!Recovery_from_file) || (Recovery_from_file.tellg()==std::streampos(0)) || (Recovery_from_file.tellg()==std::streampos(-1)) )
// tellg()==0 indicates empty file -> we cannot recover
// tellg()==-1 indicates failure -> we cannot recover
// remark:
//  in gcc 3.4 (cvs-experimental-2003-10-17 we cannot compare with "<" !!)
//  do we really need a workaround to check this condition? 
#else
  if ( (!Recovery_from_file) || (Recovery_from_file.tellg()<1) )
#endif /* STL_STREAM_workaround */
    {
      cerr << "No recovery data found. Cannot validate!" << endl;
      exit(1);
    }
      
  Recovery_from_file.seekg(0,ios::beg);
  Recovery_to_file.seekp(0,ios::beg);
  Recovery_from_file >> n; // retrieve number
  Recovery_from_file.ignore(1,'\n');

  cout << "Checking relations for" << endl;
  cout << n << endl;

  determine_best_MPQS_Multiplier(n,kN,MPQS_Multiplier); // calculate a good/optimal value for fastest possible sieving
  tune_parameters(mpz_sizeinbase(n,10)); // tune parameters for n

#ifdef VERBOSE_NOTICE
  cout << "n=" << n << endl;
  cout << "Size of static factorbase: " << StaticFactorbase::Size() << endl;
  cout << "Factor-Threshold (Exponent): " << Factor_Threshold << endl;
  cout << "Sieveinterval per polynomial: [" << -LogicalSieveSize << "," << LogicalSieveSize << "]" << endl;
#endif

  if ( sqrt(mpz_get_d(kN)) < PhysicalSieveSize )
    {
      cerr << "Sieve size too big (you may want to reduce its size)!" << endl;
      exit(1);
    }
  
  StaticFactorbase::compute_StaticFactorbase();

  // are all the relations valid?
  CRelation::SanityCheckRelationsFile(StaticRelationsFile);
  CRelation::SanityCheckRelationsFile(DynamicRelationsFile);
  CRelation::SanityCheckRelationsFile(SpecialRelationsFile);
  return 0;
}
