#ifndef UNIX_BUFFER_
#define UNIX_BUFFER_

// some abstraction layers for using UNIX-network services in C++ (mainly TCP)
// written by Thorsten Reinecke (reinecke@thorstenreinecke.de)
// last change: 2005-06-05

/*! @file
 * @brief
 * some abstraction layers for using UNIX-network services in C++ (mainly TCP)
 */


#ifdef CYGWIN_COMPAT
 #warning cygwin compatibility hacks enabled
#endif

extern "C"
{
  #include <sys/socket.h>
  #include <sys/poll.h>
  #include <netinet/in.h>
  #include <netinet/tcp.h>
  #include <netdb.h>
  #include <unistd.h>

 #if defined(_REENTRANT) || defined(_THREAD_SAFE)
  // POSIX specifies, that (blocking) system calls are cancellation points,
  // but LinuxThreads may not be posix-conform to this issue,
  // (refer manpage PTHREAD_CANCEL(3))
  // so PTHREAD_TESTCANCEL is called explicitly at some important points
  #include <pthread.h>
  #define PTHREAD_TESTCANCEL pthread_testcancel(); /* explicit thread cancellation point */
 #else
  #define PTHREAD_TESTCANCEL /* this would be an explicit thread cancellation point */
 #endif 

}


// I had some problems while debugging under valgrind; this one
// hacks the problems away (but maybe not the bugs...)
//#define WEIRD_VALGRIND_HACK

#ifdef WEIRD_VALGRIND_HACK
 #if !( defined(_REENTRANT) || defined(_THREAD_SAFE) )
  // for single-threaded versions the hack needs to be disabled
  #warning "weird valgrind hack disabled"
  #undef WEIRD_VALGRIND_HACK
 #endif
#endif 

#include <cerrno>
#include <string>
#include <iostream>
#include <sstream>
#include <exception>

#ifdef WEIRD_VALGRIND_HACK
 #warning "weird valgrind hack enabled"
 #include "mutex.H"
#endif

#if defined(CYGWIN_COMPAT)
 #if defined(_REENTRANT) || defined(_THREAD_SAFE)
  #include "mutex.H"
  static CMutex cygwin_mutex;
  #define CYGW_LOCK cygwin_mutex.lock();
  #define CYGW_UNLOCK cygwin_mutex.unlock();
 #else
  #define CYGW_LOCK /* cygwin_mutex.lock(); */
  #define CYGW_UNLOCK /* cygwin_mutex.unlock(); */
 #endif
#endif   


using std::cerr;
using std::endl;
using std::flush;
using std::iostream;
using std::streambuf;


//! This class provides basic exception handling
class unix_buffer_exception : public std::exception
{
 private:
  const std::string Message;

 public:
  explicit unix_buffer_exception(const std::string& Msg) throw()
   : Message(Msg)
   {
#if defined(VERBOSE) || defined(DEBUG)
     cerr << "throwing exception: " << Msg << endl;
#endif
   }
 
  virtual ~unix_buffer_exception() throw() { }
  
  virtual const char* what() const throw()
   {
     return Message.c_str();
   }  
};

//! threadsafe version of strerror that returns error message as string
inline std::string my_strerror(const int errnum)
{
  char buffer[1024];

  // please note: not the local array "buffer", but an implicitly constructed
  // std::string(buffer) is returned by this function!

#if 0
  // this is POSIX conform, but does not compile on my system:
  // POSIX -> int strerror_r(int errnum, char *buf, size_t n);
  if (strerror_r(errnum,buffer,sizeof(buffer))==-1) return buffer;
  else return "";
#else
  // this is using the GNU extension and compile on my system:
  // (see "man 3 strerror")
  char *ret = strerror_r(errnum,buffer,sizeof(buffer));
  if (ret) return ret; else return "";
#endif
}



/*!
 * @param port port to listen for a tcp based network connection
 * @param qsize optional "backlog" parameter (refer "listen(2)" manpage)
 * @result socket descriptor for the connection, or -1 on error
 *
 * Use this function to wait for incoming tcp connection requests
 * and for handling the serversided part to establish the network connection.
 */ 
int open_internet_port(const int port, const int qsize = 5)
{
  int sd;
  int val = 1;
  struct sockaddr_in server;
  struct linger lingval;

  if (port <= 0) return -1;

  sd = socket(AF_INET, SOCK_STREAM, 0);
  if (sd < 0) return sd;
  if (setsockopt(sd, SOL_SOCKET, SO_REUSEADDR, &val, sizeof(val)) <0)
    return -1;

  // activate linger option
  lingval.l_onoff=1; // if applicable, keep socket unavailable after closing
  lingval.l_linger=10*100; // timeout = 10 seconds
  if (setsockopt(sd, SOL_SOCKET, SO_LINGER, &lingval, sizeof(lingval)) <0) return -1;

  server.sin_family = AF_INET;
  server.sin_addr.s_addr = INADDR_ANY;
  server.sin_port = htons(port);
  if (bind(sd, reinterpret_cast<struct sockaddr *>(&server), sizeof(server)) < 0) return -1;
  listen (sd, qsize);
  return sd;
}



// utils for verbose output of meta-data to streams

//! returns information about the specified socket to the given stream
inline std::ostream& operator<< (std::ostream& ostr, const sockaddr &sa)
{
#ifdef CYGWIN_COMPAT
  // implement the obsolete stuff using gethostbyaddr
  CYGW_LOCK;
  u_int32_t ip = reinterpret_cast<const sockaddr_in& >(sa).sin_addr.s_addr;
  struct hostent *p_hostent = gethostbyaddr(reinterpret_cast<const char*>(&ip),sizeof(ip),AF_INET);
   // p_hostent is a static resource and therefore not threadsafe!
  if (p_hostent)  ostr << p_hostent->h_name;
  else ostr << "(unknown)";
  CYGW_UNLOCK;
#else
  // outputs sockaddr info to given stream
  char host[1024];
  char service[1024];
  const int flags = NI_NOFQDN;
//  FIXME! sometimes hostname lookup segfaults on my system (when dynamically linked)
// this works better: const int flags = NI_NOFQDN | NI_NUMERICHOST | NI_NUMERICSERV;
// is this not threadsafe?
  const int retval = getnameinfo(&sa,sizeof(sa),host,sizeof(host),service,sizeof(service),flags);

  switch (retval)
  {
    case 0:
     ostr << host << " " << service;
     break;
    case EAI_AGAIN:
     ostr << "(The name could not be resolved at this time.)"; break;
    case EAI_BADFLAGS:
     ostr << "(The flags parameter has an invalid value.)"; break;
    case EAI_FAIL:
     ostr << "(A non-recoverable error occurred.)"; break;
    case EAI_FAMILY: 
     ostr << "(The address family was not recognized.)"; break;
    case EAI_MEMORY:
     ostr << "(Out of memory.)"; break;
    case EAI_NONAME:
     ostr << "(The  name  does  not  resolve.)"; break;
    case EAI_SYSTEM:
     ostr << "(A system error occurred.)"; break;
    default:
     ostr << "(unspecified error in getnameinfo.)"; 
   }
#endif
  return ostr;
}

/*!
 * @param socket_descriptor a socket descriptor
 * @result string that contains human readable information
 *         about the client who has initiated the network connection
 */
inline std::string peer_info(const int socket_descriptor)
{
 // outputs meta-data for socket_descriptor
 std::ostringstream ostr;

 struct sockaddr sa;
 socklen_t sl = sizeof(sa);

 const int retval = getpeername(socket_descriptor, &sa, &sl);
 if (retval==0) ostr << sa;
 return ostr.str();
}

/*!
 * @param socket_descriptor a socket descriptor
 * @result string that contains human readable information
 *         about the server who has "picked up" the network connection
 */
inline std::string socket_info(const int socket_descriptor)
{
 // outputs meta-data for socket_descriptor
 std::ostringstream ostr;

 struct sockaddr sa;
 socklen_t sl = sizeof(sa);

 const int retval = getsockname(socket_descriptor, &sa, &sl);
 if (retval==0) ostr << sa;
 return ostr.str();
}

/*!
 * @param socket_descriptor a socket descriptor
 * @result string that contains human readable information
 *         about the client who has initiated the network connection
 *         and the server who has "picked up" the network connection
 */
inline std::string connection_info(const int socket_descriptor)
{
 // outputs meta-data for socket_descriptor
 std::ostringstream ostr;

 struct sockaddr sa;
 socklen_t sl = sizeof(sa);

 const int retval1 = getpeername(socket_descriptor, &sa, &sl);
 ostr << "connection from ";
 switch (retval1)
  {
    case 0       : ostr << sa; break;
    case EBADF   : ostr << "(invalid descriptor)"; break;
    case ENOTSOCK: ostr << "(file, not a socket)"; break;
    default : ostr << "(unable to perform getpeername)";
  }

 const int retval2 = getsockname(socket_descriptor, &sa, &sl);
 ostr << " to ";
 switch (retval2)
  {
    case 0       : ostr << sa; break;
    case EBADF   : ostr << "(invalid descriptor)"; break;
    case ENOTSOCK: ostr << "(file, not a socket)"; break;
    default : ostr << "(unable to perform getsockname)";
  }
 return ostr.str();
}

/*!
 * @param obj an object that handles a network connection and provides
 *            the attached socket descriptor through @p get_descriptor()
 * @result string that contains human readable information
 *         about the client who has initiated the network connection
 */
template<class T> inline std::string peer_info(const T& obj)
{
 // outputs connection meta-data for class object to string
 return peer_info(obj.get_descriptor());
}

/*!
 * @param obj an object that handles a network connection and provides
 *            the attached socket descriptor through @p get_descriptor()
 * @result string that contains human readable information
 *         about the server who has "picked up" the network connection
 */
template<class T> inline std::string socket_info(const T& obj)
{
 // outputs connection meta-data for class object to string
 return socket_info(obj.get_descriptor());
}

/*!
 * @param obj an object that handles a network connection and provides
 *            the attached socket descriptor through @p get_descriptor()
 * @result string that contains human readable information
 *         about the client who has initiated the network connection
 *         and the server who has "picked up" the network connection
 */
template<class T> inline std::string connection_info(const T& obj)
{
 // outputs connection meta-data for class object to string
 return connection_info(obj.get_descriptor());
}


//! a virtual base class to provide access to a file/socket descriptor
class Cprovide_descriptor_access
{
 protected:
  virtual const int _get_descriptor() const = 0;
 public:
  virtual ~Cprovide_descriptor_access() { }
};


/*!
 * @brief
 * virtual abstraction layer for the poll system call
 *
 * poll - wait for some event on a file descriptor;
 * refer manpage: Linux Programmers's Manual, POLL(2)
 *
 * This class wraps the linux system call "poll" and provides methods
 * to test whether a stream is readable, writeable and connected/alive.
 */
class Cpoll_methods : public Cprovide_descriptor_access
{
public:
  virtual ~Cpoll_methods() {};

  /*! tiny poll wrapper
   * @param requested_events (refer manpage POLL(2))
   * @param timeout_ms timeout in milliseconds (default 1)
   * @result -1 on error, or @p result_events otherwise
   */
  const int poll_events(short requested_events, const int timeout_ms=1) const
   {
     pollfd pfd;
     int res=0;
     do
      {
        pfd.fd=_get_descriptor(); pfd.events=requested_events; pfd.revents=0;
        res=poll(&pfd,1,timeout_ms);
        if (res==-1) if (errno!=EINTR) return -1;
      } while (res==-1);
     return pfd.revents;
   }

 //! is the given stream in a bad mode?
 inline const bool bad() const
  {
    const int revents=poll_events(POLLIN);
    return (revents==-1) | (revents&(POLLERR|POLLHUP|POLLNVAL));
  }

 //! is the given stream in a good mode?
 inline const bool good() const
  {
    return !bad();
  }

 //! how many chars are readable on the stream within the a given time?
 const int readable_chars_within(const int size, const int timeout_ms=1000) const
  {
    const int revents=poll_events(POLLIN,timeout_ms);
    if (revents==-1) return -1;
    if (revents&(POLLERR|POLLHUP|POLLNVAL)) return -1;
    if (revents&POLLIN)
     {
       char *buf = new char[size];
       int count = recv(_get_descriptor(), buf, size, MSG_PEEK|MSG_NOSIGNAL);
       delete [] buf;
       return count;
     }
    return -1;
  };

 //! is the stream writable without blocking?
 const bool writable_now() const
  {
    const int revents=poll_events(POLLOUT);
    if (revents==-1) return false;
    if (revents&(POLLERR|POLLHUP|POLLNVAL)) return false;
    return (revents&POLLOUT);
  }
};



/*!
 * @brief
 * abstraction layer for the poll system call
 *
 * poll - wait for some event on a file descriptor;
 * refer manpage: Linux Programmers's Manual, POLL(2)
 *
 * This class wraps the linux system call "poll" and provides methods
 * to test whether a stream is readable, writeable and connected/alive.
 */
class Cpoll : public Cpoll_methods
{
private:
  int fd; // descriptor to poll

protected:
  virtual const int _get_descriptor() const { return fd; }  

public:
  //! create an Cpoll abstraction layer (attached to a socket descriptor)
  explicit Cpoll(const int _fd=0) : fd(_fd) {}

  //! create an Cpoll abstraction layer attached to an object that provides @p get_descriptor()
  template<class T> explicit Cpoll(const T &obj) : fd(obj.get_descriptor()) {}

  virtual ~Cpoll() {};

  //! attach it to a socket descriptor
  void attach(const int _fd) { fd=_fd; }

  /*! attach it to an object that provides @p get_descriptor()
   *
   * @param obj an object that handles a network connection and provides
   *            the attached socket descriptor through @p get_descriptor()
   */
  template<class T> void attach(const T &obj) { attach(obj.get_descriptor()); }
};



//! a virtual base class
class Cnetwork_connection_methods : public Cprovide_descriptor_access
{
 public:
  virtual ~Cnetwork_connection_methods() { }

  /*!
   *  set buffer size for input
   *  @param size  new size of input buffer
   *  @remark 
   *          - throws an exception on failure
   *          - For network connections SO_RCVBUF should be set after calling socket()
   *            but before bind(); this is because the receiving window will be negotiated during bind().
   *          - for Linux, "/proc/sys/net/core/rmem_max" can be used to get/set the system limit
   */
  inline void set_RCVBUF(const int size)
   {
     const int fd = _get_descriptor();
     int val = size; // new buffersize
     if (setsockopt(fd, SOL_SOCKET, SO_RCVBUF, &val, sizeof(val))==-1)
       throw unix_buffer_exception(std::string("Cnetwork_connection_methods::set_RCVBUF(): "+my_strerror(errno)));
   }

  /*!
   *  set buffer size for output
   *  @param size  new size of output buffer
   *  @remark 
   *          - throws an exception on failure
   *          - for Linux, "/proc/sys/net/core/wmem_max" can be used to get/set the system limit
   */
  inline void set_SNDBUF(const int size)
   {
     const int fd = _get_descriptor();
     int val;
     val=size; // new buffersize
     if (setsockopt(fd, SOL_SOCKET, SO_SNDBUF, &val, sizeof(val))==-1)
       throw unix_buffer_exception(std::string("Cnetwork_connection_methods::set_SNDBUF(): "+my_strerror(errno)));
   }

  //! get buffer size for input
  inline const int get_RCVBUF()
   {
     const int fd = _get_descriptor();
     int val = -1;
     socklen_t optlen = sizeof(val);
     if (getsockopt(fd, SOL_SOCKET, SO_RCVBUF, &val, &optlen)==-1)
      {
        perror("Cnetwork_connection_methods::get_RCVBUF()");
        return -1;
      }
     else return val;
   }

  //! get buffer size for output
  inline const int get_SNDBUF()
   {
     const int fd = _get_descriptor();
     int val = -1;
     socklen_t optlen = sizeof(val);
     if (getsockopt(fd, SOL_SOCKET, SO_SNDBUF, &val, &optlen)==-1)
      {
        perror("Cnetwork_connection_methods::get_SNDBUF()");
        return -1;
      }
     else return val;
   }

  //! set a connection timeout for receiving data
  inline void set_RCV_timeout(const int secs, const int u_secs = 0)
   {
     const int fd = _get_descriptor();
     timeval tv;
     tv.tv_sec = secs; // secs timeout
     tv.tv_usec = u_secs; // micro-secs timeout
     if (setsockopt(fd, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv))==-1)
       throw unix_buffer_exception(std::string("Cnetwork_connection_methods::set_RCV_timeout(): "+my_strerror(errno)));
   }

  //! set a connection timeout for sending data
  inline void set_SND_timeout(const int secs, const int u_secs = 0)
   {
     const int fd = _get_descriptor();
     timeval tv;
     tv.tv_sec = secs; // secs timeout
     tv.tv_usec = u_secs; // micro-secs timeout
     if (setsockopt(fd, SOL_SOCKET, SO_SNDTIMEO, &tv, sizeof(tv))==-1)
       throw unix_buffer_exception(std::string("Cnetwork_connection_methods::set_SND_timeout(): "+my_strerror(errno)));
   }

  //! get connection timeout for receiving data
  inline const int get_RCV_timeout()
   {
     const int fd = _get_descriptor();
     timeval tv;
     socklen_t optlen = sizeof(tv);
     if (getsockopt(fd, SOL_SOCKET, SO_RCVTIMEO, &tv, &optlen)==-1)
      {
        perror("Cnetwork_connection_methods::get_RCV_timeout()");
        return -1;
      }
     else return tv.tv_sec;
   }

  //! get connection timeout for sending data
  inline const int get_SND_timeout()
   {
     const int fd = _get_descriptor();
     timeval tv;
     socklen_t optlen = sizeof(tv);
     if (getsockopt(fd, SOL_SOCKET, SO_SNDTIMEO, &tv, &optlen)==-1)
      {
        perror("Cnetwork_connection_methods::get_SND_timeout()");
        return -1;
      }
     else return tv.tv_sec;
   }

  //! set a connection timeout for receiving and sending data
  inline void set_timeout(const int secs, const int u_secs = 0)
   {
     const int fd = _get_descriptor();
     timeval tv;
     tv.tv_sec = secs; // secs timeout
     tv.tv_usec = u_secs; // micro-secs timeout
     if (setsockopt(fd, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv))==-1)
       throw unix_buffer_exception(std::string("Cnetwork_connection_methods::set_timeout(), RCVTIMEO: "+my_strerror(errno)));
     if (setsockopt(fd, SOL_SOCKET, SO_SNDTIMEO, &tv, sizeof(tv))==-1)
       throw unix_buffer_exception(std::string("Cnetwork_connection_methods::set_timeout(), SNDTIMEO: "+my_strerror(errno)));
   }

  //! set tcp connection to TCP_NODELAY mode (on/off)
  inline void set_TCP_NODELAY(const bool flag = true)
   {
     const int fd = _get_descriptor();
     int optval=flag;
     if (setsockopt(fd, SOL_TCP, TCP_NODELAY, &optval, sizeof(optval))==-1)
      throw unix_buffer_exception(std::string("Cnetwork_connection_methods::set_TCP_NODELAY(): "+my_strerror(errno)));
   }

  //! get tcp connection TCP_NODELAY mode (on/off)
  inline const bool get_TCP_NODELAY()
   {
     const int fd = _get_descriptor();
     int optval;
     socklen_t optlen = sizeof(optval);
     if (getsockopt(fd, SOL_TCP, TCP_NODELAY, &optval, &optlen)==-1)
      perror("Cnetwork_connection_methods::get_TCP_NODELAY");
     return optval;
   }

  //! set linger value (time that a dead connection lingers before it can be reassigned)
  inline void set_LINGER(const signed int seconds)
   {
     // seconds<0  : turn linger off
     // soconds>=0 : turn linger on with specified value seconds
     const int fd = _get_descriptor();
     struct linger lingval;
     if (seconds<0) { lingval.l_onoff=0; lingval.l_linger=0; }
     else { lingval.l_onoff=1; lingval.l_linger=seconds; }
     if (setsockopt(fd, SOL_SOCKET, SO_LINGER, &lingval, sizeof(lingval))==-1)
      throw unix_buffer_exception(std::string("Cnetwork_connection_methods::set_LINGER(): "+my_strerror(errno)));
   }

  //! get linger value (time that a dead connection lingers before it can be reassigned)
  inline const signed int get_LINGER()
   {
     // returns linger seconds or -1 for linger off or -2 for error
     const int fd = _get_descriptor();
     struct linger lingval;
     socklen_t optlen = sizeof(lingval);
     if (getsockopt(fd, SOL_SOCKET, SO_LINGER, &lingval, &optlen)==-1)
      {
        perror("Cnetwork_connection_methods::get_LINGER()");
        return -2;
      }
     if (lingval.l_onoff==0) return -1;
     else return lingval.l_linger;
   }

  /*!
   * @result string that contains human readable information
   *         about the client who has initiated the network connection
   */
  inline const std::string peer_info()
  {
   // outputs connection meta-data for class object to string
   return ::peer_info(_get_descriptor());
  }

  /*!
   * @result string that contains human readable information
   *         about the server who has "picked up" the network connection
   */
  inline const std::string socket_info()
  {
   // outputs connection meta-data for class object to string
   return ::socket_info(_get_descriptor());
  }

  /*!
   * @result string that contains human readable information
   *         about the client who has initiated the network connection
   *         and the server who has "picked up" the network connection
   */
  inline const std::string connection_info()
  {
   // outputs connection meta-data for class object to string
   return ::connection_info(_get_descriptor());
  }

};




/*! opens a server socket on a given port on construction,
 *  returns incoming client connections on request,
 *  and shuts down the server socket on destruction.
 */
class connection_waiter : public Cnetwork_connection_methods
{
 protected:
  int server_socket;
  const int client_default_timeout;

  virtual const int _get_descriptor() const { return server_socket; } 
 public:

  /*! tries to open the given server port for incoming client connections
   * 
   * @param server_port server port to listen for new connections
   * @param server_port_tries (optional) number of tries to establish the server socket before giving up
   * @param _client_default_timeout (optional) default timeout in seconds for client connections
   *                                (to automate/ease unix_buffer initializations) 
   */
  explicit connection_waiter(const int server_port, int server_port_tries = 10, const int _client_default_timeout = 75)
   : server_socket(-1), client_default_timeout(_client_default_timeout)
   {
     while ((server_socket = open_internet_port(server_port,10))<0
            && --server_port_tries>=0) sleep(1);
     if (server_socket<0)
      {
        cerr << "connection_waiter::constructor: cannot open serverport " << server_port << endl;
        throw unix_buffer_exception("connection_waiter::constructor: cannot open serverport");
      }
   }

  //! closes the server socket
  ~connection_waiter()
   {
     if (shutdown(server_socket, SHUT_RDWR)==-1) perror("~connection_waiter():shutdown");
     close(server_socket);
   }

  /*! waits for a new client connection an returns a socket descriptor
   * 
   * @returns socket descriptor for a new client connection
   */
  const int get_new_client_socket_descriptor() const
   {
     int sd;
     do
      {
        sd=accept(server_socket, NULL, NULL);
	PTHREAD_TESTCANCEL;
      } while (sd==0);
     return sd;
   }

  /*!
   * 
   * @returns default timeout in seconds for client connections
   */
  const int get_client_default_timeout() const
   {
     return client_default_timeout;
   }
};


//! This class provides a streambuffer for handling file/socket descriptor resources.
class unix_fd_buffer : public streambuf, public Cprovide_descriptor_access
{
protected:
  static const int buffersize = 4096;
  static const int pushbacksize = 10;
  int fd; // -1 will explicitly denote an invalid descriptor
  bool fd_autoclose; // whether a valid descriptor shall be closed at destructor call
  char obuffer[buffersize];
  char ibuffer[buffersize+pushbacksize];

  virtual const int _get_descriptor() const { return fd; } 

  virtual const int empty_buffer()
   {
     if (fd<0) // invalid file descriptor?
      {
        cerr << "unix_fd_buffer::empty buffer(): invalid descriptor! fd=" << fd << endl;
        throw unix_buffer_exception("unix_fd_buffer::empty buffer(): invalid descriptor!");
        return EOF;
      }
     const int total_to_write = pptr()-pbase();
     if (!total_to_write) return 0;
     if (total_to_write<0 || total_to_write>buffersize)
      {
        throw unix_buffer_exception("unix_fd_buffer::empty_buffer() sanity check failed!");
      }
     int total_written = 0;
     signed int retries = 2; // maximum of retries in case of temporary errors
     do
     {
       PTHREAD_TESTCANCEL;
       const int count = total_to_write-total_written;
       const int written=write(fd, obuffer, count);
       PTHREAD_TESTCANCEL;
       if (written!=count)
        {
          cerr << "unix_fd_buffer::empty_buffer(): "
               << written << " bytes of " << count << " sent." << endl;
          if (written<0)
           {
	     --retries;
             const int err=errno;
             perror("unix_fd_buffer::empty_buffer()");
             if (written==-1 && err == EAGAIN && retries>=0) continue; // this may fix some problems...
             if (written==-1 && err == EINTR && retries>=0) continue; // this may fix some problems...
             throw unix_buffer_exception(std::string("unix_fd_buffer::empty_buffer() send failed: ")+my_strerror(err));
             return EOF;
           }
        }
       pbump(-written);
       total_written+=written;
     } while (total_written<total_to_write);
     return total_written;
   }

  virtual int overflow (int c) 
   {
     if (c != EOF) { *pptr() = c; pbump(1); }
     if (empty_buffer() == EOF) return EOF;
     return c;
   }

  virtual int sync ()
   {
     return (empty_buffer() == EOF) ? -1 : 0;
   }

  virtual int underflow ()
   {
     if (gptr() < egptr()) return *gptr();
     int pbcount = gptr() - eback();
     if (pbcount > pushbacksize) pbcount = pushbacksize;
     if (pbcount > 0) memmove (ibuffer+(pushbacksize-pbcount), gptr()-pbcount, pbcount);
     int count;
     signed int retries = 2; // maximum of retries in case of temporary errors
     do
     {
       PTHREAD_TESTCANCEL;
       count = read(fd, ibuffer+pushbacksize, buffersize);
       PTHREAD_TESTCANCEL;
       if (count <= 0) // any unexpected behaviour?
        {
         if (count==0) return EOF;
	 --retries;
         const int err=errno;
         perror("unix_fd_buffer::underflow()");
         if (count==-1 && err == EAGAIN && retries>=0) continue; // this should fix some problems...
         if (count==-1 && err == EINTR && retries>=0) continue; // this should fix some problems...
         throw unix_buffer_exception(std::string("unix_fd_buffer::underflow() recv failed: ")+my_strerror(err));
 	 return EOF;
        }
     } while (count<=0);
     //cerr << "---> count: " << count << endl;
     setg(ibuffer+(pushbacksize-pbcount), ibuffer+pushbacksize,
	  ibuffer+pushbacksize+count);
     return *gptr();
   }
  
public:

  //! returns the attached file/socket descriptor
  inline const int get_descriptor() const { return fd; }


  /*!
   *  attach a file / network connection via the given file/socket descriptor
   *
   *  @param _fd  file/socket descriptor to attach
   *  @param autoclose  (optional) whether _fd should be closed on destruction of this object
   *
   *  @remark 
   *          - it is not possible to attach more than one connection for a buffer simultaneously
   *          - you shouldn't close the attached descriptor without destructing the buffer
   *          - you should use the connection pointed by the descriptor exclusively by
   *            this unix_fd_buffer object
   *          - if you want to reuse the descriptor after destructing this unix_fd_buffer object,
   *            you should disable the autoclose feature
   */
  void attach(const int _fd, const bool autoclose = true)
  {
    if (fd>0)
     {
       throw unix_buffer_exception("unix_fd_buffer::open(const int _fd): descriptor is already in use!");
     }
    if (_fd<0)
     {
       throw unix_buffer_exception("unix_fd_buffer::open(const int _fd): descriptor is invalid!");
     }
    fd = _fd; fd_autoclose=autoclose;
    setp(obuffer, obuffer+(buffersize-1));
    setg(ibuffer+pushbacksize, ibuffer+pushbacksize, ibuffer+pushbacksize);
  }

  /*!
   *  detach the buffer from the file/socket descriptor
   *
   *  @returns an open file/socket descriptor or -1 if there was no descriptor or it has been autoclosed
   *
   *  @remark If @p returnvalue is not -1, it is most possibly open and should not be thrown away.
   */
  virtual const int detach()
  {
    if (fd<0) return -1; // no valid descriptor attached
    sync();
    if (fd_autoclose)
     {
       if (shutdown(fd, SHUT_RDWR)==-1) perror("~unix_fd_buffer():shutdown");
       close(fd);
       fd=-1; // fd now invalid
     }
    const int h=fd;
    fd=-1; // no descriptor attached to this object now
    return h;
  }

  //! create an unix_fd_buffer that is ready to be connected to a file/socket descriptor
  unix_fd_buffer(void) // default constructor
   : fd(-1), // -1 denotes no descriptor defined yet
     fd_autoclose(true)
  {
    setp(obuffer, obuffer+(buffersize-1));
    setg(ibuffer+pushbacksize, ibuffer+pushbacksize, ibuffer+pushbacksize);
  }

  /*! create an unix_fd_buffer that is connected to the given file/socket descriptor
   * @param _fd descriptor of an already opened connection
   *
   * @remark - resource will be closed on destruction
   *         - if you want to reuse the descriptor, you should use the default constructor
   *           and attach a descriptor later on with @p autoclose set to @p false
   */
  explicit unix_fd_buffer(const int _fd)
   : fd(_fd), fd_autoclose(true)
  {
    setp(obuffer, obuffer+(buffersize-1));
    setg(ibuffer+pushbacksize, ibuffer+pushbacksize, ibuffer+pushbacksize);
  }


  //! destroy the unix_fd_buffer and close resource (if any was open)
  virtual ~unix_fd_buffer()
   {
     if (fd>=0)
      {
        // valid descriptor -> free resource
        sync();
        if (fd_autoclose) close(fd);
        fd=-1;
      }
   }

};



//! This class provides a streambuffer for handling network connections.
class unix_buffer : public unix_fd_buffer, public Cnetwork_connection_methods
{
protected:
  virtual const int _get_descriptor() const { return fd; } 

  virtual const int empty_buffer()
   {
     if (fd<0) // invalid file descriptor?
      {
        cerr << "unix_buffer::empty buffer(): invalid descriptor! fd=" << fd << endl;
        throw unix_buffer_exception("unix_buffer::empty buffer(): invalid descriptor!");
        return EOF;
      }
     const int total_to_write = pptr()-pbase();
     if (!total_to_write) return 0;
     if (total_to_write<0 || total_to_write>buffersize)
      {
        throw unix_buffer_exception("unix_buffer::empty_buffer() sanity check failed!");
      }
     int total_written = 0;
     signed int retries = 2; // maximum of retries in case of temporary errors
     do
     {
       PTHREAD_TESTCANCEL;
       const int count = total_to_write-total_written;
       const int written=send(fd, obuffer, count, MSG_NOSIGNAL);
       PTHREAD_TESTCANCEL;
       if (written!=count)
        {
          cerr << "unix_buffer::empty_buffer(): "
               << written << " bytes of " << count << " sent." << endl;
          if (written<0)
           {
	     --retries;
             const int err=errno;
             perror("unix_buffer::empty_buffer()");
             if (written==-1 && err == EAGAIN && retries>=0) continue; // this may fix some problems...
             if (written==-1 && err == EINTR && retries>=0) continue; // this may fix some problems...
             throw unix_buffer_exception(std::string("unix_buffer::empty_buffer() send failed: ")+my_strerror(err));
             return EOF;
           }
        }
       pbump(-written);
       total_written+=written;
     } while (total_written<total_to_write);
     return total_written;
   }

  virtual int underflow ()
   {
     if (gptr() < egptr()) return *gptr();
     int pbcount = gptr() - eback();
     if (pbcount > pushbacksize) pbcount = pushbacksize;
     if (pbcount > 0) memmove (ibuffer+(pushbacksize-pbcount), gptr()-pbcount, pbcount);
     int count;
     signed int retries = 2; // maximum of retries in case of temporary errors
     do
     {
       PTHREAD_TESTCANCEL;
       count = recv(fd, ibuffer+pushbacksize, buffersize, MSG_NOSIGNAL);
       PTHREAD_TESTCANCEL;
       if (count <= 0) // any unexpected behaviour?
        {
         if (count==0) return EOF;
	 --retries;
         const int err=errno;
         perror("unix_buffer::underflow()");
         if (count==-1 && err == EAGAIN && retries>=0) continue; // this should fix some problems...
         if (count==-1 && err == EINTR && retries>=0) continue; // this should fix some problems...
         throw unix_buffer_exception(std::string("unix_buffer::underflow() recv failed: ")+my_strerror(err));
 	 return EOF;
        }
     } while (count<=0);
     //cerr << "---> count: " << count << endl;
     setg(ibuffer+(pushbacksize-pbcount), ibuffer+pushbacksize,
	  ibuffer+pushbacksize+count);
     return *gptr();
   }
  
public:

  /*!
   *  detach the buffer from the socket descriptor
   *
   *  @returns an open socket descriptor or -1 if there was no socket descriptor or it has been autoclosed
   *
   *  @remark If @p returnvalue is not -1, it is most possibly open and should not be thrown away.
   */
  virtual const int detach()
  {
    if (fd<0) return -1; // no valid descriptor attached
    sync();
    if (fd_autoclose)
     {
       if (shutdown(fd, SHUT_RDWR)==-1) perror("~unix_buffer():shutdown");
       close(fd);
       fd=-1; // fd now invalid
     }
    const int h=fd;
    fd=-1; // no descriptor attached to this object now
    return h;
  }

  //! create an unix_buffer that is ready to be connected to a socket descriptor
  unix_buffer(void) // Default constructor
   : unix_fd_buffer() // use defaults from unix_fd_buffer
  {
    set_timeout(75); // default: 75 seconds
  }

  /*! create an unix_buffer that is connected to the given socket descriptor
   * @param _fd descriptor of an already opened connection
   * @param secs_timeout (optional) timeout in seconds before read/write operations may fail
   *
   * @remark - connection will be closed on destruction
   *         - if you want to reuse the descriptor, you should use the default constructor
   *           and attach a descriptor later on with @p autoclose set to @p false
   */
  explicit unix_buffer(const int _fd, const int secs_timeout = 75)
   : unix_fd_buffer(_fd)
  {
    set_timeout(secs_timeout);
  }

  /*! create an unix_buffer that is connected to the given socket descriptor
   * @param cw a connection_waiter object, that returns a descriptor of
   *           the next incoming client connection request
   * @remark connection will automatically be closed on destruction
   */
  explicit unix_buffer(const connection_waiter& cw)
   : unix_fd_buffer(cw.get_new_client_socket_descriptor())
  {
    set_timeout(cw.get_client_default_timeout());
  }

 /*! create an unix_buffer that connects to the given socket
  *  (and create a socket descriptor by itself)
  *
  * @param host string of symbolic/numerical internet address of host to connect
  * @param port port number to connect
  */
 unix_buffer(const std::string &host, const int port)
  : unix_fd_buffer()
  {

#ifdef CYGWIN_COMPAT
    // cygwin compatibility mode

    struct sockaddr_in server;
    struct hostent *hp;
    int ret;
    int err_number;
    fd = socket(AF_INET, SOCK_STREAM, 0);
    if (fd < 0)
     {
       throw unix_buffer_exception("unix_buffer::constructor: Can't open socket");
     }
    server.sin_family = AF_INET;

    CYGW_LOCK;
    hp = gethostbyname(host.c_str());
    if (hp == NULL)
     {
       CYGW_UNLOCK;
       throw unix_buffer_exception("unix_buffer::constructor: Unknown host: "+host);
     }
    memcpy((char*)&server.sin_addr, (char*)hp->h_addr, hp->h_length);
    CYGW_UNLOCK;

    server.sin_port = htons(port);
    {
      int retries_on_EAGAIN = 3;
      while ((ret = connect(fd, reinterpret_cast<struct sockaddr *>(&server), sizeof(server))) < 0)
       {
	switch (err_number=errno)
         {
	   case ETIMEDOUT:
            cerr << "." << flush; sleep(10); break;
	   case ENETUNREACH:
            cerr << "%" << flush; sleep(10); break;
           case EAGAIN:
            if (retries_on_EAGAIN--)
             {
               cerr << "Can't connect socket (" << my_strerror(err_number) << ")" << endl;
               if (err_number==EAGAIN)
                {
                  cerr << "Trying it again in 10 secs..." << endl;
                  sleep(10);
                }
	       break;
             }
	   default:
            throw unix_buffer_exception(std::string("unix_buffer::constructor: Can't connet socket: ")+my_strerror(err_number));
	 }
       }
    }

#else
    // POSIX mode

    // refer "man getaddrinfo" and "man socket" 
    struct addrinfo hints; // our wishes are placed here
    memset(&hints,0,sizeof(hints)); // must be zeroed for default options
    hints.ai_family=PF_INET; // we want IPv4 as protocol
    hints.ai_socktype=SOCK_STREAM; // and we need a stream, not datagram!
    struct addrinfo *addrinfo_res = NULL; // here the result will be delivered
#ifdef WEIRD_VALGRIND_HACK
    static CMutex serial;
    serial.lock();
    // normally (by specification) getaddrinfo is thread-safe!
    // however: under valgrind, multithreaded sessions will hang exactly at this
    // point; probably there are other errors as well; but how to track them, when
    // every session hangs here (without any useful information)?
    // -> with locked access: no error occurs!
    const int retval = getaddrinfo(host.c_str(),NULL,&hints,&addrinfo_res);
    serial.unlock();
#else
    const int retval = getaddrinfo(host.c_str(),NULL,&hints,&addrinfo_res);
#endif
    if ( retval || addrinfo_res==NULL ) // any error?
     {
       cerr << "can't reach " << "\"" <<  host << "\"" << endl;
       cerr << "Error given by getaddrinfo: " << endl;
       cerr << gai_strerror(retval) << endl;
       throw unix_buffer_exception("unix_buffer::constructor: Can't reach "+host+": "+gai_strerror(retval));
     }
    if (addrinfo_res->ai_socktype!=SOCK_STREAM) // we got a "stream"-protocol?
     {
       throw unix_buffer_exception("unix_buffer::constructor: provided protocol doesn't support SOCK_STREAM");
     }
    switch (addrinfo_res->ai_family)
     {
      case PF_INET:
       //cerr << "PF_INET: using IPv4!" << endl;
       fd = socket(AF_INET, SOCK_STREAM, 0);
       reinterpret_cast<sockaddr_in*>(addrinfo_res->ai_addr)->sin_port=htons(port);
       break;
      case PF_INET6:
       cerr << "PF_INET6: try using IPv6 (untested feature!)" << endl;
       fd = socket(AF_INET6, SOCK_STREAM, 0);
       reinterpret_cast<sockaddr_in6*>(addrinfo_res->ai_addr)->sin6_port=htons(port);
       break;
      default:
       throw unix_buffer_exception("unix_buffer::constructor: too bad! ai_family isn't supported by unix_buffer!");
     }

    if (fd < 0) // is the descriptor valid?
     {
       throw unix_buffer_exception("unix_buffer::constructor: Can't open socket");
     }

    {
      int retries_on_EAGAIN = 3;
      while ( connect(fd, addrinfo_res->ai_addr, addrinfo_res->ai_addrlen) <0 )
       {
        PTHREAD_TESTCANCEL;
        switch (int err_number=errno)
         {
           case ETIMEDOUT:
            cerr << "." << flush; sleep(10); break;
           case ENETUNREACH:
            cerr << "%" << flush; sleep(10); break;
           case EAGAIN:
            if (retries_on_EAGAIN--)
             {
               cerr << "Can't connect socket (" << my_strerror(err_number) << ")" << endl;
               if (err_number==EAGAIN)
                {
                  cerr << "Trying it again in 10 secs..." << endl;
                  sleep(10);
                }
               break;
             }
           default:
            throw unix_buffer_exception(std::string("unix_buffer::constructor: Can't connect socket: ")+my_strerror(err_number));
         }
       }
    }
    freeaddrinfo(addrinfo_res);
#endif

    setp(obuffer, obuffer+(buffersize-1));
    setg(ibuffer+pushbacksize, ibuffer+pushbacksize, ibuffer+pushbacksize);
  }

  //! destroy the unix_buffer and close network connection (if any was open)
  virtual ~unix_buffer()
   {
     if (fd>=0)
      {
        // valid descriptor -> close the socket
        // but be careful: the socket may be already dead, so we nee to catch exceptions!
        try { sync(); }
        catch (unix_buffer_exception &e)
         {
           cerr << "unix_buffer::destructor caught exception: " << e.what() << endl;
         }
        if (fd_autoclose)
         {
           if (shutdown(fd, SHUT_RDWR)==-1) perror("~unix_buffer():shutdown");
           close(fd);
         }
        fd=-1; // very important! tell the ancestor not to destruct fd twice!
      }
   }

};




//! a simple network connection stream
class unix_io_stream : public iostream, public Cnetwork_connection_methods
{
protected:
  unix_buffer buf;
  virtual const int _get_descriptor() const { return buf.get_descriptor(); } 

public:

  //! returns the attached socket descriptor
  inline const int get_descriptor() const { return buf.get_descriptor(); }


  /*!
   * @param host string of symbolic/numerical internet address of host to connect
   * @param port port number to connect
   *
   * A tcp based network connection will be established and provided as stream.
   */
  unix_io_stream(const std::string &host, const int port) : iostream(&buf), buf(host, port)
   {
   }


  /*!
   * @param cw a connection_waiter object, that returns a descriptor of
   *           the next incoming client connection request
   * @remark connection will automatically be closed on destruction
   */
  explicit unix_io_stream(const connection_waiter& cw)
   : iostream(&buf), buf(cw)
   {
   }


  /*!
   * @param _fd descriptor of an already opened connection
   * @param secs_timeout (optional) timeout in seconds before read/write operations may fail
   */
  explicit unix_io_stream(const int _fd, const int secs_timeout = 75) : iostream(&buf), buf(_fd, secs_timeout)
   {
   }


  virtual ~unix_io_stream()
   {
   }
};


//! iostream attached to a file descriptor (can be reduced to an istream or ostream!)
class fd_iostream : public std::iostream, public Cprovide_descriptor_access
 {
   protected:
    unix_fd_buffer buf;
    virtual const int _get_descriptor() const { return buf.get_descriptor(); } 
   public:

    //! returns the attached file/socket descriptor
    inline const int get_descriptor() const { return buf.get_descriptor(); }

    explicit fd_iostream(const int _fd) : std::iostream(&buf), buf(_fd) { }
    virtual ~fd_iostream() { }
 };

//! iostream attached to a socket descriptor (can be reduced to an istream or ostream!)
class sd_iostream : public fd_iostream, public Cnetwork_connection_methods
 {
   protected:
    virtual const int _get_descriptor() const { return buf.get_descriptor(); }
   public:
    explicit sd_iostream(const int _fd) : fd_iostream(_fd) { }
    virtual ~sd_iostream() { }
 };


//! this class provides a socket creator for unnamed pipes
class socket_piper
{
 private:
  int socketdes[2];

 public:

  //! create a pipe and store the two descriptors privately
  socket_piper()
  {
    socketdes[0]=socketdes[1]=-1;
    int retval=socketpair(AF_UNIX, SOCK_STREAM, 0, socketdes); // create a pipe (refer manpage SOCKETPAIR(2))
    if (retval) throw unix_buffer_exception(std::string("socket_piper::constructor: "+my_strerror(errno)));
  }

  //! destruct the pipe creator and undetached descriptors
  ~socket_piper()
  {
    // delete undetached descriptors
    if (socketdes[0]>=0) close(socketdes[0]);
    if (socketdes[1]>=0) close(socketdes[1]);
  }

  /*!
   * spawn the read part of the constructed pipe and detach it
   * @param bufsize (optional) buffer size
   * @remark buffersize<0 for system default buffer size
   */
  std::istream* detach_InPipe(const int bufsize=-1)
  {
    if (socketdes[0]<0) throw unix_buffer_exception("socket_piper::detach_InPipe: no descriptor available!"); 
    sd_iostream *pInPipe = new sd_iostream(socketdes[0]);
    //std::cout << "InPipe bufsize (prev): " << pInPipe->get_RCVBUF() << endl;
    if (bufsize>=0) pInPipe->set_RCVBUF(bufsize);
    //std::cout << "InPipe bufsize (post): " << pInPipe->get_RCVBUF() << endl;
    socketdes[0]=-1; // detach
    return pInPipe;
  }

  /*!
   * spawn the write part of the constructed pipe and detach it
   * @param bufsize (optional) buffer size
   * @remark buffersize<0 for system default buffer size
   */
  std::ostream* detach_OutPipe(const int bufsize=-1)
  {
    if (socketdes[1]<0) throw unix_buffer_exception("socket_piper::detach_OutPipe: no descriptor available!"); 
    sd_iostream *pOutPipe = new sd_iostream(socketdes[1]);
    //std::cout << "OutPipe bufsize (prev): " << pOutPipe->get_SNDBUF() << endl;
    if (bufsize>=0) pOutPipe->set_SNDBUF(bufsize);
    //std::cout << "OutPipe bufsize (post): " << pOutPipe->get_SNDBUF() << endl;
    socketdes[1]=-1; // detach
    return pOutPipe;
  }  
};


#endif
