/*! @file
 * @brief
 * a small program to transfer sieved relations to the server 
 *
 * If the sieving is done offline by a file-client, the sieved relations
 * are stored in files. These data need to be transmitted to the server
 * somehow. This is done by the transfer-client, which is implemented here.
 *
 */


#include <cstdlib>
#include "unix_buffer.H"
#include <fstream>
#include <string>
#include "qsieve-fwd.H" // for server port

extern "C"
{
 #include <getopt.h>
}

using namespace std;

const char* const option_string = "+hgpvi:f:";
static const struct option long_options[] =
{
  { "help", 0, 0, 'h' },
  { "get", 0, 0, 'g' },
  { "put", 0, 0, 'p' },
  { "verbose", 0, 0, 'v' },
  { "interval", 1, 0, 'i' },
  { "file", 1, 0, 'f' },
  { 0, 0, 0, 0 }
}; // refer "man 3 getopt" for this struct

void usage(void)
{
  cout << "usage: transfer-client [-h] [-v] [-g]|-p [-i interval] [-f file] <server>" << endl;
  cout << " -h, --help     this text" << endl;
  cout << " -g, --get      get-mode for fetching of sieve parameters from server" << endl;
  cout << " -p, --put      put-mode for sending relations to server" << endl;
  cout << " -v, --verbose  be verbose" << endl;
  cout << " -i, --interval <interval>" << endl;
  cout << "                number of polynomial intervals to fetch (default:25000)" << endl;
  cout << " -f, --file <file>" << endl;
  cout << "                file to save transmitted data (default:qsieve-fc.param)" << endl;
}

int main(const int argc, char* const argv[])
{
  int polynome = 25000;
  const char *file = NULL;
  bool get = true;
  bool option = false;
  bool help = false;
  bool verbose = false;
  string s;
  int i, option_index;
  // check for parameters ;-)
  while ((i=getopt_long(argc,argv,option_string,long_options,&option_index))!=-1)
    {
      switch (i)
	{
	case 'g':
	  if (option && !get)
	    {
	      cerr << "error: -f und -p are mutual exclusive!" << endl;
	      exit(1);
	    }
	  option = get = true;
	  break;
	case 'p':
	  if (option && get)
	    {
	      cerr << "error: -f und -p are mutual exclusive!" << endl;
	      exit(1);
	    }
	  option = true;
	  get = false;
	  break;
	case 'f':
	  if (file != NULL)
	    {
	      cerr << "error: -f must be specified at most once!" << endl;
	      exit(1);
	    }
	  file = optarg;
	  break;
	case 'h':
	  help = true;
	  usage();
	  break;
	case 'i':
	  polynome = atoi (optarg);
	  break;
	case 'v':
	  verbose = true;
	  break;
	default:
	  usage();
	  exit(1);
	}
    }
  
  if (file == NULL)
    {
      if (get)
	file = "qsieve-fc.param";
      else
	file = "qsieve-fc.param.out";
    }
  if (optind+1 != argc) {
    if (help) exit(0);
    cerr << "wrong number of parameters!" << endl;
    exit(1);
  }
  const string server = argv[optind];
  
  if(get)
    {
      char c;
      ofstream file_stream(file, ios::out|ios::trunc);
      {
	unix_io_stream communication_stream(server, server_port);
	communication_stream << QsieveLogon << endl;
	if (communication_stream.eof())
         {
           cerr << "Unexpected EOF!" << endl;
           exit(1);
         }
        else
         {
           communication_stream.get(c);
           if (c=='-')
            {
             cout.put(c);
             while (!communication_stream.eof())
	      {
	       communication_stream.get(c);
               cout.put(c);
	      }
             exit(1);
            }
           else communication_stream.putback(c);
         }
	while (!communication_stream.eof())
	  {
	    communication_stream.get(c);
	    file_stream.put(c);
            if (verbose) cout.put(c);
	  }
      }
      {
	unix_io_stream communication_stream(server, server_port);
	communication_stream << "DynamicFactors?_ab_fpos 0" << endl;
	while (!communication_stream.eof())
	  {
	    communication_stream.get(c);
	    file_stream.put(c);
            if (verbose) cout.put(c);
	  }
      }
      {
	unix_io_stream communication_stream(server, server_port);
	communication_stream << "Polynom? " << polynome << endl;
	while (!communication_stream.eof())
	  {
	    communication_stream.get(c);
	    file_stream.put(c);
            if (verbose) cout.put(c);
	  }
      }
    }
  else
    {
      char c;
      ifstream file_stream(file, ios::in);
      if (!file_stream)
       {
         cerr << "Unable to open \"" << file << "\"" << endl;
         exit(1);
       }
      char kN_str[1024]; // "1024 bytes should be enough for everyone"
      file_stream.getline(kN_str,sizeof(kN_str),'\n');

      char line[1024];
      while (file_stream.peek()!=EOF)
	{
	  int count = 1000+1; // transmit maximal count-1 relations per block
          unix_io_stream communication_stream(server, server_port);

          communication_stream << "NewRelations! " << kN_str << endl;
          // verify, that relations belong to our factorization
          communication_stream >> s;
          if (s!="proceed")
           {
             cerr << "Oops! Server will not accept my relations: " << endl;
             cerr << s << endl;
             exit(1);
           }

          while (file_stream.peek()!=EOF && --count)
	  {
            communication_stream << "RL "; // new token for "Relation! "
            while (file_stream.peek()!='\n')
             {
               file_stream.get(line,sizeof(line),'\n');
               communication_stream << line;
               if (verbose) cout << line;
             }
            file_stream.get(c);
            communication_stream << endl;
            if (verbose) cout << endl;
          }
          communication_stream << "Relationsblock_Ende " << flush;
	  communication_stream >> s;
	  if (s!="empfangen.")
            cerr << "transmitted relations were not acknowledged by server!" << endl;
          if (s=="ignoriert!")
            {
              cerr << "Oops! - Relations were ignored by server!" << endl;
              cerr << "Please check, whether transmitted data in "
                   << "\"" << file << "\"" << endl
                   << "really belong to current server session!"
                   << endl;
              exit(1);
            }
	}
    }
}
