/*! @file
 * @brief
 * main implementation file of Qsieve server
 */


#include "at_startup.H"

// sanity checks
#ifdef IS_CLIENT
 #error "The server is not a client!"
#endif

#ifndef IS_SERVER
 #error "The server needs IS_SERVER to function correctly!"
#endif

#ifndef USE_NETWORK
 #error "The server uses network features!"
#endif

#ifdef IS_STANDALONE
 #error "The server is not standalone!"
#endif



extern "C"
{
 #include <unistd.h>
}

#include <cstdio>
#include <cstdlib>
#include <fstream>
#include <sstream>
#include <cmath>
#include <gmp.h>
#include <list>
#include <set>
#include <ctime>

#include "qsieve.H"
#include "StaticFactorbase.H"


#include "FactorFound.H"
TFoundFactors FoundFactors;

#include <vector>
#include <stack>
#include <algorithm>

extern "C"
{
 #include <sys/utsname.h>
 #include <pthread.h>
}

#include <csignal>
#include "mutex.H"



const string RecoveryFile         = "recovery.dat";
const string FoundFactorsFile     = "FoundFactors.dat";
const string StaticRelationsFile  = "static_relations.dat";
const string SpecialRelationsFile = "special_relations.dat";


// provide streams for storing relations and other data
// *************************************************************************
// **** important:
// **** If "ostream" and "istream" are handled by the same filebuf,
// **** then you *must not* rely on 
// **** "tellp()" and "tellg()" behave independently from each other!
// **** Any write operation may change also the value of "tellg()"
// **** and any read operation may change the value of "tellp()"!
// ************************************************************************* 


// for Recovery
filebuf Recovery_buffer;
ostream Recovery_to_file (&Recovery_buffer);
istream Recovery_from_file (&Recovery_buffer);


#include "mpqsPolynom.H"


mpz_t n, // number to factorize (will be reduced during factorization)
      kN; // input for MPQS (includes a suitable multiplier)


#include "StaticRelations.H"
#include "DynamicRelations.H"
#include "SpecialRelations.H"

#include "Sieving.H"
#include "ConfigFile.cc"


CmpqsPolynom Polynom; // object to manage polynomial computations for multipolynomial sieve (MPQS)
#include "mpqsStatistics.cc" // statistical stuff about found relations, client connections, etc.


#include "modulo.H" // modulo operations for unsigned int
using namespace numtheory;

#include "FactorFound.cc"
#include "polynomial.H" /* for fast polynomial arithmetic & fft-continuation */
#include "fft_param.cc"  /* discrete fast fourier transform */
#include "elliptic_curve.H" /* invariant part of elliptic curve method (ecm) */
#include "elliptic_curve-variant.cc" /* variant part of elliptic curve method (ecm) */

#include "ExitManager.cc" /* controlled termination of program */
#include "StaticRelations.cc"


// this stuff may only be needed by server or stand-alone version
#include "easy_factor.H"


#include "CRelation-inc.cc"


#include "parse_term.cc"
// this is a predicate stating "false"
const bool without_multi_combine_init = false; // true ->multi-init, false ->no multi_init on StaticRelations::insert


#include "SpecialRelations.cc"


#include "Cprocess_clients.cc"
#include "XML_StatusServer.cc"



void kill_all_other_threads()
{
  //cout << "Killing explicitly all other threads..." << endl;
  //pthread_kill_other_threads_np();
}



void cleanup_files()
{
#ifdef VERBOSE_INFO
  cout << "cleaning files..." << endl;
#endif
  StaticRelations::cleanup_files();
  SpecialRelations::cleanup_files();
  DynamicRelations::cleanup_files();
  Recovery_buffer.close();
  remove(RecoveryFile.c_str());
}

void cleanup_memory()
{
#ifdef VERBOSE_INFO
  cout << "cleanup allocated memory" << endl;
#endif
  StaticRelations::cleanup_memory();
  mpz_clear(kN); mpz_clear(n);
}


void signalhandler(int signr)
{
#ifdef VERBOSE_WARN
  cout << "Signal " << signr << " received. (ThreadId=" << pthread_self() << ")" << endl;
#endif
  exit(1);
}




int main(const int argc, const char* const argv[])
{

#ifdef USE_NCURSES
  new Cncursed(); // trigger activation of ncursed streams
#endif

  PrintHeader("Qsieve server");
    
  if (argc>2)
    {
      cerr << "number for factorization expected!" << endl;
      exit(1);
    }
  
  const bool recover = (argc==1); // without argument: Recovery-Mode

  atexit(kill_all_other_threads);
  cout.setf(ios::fixed); // fixed decimal notation, not scientific notation!

  mpz_init(n); // our number to factorize
  mpz_init(kN);
  ExitManager::register_exithandler(cleanup_memory); // on successful exit free allocated data

  Read_ConfigFile();


  // establish the network service at this point,
  // to allow clients to connect
  const connection_waiter my_connection_waiter(server_port);

  if (!recover) // not in recovery-mode -> evaluate arguments
    { 
      // get and evaluate given number
      char* const neuer_str = strdup(argv[1]);
      char* str = neuer_str; 
      if (!parse_term::get_number(n, str))
	{
	  cout << "Wrong input at: '" << str << "'" << endl;
	  exit(1);
	}
      else
	if (str[0]!='\0')
	  {    
	    cout << "Syntax error in input term. (parenthesis?)" << endl;
	    exit(1);
	  }
	else
	  if (mpz_cmp_ui(n,0)<=0)
	    {
	      cout << "Input must be positive natural number!" << endl;
	      exit(1);
	    }
      free(neuer_str); // don't call "delete []" because of "stdup/malloc"
      FoundFactors.regarding=argv[1];
      mpz_set(FoundFactors.regarding_numeric_value,n);
    }

  if (recover)
    {
      // Recovery: continue a factorization, which was previously aborted
      // (try to) open the necessary file streams
      StaticRelations::FileBuffer.open(StaticRelationsFile.c_str(),ios::in|ios::out);
      SpecialRelations::FileBuffer.open(SpecialRelationsFile.c_str(),ios::in|ios::out);
      if (!DynamicRelations::FileBuffer.open(DynamicRelationsFile.c_str(),ios::in|ios::out))
       {
         cerr << "Cannot access " << DynamicRelationsFile << endl;
         exit(1);
       }
      Recovery_buffer.open(RecoveryFile.c_str(),ios::in|ios::out|ios::ate);

      // offenbar wird trotz ios::ate nicht (immer) ans Ende positioniert
      // deshalb wird die Testabfrage modifiziert:
      if (Recovery_from_file) Recovery_from_file.seekg(0,ios::end);
#ifdef STL_STREAM_workaround
      if ( (!Recovery_from_file) || (Recovery_from_file.tellg()==std::streampos(0)) || (Recovery_from_file.tellg()==std::streampos(-1)) )
// tellg()==0 indicates empty file -> we cannot recover
// tellg()==-1 indicates failure -> we cannot recover
// remark:
//  in gcc 3.4 (cvs-experimental-2003-10-17 we cannot compare with "<" !!)
//  do we really need a workaround to check this condition? 
#else
      if ( (!Recovery_from_file) || (Recovery_from_file.tellg()<1) )
#endif /* STL_STREAM_workaround */
	{
	  cerr << "Recovery not possible!" << endl;
	  exit(1);
	}
      
      Recovery_from_file.seekg(0,ios::beg);
      Recovery_to_file.seekp(0,ios::beg);
      Recovery_from_file >> n; // retrieve number
      Recovery_from_file.ignore(1,'\n');

      cout << "Continue factorization for " << endl;
      cout << n << endl;

      // mark recovery-mode in Factorization-File! One never knows, what has
      // happened to that file in the mean time!
      Factorization_to_file << " [RECOVERY] "; // flush not necessary... 
      FoundFactors.AutoLoad();
    }
  else
    {
      // First start of this factorization (i.e. not a recovery)
      // open streams
      StaticRelations::FileBuffer.open(StaticRelationsFile.c_str(),ios::out|ios::trunc);
      SpecialRelations::FileBuffer.open(SpecialRelationsFile.c_str(),ios::in|ios::out|ios::trunc);
      if (!DynamicRelations::FileBuffer.open(DynamicRelationsFile.c_str(),ios::in|ios::out|ios::trunc))
       {
         cerr << "Cannot access " << DynamicRelationsFile << endl;
         exit(1);
       }

      Recovery_buffer.open(RecoveryFile.c_str(),ios::in|ios::out|ios::trunc);
      
      cout_status << "Starting factorization for" << endl;
      cout_status << n << endl;
      
      Factorization_to_file << endl << "Factorization of " << argv[1] << ":" << endl;
      Factorization_to_file << n << " = " << endl;
      
      easy_factor(); // remove "easy" detectable factors
      
#ifdef VERBOSE_NOTICE
      cout_status << "Starting factorization with ECM and MPQS for" << endl;
      cout_status << n << endl;
#endif
      Recovery_to_file << n << endl;
      unlink("ecm-recover.dat"); // remove any existent ecm recovery file
    }
  ExitManager::register_exithandler(cleanup_files); // on successful exit delete the files

  signal(SIGINT, signalhandler);
  tune_parameters(mpz_sizeinbase(n,10));

#ifdef NOTIFY_PARENT
  // notify parent that now the distributed phase starts
  // and the server will process client connections
  kill(getppid(),SIGUSR1); // send SIGUSR1 to the parent process
#endif

  CXML_StatusServer::install_XML_StatusServer();
  if (!SkipECM)
   {
#ifdef VERBOSE_NOTICE
     cout << "Waiting for net-clients (ECM)..." << endl;
#endif
     Cprocess_clients::process_data_stream_ecm(my_connection_waiter); // start distributed ECM...
   }


#if defined(USE_DFT) || (CONTINUATION_METHOD==2)
  // at this place, all possible functions that made possible use of
  // discrete fast fourier transform have been finished.
  //  --> we can release the DFT-object and associated auxiliary memory...
  polynomial::clear_dft_tempmemory();
#endif /* USE_DFT */

  // now prepare everything for the Quadratic Sieve...

  determine_best_MPQS_Multiplier(n,kN,MPQS_Multiplier); // calculate a good/optimal value for fastest possible sieving
  tune_parameters(mpz_sizeinbase(n,10)); // tune parameters for n


  
  if ( sqrt(mpz_get_d(kN)) < PhysicalSieveSize )
    {
      cerr << "Sieve size too big (you may want to reduce its size)!" << endl;
      exit(1);
    }

  // Set a threshold for Double-Large-Primes,
  // this is the square of the maximal Single Large Prime...
  mpz_init(CmpqsFactor::DLP_Threshold);
  mpz_set_ui(CmpqsFactor::DLP_Threshold,SingleLargePrime_Threshold);
  mpz_mul(CmpqsFactor::DLP_Threshold,CmpqsFactor::DLP_Threshold,CmpqsFactor::DLP_Threshold);
  
  StaticFactorbase::compute_StaticFactorbase();

  // the server does not sieve anything, so initialize the sieve only, if
  // the program is not compiled in server mode


  Polynom.compute_first_polynomial(kN,LogicalSieveSize); // compute the first MPQS polynomial to sieve with

  
  if (recover)
    {
      StaticRelations::Load(); // read in Static Relations
      DynamicRelations::Load(); // read in Dynamic Relations
      SpecialRelations::Load();
      statistical_data::ProgressStats.take_sample(); // first statistical sample right here
        // CycleSearch triggers taking a sample when finding a cycle,
        // but this sample wouldn't contain the full DLP size (see CycleSearch)!
        // So we need the first sample taken before the first CycleSearch...
      SpecialRelations::CycleSearch();
  
#ifdef SAFEMODE
      // are all the relations valid?
      CRelation::SanityCheckRelationsFile(StaticRelationsFile);
      CRelation::SanityCheckRelationsFile(DynamicRelationsFile);
      CRelation::SanityCheckRelationsFile(SpecialRelationsFile);
#endif      
      streampos fpos = Recovery_from_file.tellg();
      Polynom.load_if_available(Recovery_from_file); // get current polynomial (to continue factorization)
      Recovery_to_file.seekp(fpos);
    }

  display_StatusLegend();

  // install seek_emergency_handler for handling subtle seek errors... (see Cprocess_clients.cc)
  CRelation::seek_emergency_handler = Cprocess_clients::seek_emergency_handler;

#ifdef VERBOSE_NOTICE
  cout << "Waiting for clients (MPQS)..." << endl;
#endif
  //cout << "I'm the main thread with id = " << pthread_self() << endl;
  while (true)
   {    
     int* pdata = new int;
     *pdata = my_connection_waiter.get_new_client_socket_descriptor();
     pthread_attr_t detached_thread;
     pthread_t thread_process_client_data;

     pthread_attr_init(&detached_thread);
     pthread_attr_setdetachstate(&detached_thread, PTHREAD_CREATE_DETACHED);

     //cout << "creating thread..." << endl;
     const int retcode = pthread_create(&thread_process_client_data,&detached_thread,
                          Cprocess_clients::THREAD_process_data_stream, pdata);
     if (retcode != 0)
      {
        cerr << "pthread_create failed!" << endl;
	delete pdata;
        exit(1);
      }
   }



#ifdef VERBOSE_INFO  
  cout << endl << "Session ended successfully." << endl;
#endif
  return 0;
}
