/*! @file
 * @brief
 * contains some types and constants for qsieve.H and qsieve.cc
 */

#ifndef QSIEVE_FWD_HEADER_
#define QSIEVE_FWD_HEADER_

#include "utils.H"

const int server_port = 12364;
// server port for networked distributed factoring

const int XML_status_port = 12370;
// port for accessing XML status information

const char* const QsieveLogon ="Qsieve3-Logon";
// Logon command for accessing the server

const int SingleLargePrime_Threshold = 1000000000;
// up to this this value the sieved single large primes are
// accepted as such. Only SLP (single large primes) up to this
// limit are inserted into the DynamicFactor set.

const int DynamicFactor_SievingThreshold = MIN(SingleLargePrime_Threshold,100000000);
// up to this value we also actively sieve with SingleLarge-Primes.
// (If the value is smaller than SingleLargePrime_Threshold, then
//  there is a cut-off in sieving functions for dynamic factors.)

#define TSIEVEELEMENTSIZE 1 /* 1 = signed char, 2 = short signed int, 4 = signed int, 0 = no assembler inline code (=generic code) */
// this define triggers some types to be set to approriate values.
// if you don't define it here, a default value will be used, which is perfectly okay.
// if ASM_386 is defined, then TSIEVEELEMENTSIZE defines the type which is used in inline-assembler subroutines


/*!
   @short
   size of sieve interval in cacheable memory for fast access

   The logical sieve interval is sieved in separate smaller parts, because
   memory access in cache can be up to 200 times faster than uncached memory
   access (cost of a cachemiss). The value of @p PhysicalSieveSize is
   ideally the amount of memory that fits into processor cache while not
   replacing too much other cacheable data.

   The relations are sieved in logical intervals [-M,M] using a loop of
   physical intervals of PhysicalSieveSize.
   PhysicalSieveSize must be set at compile time and is constant.

   If cache effects wouldn't matter, then a greater value of
   PhysicalSieveSize would be better than a smaller one. But cache effects
   do matter! A cache miss can cost between 10 and 200 instruction cycles!

   Smaller sieve sizes increase computational overhead, but they
   also increase cache efficency.

   However, you should use a multiple of 1024 to be on the safe side.
   If optimized assembler code comes into play, this constraint MUST be met!
 */
const int PhysicalSieveSize = 64*1024; // = 64KB



class StaticFactorbaseSettings
{
public:
  typedef unsigned short int FBsizetype; // sizetype of the maximum size of factorbase
  //typedef int FBsizetype; // sizetype of the maximum size of factorbase

  // If the factorbase should have not more than 65535 elements,
  // you should set FBsizetype to "unsigned short int" to save memory;
  // otherwise "int" is a good choice (thereby reducing implicit type casts).
  //
  // "FBsizetype" has *only* to do with the number of elements in the static
  // factorbase. It has *nothing* to do with the type (and the size) of the
  // prime numbers in the static factorbase itself!

 static const FBsizetype MaxSize = 65000; /* maximum size of the static factorbase */
 /* The bigger the number to factorize, the bigger the factorbase should be. 
    Relations can be found easier using a larger factorbase, but also more
    relations are needed to solve the system of equations.  The consumed
    space for the linear system of equations grows quadratic with the size
    of the factorbase.

    Optimal settings depend on the speed and memory provided by the
    computers in use. The more memory (RAM+harddisk) is available and the
    faster this memory can be accessed, the larger factorbases can be
    handled efficiently. You have to experiment...

    Limits are defined by StaticFactorbase::MaxSize, which is constant at
    compile time. For 32 MB (real) RAM, 20000 is an acceptable setting.
 */

protected:
 static int PrimeNumbers[MaxSize] __attribute__ ((aligned (64)));
 static int PrimeNumberReciprocals[MaxSize] __attribute__ ((aligned (64)));
 static float PrimeNumberFloatReciprocals[MaxSize] __attribute__ ((aligned (64)));
 static FBsizetype Size_StaticFactorbase;
 static int biggest_Prime_in_Factorbase;
public:
 static const FBsizetype Size() { return Size_StaticFactorbase; }
 static const int StaticPrime(const int i) { return PrimeNumbers[i]; }
 static const int BiggestPrime() { return biggest_Prime_in_Factorbase; }

 static const int max_additional_Powers = 1000; // maximum count of additional powers of primes, with which sieving is done

 friend void tune_parameters(const unsigned int);
};

#endif /*  QSIEVE_FWD_HEADER_ */
