// polynomial arithmetic
// last change: 2004-09-13

#ifndef POLYNOMIAL_HEADER_
#define POLYNOMIAL_HEADER_

/*! @file
 * @brief
 * declaration of polynomial arithmetic interface for multiple precision numbers
 */

#include <gmp.h>

//! contains polynomial arithmetic concepts for mpz_t
namespace polynomial
{

//#define DEBUG    /* for explicit DEBUGGING-sessions */
//#define VERBOSE  /* be more verbose */

// this define should be triggered by Makefile
// #define USE_DFT /* enable use of discrete fourier transform */
#if !defined(USE_DFT) && (CONTINUATION_METHOD==2)
 #define USE_DFT /* enable use of discrete fourier transform */
#endif

typedef mpz_t* TPolynom;
typedef const mpz_t* TconstPolynom;
/* A polynomial is defined by an array of mpz_t-coefficients:
   P(x)=a0*x^0+a1*x^1+...+a[k-1]*x^[k-1],
   stored as (a0,a1,a2,...,a[k-1]).
*/

/* Hints for function calls:

 TPolynom denotes a pointer to an array of coefficients
 (which thereby defines a polynomial).

 As usual in C++, take respect to the following calling conventions:

 Parameter		Restrictions/Properties
 -------------------------------------------------------------
 mpz_t* p		 mutable pointer, mutable data
 const mpz_t* p		 mutable pointer, constant data
 mpz_t* const p		 constant pointer, mutable data
 const mpz_t* const p    constant pointer, constant data

 "const TPolynom P" and "TPolynom const P" are (unfortunately) equivalent.
 Both define a constant pointer to a mutable polynomial!

 For this reason, I have defined an auxiliary datatype
 for constant polynomials: TconstPolynom
 So keep this in mind:

 Parameter              Restrictions/Properties
 ----------------------------------------------------------------
 TPolynom P              P mutable,     coefficients mutable
 const TPolynom P        P constant,    coefficients mutable
 TPolynom const P	 P constant,    coefficients mutable
 TconstPolynom P         P mutable,     coefficients constant
 const TconstPolynom P	 P constant,    coefficients constant
 TconstPolynom const P   P constant,    coefficients constant

*/


/*!
 * @short
 * a tiny helper class for temporary arrays mpz_t[] in C++
 *
 * self initializing and self destructing.
 * This class is here to circumvent automatic arrays (on stack) that
 * are a gnu extension, but not allowed according to ISO standard.
 */
class TTempPolynom : private ForbidAssignment
{
private:
  mpz_t* const data;
  const int _capacity;
public:
  explicit TTempPolynom(const int n) : data(new mpz_t[n]), _capacity(n)
   {
     //cout << "allocate polynomial of size " << n << endl;
     for (int i=0; i<n; ++i) mpz_init(data[i]);
   }
  TTempPolynom(const int n, const int estimated_operand_size) : data(new mpz_t[n]), _capacity(n)
   {
     //cout << "allocate polynomial of size " << n << endl;
     for (int i=0; i<n; ++i) mpz_init2(data[i],estimated_operand_size);
   }
  ~TTempPolynom()
   {
     //cout << "dispose polynomial of size " << _capacity << endl;
     for (int i=_capacity-1; i>=0; --i) mpz_clear(data[i]);
     delete [] data;
   }
  const int capacity() const { return _capacity; }

  //const mpz_t& operator[] (const unsigned int i) const { return data[i]; }
  //mpz_t& operator[] (const unsigned int i) { return data[i]; }

  // dangerous, but helpful:
  // remember: TPolynom will not live longer than TempPolynom!
  operator const TPolynom() const { return data; }
};


/*!
 * @param P polynomial
 * @param k size of polynomial (numer of coefficients)
 * @remark
 *	   - This function prints the given polynomial to stdout.
 */
void print (const TconstPolynom P, int k);


/*!
 * @param res result value
 * @param P polynomial to evaluate
 * @param k size of @p P
 * @param x point at which @p P is to evaluate
 * @param m result will be reduced modulo @p m
 * @remark
 *	   - This function implements the horner scheme
 *         - use it only for single point evaluations of @p P
 *         - @p res = P(x) mod m
 */
void eval(mpz_t res, const TconstPolynom P, const int k, const mpz_t x, const mpz_t m);
// berechnet mit dem Hornerschema res = P(x) mod m


#ifdef USE_DFT
 // fast discrete fourier transformation
 // (will allocate internal temporary memory, which you may want to release)

/*!
 * @remark
 *	   - When the discrete fast fourier transform is activated, then it
 *           will use some internal temporary memory, which can be safely
 *           released using this function.
 */
void clear_dft_tempmemory() ;

#endif


/*!
 * @param Pr result polynomial
 * @param kr size of result polynomial
 * @param P1 first multiplicator polynomial
 * @param k1 size of @p P1
 * @param P2 second multiplicator polynomial
 * @param k2 size of @p P2
 * @result size of the product polynomial @p Pr
 * @remark
 *	   - This function implements the classical multiplication
 *         - complexity O(k1*k2)
 *         - use it only for small sized polynomials
 *         - @p &Pr must be disjoint from @p &P1 and @p &P2
 */
int classic_mul(const TPolynom Pr, const int kr,
                const TconstPolynom P1, const int k1,
                const TconstPolynom P2, const int k2);
// Pres = P1*P2, &Pres must be different from &P1,&P2
// returns degree of the resulting polynomial
// complexity: O(k1*k2)
// (classical method, implemented alternatively to Karatsuba for small polynomials)


/*!
 * @param Pr result polynomial
 * @param kr size of result polynomial
 * @param P1 first multiplicator polynomial
 * @param k1 size of @p P1
 * @param P2 second multiplicator polynomial
 * @param k2 size of @p P2
 * @param m the result will be reduced modulo @p m
 * @result size of the product polynomial @p Pr
 * @remark
 *	   - This function implements the classical multiplication
 *         - complexity O(k1*k2)
 *         - use it only for small sized polynomials
 *         - @p &Pr must be disjoint from @p &P1 and @p &P2
 */
int classic_mul(const TPolynom Pr, const int kr,
                const TconstPolynom P1, const int k1,
                const TconstPolynom P2, const int k2, const mpz_t m);
// Pres = P1*P2, &Pres must be different from &P1,&P2
// returns degree of the resulting polynomial
// complexity: O(k1*k2)
// (classical method, implemented alternatively to Karatsuba for small polynomials)



/*!
 * @param R result polynomial
 * @param kR size of result polynomial
 * @param P multiplicator polynomial
 * @param k size of @p P
 * @param m the result will be reduced modulo @p m
 * @result size of the product polynomial @p R
 * @remark
 *	   - This function implements the Karatsuba multiplication
 *         - complexity O(k^1.59)
 *         - @p &R must be disjoint from @p &P
 */
int square(const TPolynom R, const int kR, const TconstPolynom P, const int k, const mpz_t m);
// R = P^2, &R must be different from &P
// returns degree of resulting polynomial
// complexity: O(k^1.59) -> Karatsuba


/*!
 * @param R result polynomial
 * @param kR size of result polynomial
 * @param P first multiplicator polynomial
 * @param k size of @p P1
 * @result size of the product polynomial @p R
 * @remark
 *	   - This function implements the Karatsuba multiplication
 *         - complexity O(k^1.59)
 *         - @p &R must be disjoint from @p &P
 */
int square(const TPolynom R, const int kR, const TconstPolynom P, const int k);
// R = P^2, &R must be different from &P
// returns degree of resulting polynomial
// complexity: O(k^1.59) -> Karatsuba


/*!
 * @param R result polynomial
 * @param kR size of result polynomial
 * @param P1 first multiplicator polynomial
 * @param k1 size of @p P1
 * @param P2 second multiplicator polynomial
 * @param k2 size of @p P2
 * @param m the result will be reduced modulo @p m
 * @result size of the product polynomial @p R
 * @remark
 *	   - This function implements the various multiplication algorithms
 *         - complexity O(max(k1,k2)^1.59), Karatsuba
 *         - complexity O((k1+k2)*ld(k1+k2)), dft & chinese remaindering
 *         - use it for medium and big sized polynomials
 *         - @p &R must be disjoint from @p &P1 and @p &P2
 */
int mul(const TPolynom R, const int kR,
        TconstPolynom P1, int k1,
        TconstPolynom P2, int k2, const mpz_t m);
// Pres = P1*P2, &R must be different from &P1,&P2
// returns degree of the resulting polynomial
// complexity: O(max(k1,k2)^1.59) -> Karatsuba
// resp. complexity: O((k1+k2)*ld(k1+k2)) -> using (optimal) dft


/*!
 * @param R result polynomial
 * @param kR size of result polynomial
 * @param P1 first multiplicator polynomial
 * @param k1 size of @p P1
 * @param P2 second multiplicator polynomial
 * @param k2 size of @p P2
 * @result size of the product polynomial @p R
 * @remark
 *	   - This function implements the Karatsuba algorithms
 *         - complexity O(max(k1,k2)^1.59), Karatsuba
 *         - use it for medium and big sized polynomials
 *         - @p &R must be disjoint from @p &P1 and @p &P2
 */
int mul(const TPolynom R, const int kR,
        TconstPolynom P1, int k1,
        TconstPolynom P2, int k2);
// Pres = P1*P2, &R must be different from &P1,&P2
// returns degree of resulting polynomial
// complexity: O(max(k1,k2)^1.59) -> Karatsuba


/*!
 * @param R result polynomial
 * @param kR size of result polynomial
 * @param P argument polynomial
 * @param k size of @p P
 * @param m the result will be reduced modulo @p m
 * @param scale (optional, with default 0, no scale)
 * @remark
 *	   - This function computes the reciprocal polynomial of R
 *         - The argument polynomial @p P will be multiplied (shifted) by x^(scale)
 */
void reciprocal(TPolynom R, int &kR, const TconstPolynom P, const int k, const mpz_t m, const unsigned int scale=0);
// computes the reciprocal polynomial of P,
// R must provide enough memory!
// the given polynomial will be scaled using x^scale as multiplier,
// thereby shifting the coefficients by scale places


/*!
 * @param Q result polynomial (quotient)
 * @param kQ size of result quotient polynomial
 * @param R result polynomial (remainder)
 * @param kR size of result remainder polynomial
 * @param P1 dividend polynomial
 * @param k1 size of @p P1
 * @param P2 divisor polynomial
 * @param k2 size of @p P2
 * @param m the result will be reduced modulo @p m
 * @remark
 *	   - This function implements the classical division
 *         - complexity O(k1*k2)
 *         - use it only for small sized polynomials
 *         - @p &Q , @p &R , @p &P1 and @p &P2 should be disjoint from each other
 *         - this function is deprecated
 */
void classic_div(TPolynom Q, int &kQ, TPolynom R, int &kR,
         const TconstPolynom P1, int k1,
         const TconstPolynom P2, int k2, const mpz_t m) __attribute__ ((deprecated));


/*!
 * @param R result polynomial (remainder)
 * @param kR size of result polynomial
 * @param P1 dividend polynomial
 * @param k1 size of @p P1
 * @param P2 divisor polynomial
 * @param k2 size of @p P2
 * @param m the result will be reduced modulo @p m
 * @remark
 *	   - This function implements the classical division
 *         - complexity O(k1*k2)
 *         - use it only for small sized polynomials
 *         - @p &Pr should be disjoint from @p &P1 and @p &P2
 */
void classic_mod(TPolynom R, int &kR,
            const TconstPolynom P1, int k1,
            const TconstPolynom P2, int k2, const mpz_t m);
// Remainder of polynomial division, O(n^2)
// returns P1 mod P2 in R


/*!
 * @param Q result polynomial (quotient)
 * @param kQ size of result polynomial
 * @param P1 dividend polynomial
 * @param k1 size of @p P1
 * @param P2 divisor polynomial
 * @param k2 size of @p P2
 * @param m the result will be reduced modulo @p m
 * @remark
 *	   - This function implements division using newton method (using reciprocal polynomial)
 *         - complexity depends on fastest multiplication, should be O(max(k1,k2)*ld(max(k1,k2))
 *         - @p &Pr should be disjoint from @p &P1 and @p &P2
 */
void div(TPolynom Q, int &kQ,
         const TconstPolynom P1, const int k1,
         const TconstPolynom P2, const int k2, const mpz_t m);
// (fast) polynomial division by multiplying with the reciprocal polynomial


/*!
 * @param R result polynomial (remainder)
 * @param kR size of result polynomial
 * @param P1 dividend polynomial
 * @param k1 size of @p P1
 * @param P2 divisor polynomial
 * @param k2 size of @p P2
 * @param m the result will be reduced modulo @p m
 * @remark
 *	   - This function implements division using newton method (using reciprocal polynomial)
 *         - complexity depends on fastest multiplication, should be O(max(k1,k2)*ld(max(k1,k2))
 *         - use it for medium and large sized polynomials
 *         - @p &Pr should be disjoint from @p &P1 and @p &P2
 */
void mod(TPolynom R, int &kR,
         const TconstPolynom P1, int k1,
         const TconstPolynom P2, int k2, const mpz_t m);
// polynomial remainder, fast(er) method
// returns P1 modulo P2 in R
/* asymptotically faster than the "naive modulo", because we can make use of
   fast multiplication algorithms. */


/*!
 * @param res result coefficients
 * @param P polynomial to evaluate
 * @param k size of @p P
 * @param array_of_arguments points at which @p P will be evaluated
 * @param size number of points to evaluate
 * @param m the results will be reduced modulo @p m
 * @remark
 *	   - This function implements fast evaluation schemes
 *         - for medium and big size faster than horner evaluation
 */
void multipoint_eval(mpz_t* res,
                     const TconstPolynom P, const int k, 
                     const mpz_t* const array_of_arguments, const int size,
                     const mpz_t m);


/*!
 * @param res result polynomial (coefficients)
 * @param roots_array roots of polynomial
 * @param size number of roots
 * @param m the results will be reduced modulo @p m
 * @result size of result polynomial
 * @remark
 *	   - This function implements fast evaluation schemes
 *         - @p res must be set to NULL when function is called
 */
int construct_polynomial_from_roots
      (TPolynom &res,
       const mpz_t* const roots_array, const int size,
       const mpz_t m);
  // task:
  // Creates a new polynomial using the zeros given in "roots_array";
  // this polynomial will be placed in "res" and its size will be returned.
  // To avoid memory leaks (due to erroneous calls), we expect that
  // "res" is initially an NULL-pointer.
  // additional remark:
  // Sure, it would be possible to return this pointer instead of using a reference
  // parameter, but what if you forget to delete it later? -- Using this technique,
  // the programmer is at least urged to provide a valid container for our data!

} // namespace polynomial

#endif /* POLYNOMIAL_HEADER_ */
