/*! @file
 * @brief
 * main implementation file of Qsieve net-client
 */


#include "at_startup.H"

// sanity checks
#ifndef IS_CLIENT
 #error "The net-client is a client!"
#endif

#ifdef IS_SERVER
 #error "The net-client is not a server!"
#endif

#ifndef USE_NETWORK
 #error "The net-client uses network features!"
#endif

#ifdef IS_STANDALONE
 #error "The net-client is not standalone!"
#endif



extern "C"
{
 #include <unistd.h>
}

#include <cstdio>
#include <cstdlib>
#include <fstream>
#include <sstream>
#include <cmath>
#include <gmp.h>
#include <list>
#include <set>
#include <ctime>

#include "qsieve.H"
#include "StaticFactorbase.H"


#include <vector>
#include <stack>
#include <algorithm>

extern "C"
{
 #include <sys/utsname.h>
 #include <pthread.h>
 #include <sys/wait.h>
 #include <unistd.h>
}

#include <csignal>
#include "mutex.H"


#ifdef NOTIFY_PARENT
 #warning "undefining NOTIFY_PARENT, since this is not a networked server"
 #undef NOTIFY_PARENT
#endif


const string StaticRelationsFile  = "static_relations.dat";
const string SpecialRelationsFile = "special_relations.dat";


string communication_name;

#include "Client_IO.H"

static socket_piper SocketPiper;
ostream& communication_stream = *SocketPiper.detach_OutPipe(256*1024); // request 256 KB output buffer
istream& CClientRelation_Delivery::PipeInput = *SocketPiper.detach_InPipe(16*1024); // as the bottleneck is the network connection to the server, the input buffer can be small


#include "mpqsPolynom.H"


mpz_t n, // number to factorize (will be reduced during factorization)
      kN; // input for MPQS (includes a suitable multiplier)


#include "StaticRelations.H"
#include "DynamicRelations.H"
#include "SpecialRelations.H"

#include "Sieving.H"
#include "ConfigFile.cc"


CmpqsPolynom Polynom; // object to manage polynomial computations for multipolynomial sieve (MPQS)
#include "mpqsStatistics.cc" // statistical stuff about found relations, client connections, etc.


#include "modulo.H" // modulo operations for unsigned int
using namespace numtheory;

#include "polynomial.H" /* for fast polynomial arithmetic & fft-continuation */
#include "fft_param.cc"  /* discrete fast fourier transform */
#include "elliptic_curve.H" /* invariant part of elliptic curve method (ecm) */
#include "elliptic_curve-variant.cc" /* variant part of elliptic curve method (ecm) */

#include "StaticRelations.cc"
#include "CRelation-inc.cc"
#include "Client_IO.cc"


void kill_all_other_threads()
{
  //cout << "Killing explicitly all other threads..." << endl;
  //pthread_kill_other_threads_np();
}


void cleanup_memory()
{
#ifdef VERBOSE_INFO
  cout << "cleanup allocated memory" << endl;
#endif
  StaticRelations::cleanup_memory();
  mpz_clear(kN); mpz_clear(n);
}



int main(const int argc, const char* const argv[])
{

#ifdef USE_NCURSES
  new Cncursed(); // trigger activation of ncursed streams
#endif

  PrintHeader("Qsieve net-client");
    
  if (argc!=2)
    {
      cerr << "servername expected!" << endl;
      exit(1);
    }

  atexit(kill_all_other_threads);
  cout.setf(ios::fixed); // fixed decimal notation, not scientific notation!

  mpz_init(n); // our number to factorize
  mpz_init(kN);

  atexit(cleanup_memory); // on successful exit free allocated data

  Read_ConfigFile();

  communication_name = argv[1];

  CClientRelation_Delivery::init();

  { 
    // first execute the elliptic curves (net-client):

    // we should try to continue an aborted ECM, if possible
    {
      ifstream in("ecm-recover.dat");
      if (in) // does a recovery file exist?
       {
        in.close();
        elliptic_curves elliptic_curve; // create curve
        elliptic_curve.go(-1,elcu_Phase1,elcu_Phase2); // try to recover curve and retry factorization with elliptic curve
       }
    }

    int ecm_sigma = -1, runde=0;
    while (true)
     {
       unix_io_stream communication_stream(communication_name, server_port);
 waitloop:
       communication_stream << "ECM?" << endl;
       communication_stream >> ecm_sigma;
       if (ecm_sigma==-3)
        {
          // -3 is special token for idle waiting loop
          string s;
          communication_stream >> s;
          cout << s << endl;
          goto waitloop;
        }

       if (ecm_sigma<0) break; // sigma<0 -> curves are done!
       communication_stream >> elcu_Phase1 >> elcu_Phase2;
       communication_stream >> n; // get number
       cout << "elliptic curve " << ++runde << endl;
       elliptic_curves elliptic_curve; // create ecm object
       elliptic_curve.go(ecm_sigma,elcu_Phase1,elcu_Phase2); // try factorization using this curve
     }
  }
  
  {
    unix_io_stream communication_stream(communication_name, server_port);
    communication_stream << QsieveLogon << endl;
    communication_stream >> n;
    communication_stream >> StaticRelations::Size_StaticFactorbase;
    communication_stream >> Factor_Threshold;
    communication_stream >> LogicalSieveSize;
#ifdef VERBOSE_NOTICE
    cout << "n=" << n << endl;
    cout << "Size of static factorbase: " << StaticFactorbase::Size() << endl;
    cout << "Factor-Threshold (Exponent): " << Factor_Threshold << endl;
    cout << "Sieveinterval per polynomial: [" << -LogicalSieveSize << "," << LogicalSieveSize << "]" << endl;
#endif
  }

  determine_best_MPQS_Multiplier(n,kN,MPQS_Multiplier); // optimal multiplier value to sieve with
  
  if ( sqrt(mpz_get_d(kN)) < PhysicalSieveSize )
    {
      cerr << "Sieve size too big (you may want to reduce its size)!" << endl;
      exit(1);
    }

  // Set a threshold for Double-Large-Primes,
  // this is the square of the maximal Single Large Prime...
  mpz_init(CmpqsFactor::DLP_Threshold);
  mpz_set_ui(CmpqsFactor::DLP_Threshold,SingleLargePrime_Threshold);
  mpz_mul(CmpqsFactor::DLP_Threshold,CmpqsFactor::DLP_Threshold,CmpqsFactor::DLP_Threshold);
  
  StaticFactorbase::compute_StaticFactorbase();
  CSieveStaticFactors::initialize();
  SieveControl::compute_SieveThreshold(); // Threshold for detecting relations during sieving phase
  for (int i=0; i<64; ++i) SieveArray_[i]=-1; // initialize array underflow trigger values
  Polynom.compute_first_polynomial(kN,LogicalSieveSize); // compute the first MPQS polynomial to sieve with

  display_StatusLegend();

  mpz_t UpperBound_D;
  mpz_init(UpperBound_D);

  while (true)
   {
     // main loop: get MPQS interval and sieve them

     CClientPolynomFetcher::fetch(UpperBound_D);

     do // sieve relations
      {
        if (StaticRelations::Count() > 4*StaticFactorbase::Size())
         {
           cerr << "#static relations > 4*StaticFactorbase::Size(): this is weird." << endl;
           cerr << "I will sleep 30 seconds for transmission to end, and then I will abort." << endl;
           sleep(30);
           exit(1);
         }
        CClientDynamicFactorFetcher::fetch();
        do_sieving();
        if (!communication_stream)
         {
           MARK;
           cerr << "communication stream is in error state! aborting..." << endl;
           exit(1);
         }
        Polynom.compute_next_polynomial();
      } while (Polynom < UpperBound_D);

   }

  // these lines are unreachable as long as nobody breaks the while(true) loop
  mpz_clear (UpperBound_D);
  return 0;
}
