#ifndef mutex_header
#define mutex_header

/*! @file
 * @brief
 * contains wrapper class for mutex (mutual exclusion) handling
 * in a C++ multithreaded environment
 */


// possible problem:
// LinuxThreads have various non-POSIX types of mutexes. Default mutex type
// is "fast": On "unlock()" it is not checked, whether the caller is the
// owner of the mutex! So any thread can unlock the mutex. You shouldn't
// rely on this "feature". In fact, it has a very serious drawback: In a
// try-catch block you must know the exact state of the mutex, otherwise
// other threads may sneak into critical sections or a deadlock occurs!


extern "C"
{
 #include <pthread.h>
}

#include "utils.H"


/// wrapper class for the Mutual-Exclusion (Mutex) mechanism
class CMutex : private ForbidAssignment
{
 private: 
  pthread_mutex_t mutex;
 public:
  inline CMutex()
   {
     pthread_mutex_init(&mutex,NULL);
   }
  inline ~CMutex()
   {
#ifndef DEBUG
     pthread_mutex_destroy(&mutex);
#else /* only for debugging */
     const int result = pthread_mutex_destroy(&mutex);
     if (result==EBUSY)
      {
        using std::cerr; using std::endl;
        cerr << "cannot destroy busy mutex!!" << endl;
        exit(1);
      }
#endif
   }
  inline void lock()
   {
     const int result = pthread_mutex_lock(&mutex);
     switch (result)
      {
        using std::cerr; using std::endl;
        case EINVAL:
          cerr << "the mutex has not been properly initialized." << endl;
          exit(1);
	case EDEADLK:
	  cerr << "the mutex is already locked by the calling thread" << endl;
	  exit(1);
      }
   }
  inline const bool trylock()
   {
     const int result = pthread_mutex_trylock(&mutex);
     switch (result)
      {
        using std::cerr; using std::endl;
        case EINVAL:
          cerr << "the mutex has not been properly initialized." << endl;
          exit(1);
	case EBUSY:
#ifdef VERBOSE
	  cerr << "the mutex could not be acquired because it was currently locked." << endl;
#endif
	  return false;
      }
     return true;
   }
  inline void unlock()
   {
     const int result = pthread_mutex_unlock(&mutex);
     switch (result)
      {
        using std::cerr; using std::endl;
        case EINVAL:
          cerr << "the mutex has not been properly initialized." << endl;
          exit(1);
        case EPERM:
          cerr << "the calling thread does not own the mutex" << endl;
          exit(1);
        case EBUSY:
          cerr << "the mutex is currently locked." << endl;
          exit(1);
      }
   }
};


/*!
 * The CUnlockMutexAtDestruction registers a mutex on construction and
 * unlocks it at destruction.
 * It can be used to guarantee, that the registered mutex is unlocked after
 * the critical section protected by this mutex is left (regardless, whether
 * by a normal ending of the block, a return call or because an exception is
 * thrown).
 */
class CUnlockMutexAtDestruction : private ForbidAssignment
{
 private:
  CMutex &mutex;
 public:
  CUnlockMutexAtDestruction(CMutex &myMutex) : mutex(myMutex) { }
  ~CUnlockMutexAtDestruction() { mutex.unlock(); }
};


/*!
 * The CConditionalUnlockMutexAtDestruction registers a mutex on
 * construction and unlocks it at destruction, iff (that is: if and only if)
 * it has been said to do so. It can be used to guarantee, that the
 * registered mutex is unlocked after the critical section protected by this
 * mutex is left (regardless, whether by a normal ending of the block, a
 * return call or because an exception is thrown).
 */
class CConditionalUnlockMutexAtDestruction : private ForbidAssignment
{
 private:
  CMutex &mutex;
  bool condition;
 public:
  CConditionalUnlockMutexAtDestruction(CMutex &myMutex, const bool initial_condition)
   : mutex(myMutex), condition(initial_condition) { }
  ~CConditionalUnlockMutexAtDestruction() { if (condition) mutex.unlock(); }
  void set_condition(const bool new_condition = true) { condition=new_condition; }
};


/*!
 * An instance of this class can be used to define a block as a
 * critical section which is locked by a given mutex:
 *  - on construction of the instance the critical section is entered.
 *  - on destruction of the instance the critical section is left.
 *
 * It is also a good idea to use this class in conjunction with exception
 * handling, because the traditional way of locking and unlocking a mutex
 * may introduce severe problems, if exceptions are thrown and the state of
 * a mutex is left undefined. (Destructors are called regardless whether the
 * block is left with or without a throwing an exception, so the state of the
 * mutex remains always well defined.)
 */
class CCriticalSection : private ForbidAssignment
{
 public:
  class Mutex : private CMutex // private inheritance to forbid usage outside this class
   {
     friend class CCriticalSection; // the owning class is allowed to call methods!
   };

 private:
  Mutex &mutex;
  bool inside;

 public:

  /// enter the critical section (it is also allowed to enter when already inside)
  inline void enter()
   {
#ifdef DEBUG
     if (inside) std::cerr << "CCriticalSection::enter():  already entered" << std::endl;
#endif
     if (!inside) mutex.lock(); // enter the critical section (wait until mutex can be obtained)
     inside=true;
   }

  /// try to enter the critical section and return result
  inline const bool try_enter()
   {
#ifdef DEBUG
     if (inside) std::cerr << "CCriticalSection::enter():  already entered" << std::endl;
#endif
     if (!inside) inside=mutex.trylock(); // try to enter the critical section (but don't wait if mutex cannot be obtained)
     return inside;
   }

  /// leave the critical section (it is also allowed to leave without being inside)
  inline void leave()
   {
#ifdef DEBUG
     if (!inside) std::cerr << "CCriticalSection::leave():  already left" << std::endl;
#endif
     if (inside) mutex.unlock();
     inside=false;
   }

  /// constructor: attach mutex to handle critical sections, also (by default): enter the critical section
  CCriticalSection(CCriticalSection::Mutex &crit_sect_mutex, const bool enter_now=true)
   : mutex(crit_sect_mutex), inside(false)
   {
     if (enter_now) enter(); // enter the critical section
   }

  /// destructor: if we were inside the critical section: leave the critical section
  ~CCriticalSection()
   {
     if (inside) leave(); // leave the critical section
   }

};

#endif
