#ifndef mpz_wrapper_dot_H_header
#define mpz_wrapper_dot_H_header


/*! @file
 * @brief
 * contains wrapped C++ stream I/O for mpz_t (multiple precision numbers)
 */

#include <iomanip>
#include <iostream>
#include <string>
#include <gmp.h>

//! contains wrapped I/O for mpz_t (multiple precision numbers)
namespace my_mpz_wrapper
{

using std::cerr;
using std::endl;


inline unsigned long int mpz_remainder_ui_(const mpz_t n, const unsigned long int d)
{
  // should speedup some easy cases...
  if (d==1 || d==2 || d==4 || d==8 || d==16 || d==32 || d==64 || d==128)
   {
     unsigned long int h = (n->_mp_d[0]);
     return (n->_mp_size > 0) ? h&(d-1)
                              : (n->_mp_size < 0) ? d-(h&(d-1)) : 0;
   }
  else
   return mpz_fdiv_ui(n,d);
}


inline unsigned long int mpz_remainder_ui(const mpz_t n, const unsigned long int d)
{
  // refer gmp manual:
  // "unsigned long int mpz_fdiv_ui(mpz_t n, unsigned long int d)" returns
  // the remainder of the division; except for not needing any mpz_t result
  // value it is identical to
  // "unsigned long int mpz_mod_ui(mpz_t t, mpz_t n, unsigned long int d)".
  
  // I find the name "mpz_fdiv_ui" too much ambiguous.
  // Defining "unsigned long int mpz_mod_ui(mpz_t n, unsigned long int d)"
  // would be somewhat more straight forward, but it is also risky, because
  // one might assume a real gmp-function and not a wrapper function.
  // mpz_remainder_ui can easily be detected as a wrapper function just
  // by grepping the sources...

  // As this is a wrapper, we can add some optimizations, too
 
  return (__builtin_constant_p(d)) ? mpz_remainder_ui_(n,d) : mpz_fdiv_ui(n,d);
}


#if defined(mpz_odd_p)
 // replace mpz_odd_p / mpz_even_p macros with inline-C++
 // (better type checking + more explicit casts) 

#undef mpz_odd_p
#undef mpz_even_p

//! equivalent to mpz_odd_p macro, but more strict
inline int mpz_odd_p(const mpz_t z)
{
  return static_cast<int>(z->_mp_size != 0) & static_cast<int>(z->_mp_d[0]);
}

//! equivalent to mpz_even_p macro, but more strict
inline int mpz_even_p(const mpz_t z)
{
  return !mpz_odd_p(z);
}

#endif


const int probab_prime_checks = 25; // #rounds for mpz_probab_prime, needed at varoius places

const int mpzbase_f = 32; // mpz-I/O format for fileoperations
// remark:
// mpzbase_f=36: highest allowed base (refer gmp manual),
//               compatible with former versions (up to V2.95),
//               less space, but also computational less efficient.
// mpzbase_f=32: is a power of two (and therefore computational more efficient
//               since conversion routines can use shifts instead of expensive
//               divisions; refer gmp source).

// For human readable input and output we use decimal system


//! wrapper function to output mpz_t numbers (to avoid/minimize explicit conversions)
inline std::ostream& operator << (std::ostream& ostr, const mpz_t value)
{ 
  size_t size = mpz_sizeinbase(value,10); 
  char* str = new char [size+2];
  str[size]='\1'; // to suppress warnings about uninitialised value(s) issued by valgrind 
  mpz_get_str(str, 10, value);
  if (str[size])
   {
     if (str[size-1]=='\0') --size;
     else if (str[size+1]=='\0') ++size;
     else
      {
        MARK; exit(1);
      }
   }
#if 1
  ostr << str << " (" << size << ")";
#else
  ostr << str;
  if (size>6) ostr << " (" << size << ")";
#endif
  delete [] str;              
  return ostr;
}

//! input wrapper function for mpz_t numbers
inline std::istream& operator >> (std::istream& istr, mpz_t value)
{ 
  
  std::string s;
  istr >> s;
  if (mpz_set_str(value, s.c_str(), 10))
   {
     MARK;
     cerr << "invalid number read." << endl;
     cerr << "given: *" << s << "*" << endl;
     cerr << "mpz_t: *" << value << "*" << endl;
     istr.setstate(std::ios::failbit);
     //exit(1);
   } else  istr >> s; // read over Sizeinbase
  return istr;
}

} // end of namespace my_mpz_wrapper

#endif /* of mpz_wrapper_dot_H_header */
