#ifndef mpqsPolynom_header
#define mpqsPolynom_header

/*! @file
 * @brief
 * header file for declaration of MPQS polynomials
 *
 * refer "The Multiple Polynomial Quadratic Sieve",
 *       Robert D. Silverman,
 *       Mathematics Of Computation, Vol. 48, January 1987, p. 329-339
 *   for mathematical background.
 */

#include <iosfwd>
#include <gmp.h>
#include "utils.H"
#include "mpz_wrapper.H"
using namespace my_mpz_wrapper;
#include "modulo.H"

using std::ostream;
using std::istream;

using std::cout;
using std::cerr;


/*! switch to define the "active" and "passive" phase of sieving:
 If enough Special-Relations and Dynamic-Relations have been collected to
 sieve Static-Relations at a high rate, then further collecting even more
 Special- and Dynamic-Relations can be suppressed by setting
 collecting_phase_finished=true.
 Normally it is set to false, and one should *not* change it.
 But if the mpqs polynomials collide with the range of dynamic factors,
 then this switch must (and will) be activated.
*/
extern bool collecting_phase_finished;

extern const int SingleLargePrime_Threshold;
extern double Factor_Threshold;

class CmpqsPolynom
{
  /* Implementation of polynomial calculations for MPQS.
     Notation of variables is similar to the one given in
     "The Multiple Polynomial Quadratic Sieve",
     Robert D. Silverman,
     Mathematics Of Computation, Vol. 48, January 1987, p. 329-339
  */  
private:
  mpz_t A,B,C,D,h0,h1,h2;
  mpz_t kN, kN_div2, D2_inv_mod_kN;
  mpz_t A2;
public:
  inline CmpqsPolynom() // constructor
    {
      mpz_init(A); mpz_init(B); mpz_init(C); mpz_init(D);
      mpz_init(h0); mpz_init(h1); mpz_init(h2);
      mpz_init(kN); mpz_init(kN_div2); mpz_init(D2_inv_mod_kN);
      mpz_init(A2);
    }
  inline ~CmpqsPolynom() // destructor
    {
      mpz_clear(A); mpz_clear(B); mpz_clear(C); mpz_clear(D);
      mpz_clear(h0); mpz_clear(h1); mpz_clear(h2);
      mpz_clear(kN); mpz_clear(kN_div2); mpz_clear(D2_inv_mod_kN);
      mpz_clear(A2);
    }
  void compute_first_polynomial(const mpz_t fuer_kN, const int M);
  void compute_next_polynomial(const int step=0);
  void get_values(const signed int SievePos, mpz_t radix, mpz_t Q) const;
  const double get_logval(const signed int SievePos) const;
  void SanityCheck(const signed int SievePos=0);
  void save(ostream &ostr); // for Recovery etc.
  void load(istream &in); // for Recovery etc.
  void load_if_available(istream &in); // for Recovery etc.
  inline bool operator < (const mpz_t K) const 
    {
      return mpz_cmp(D, K)<0;
    }

  // some helper functions to provide read-only access to internal parameters
  inline const unsigned int get_A2_mod(const unsigned int m) const
   {
     // A2 := A*2 = 2*D^2
     unsigned int h = numtheory::squaremod(mpz_remainder_ui(D,m),m)<<1;
     if (h>=m) h-=m;
     return h;
   }
  inline const mpz_t& get_A2() const { return A2; }
  inline const mpz_t& get_B() const { return B; }
  inline const mpz_t& get_D() const { return D; }
};

#endif
