/*! @file
 * @brief
 * main implementation file of Qsieve file-client
 */

// Remark: the file-client joins only the sieving part for mpqs!
// elliptic curves are not processed, since that can be done using
// the qsieve standalone version!


#include "at_startup.H"


// sanity checks
#ifndef IS_CLIENT
 #error "The file-client is a client!"
#endif

#ifdef IS_SERVER
 #error "The file-client is not a server!"
#endif

#ifdef USE_NETWORK
 #error "The file-client uses no network features!"
#endif

#ifdef IS_STANDALONE
 #error "The file-client is not standalone!"
#endif



extern "C"
{
 #include <unistd.h>
}

#include <cstdio>
#include <cstdlib>
#include <fstream>
#include <sstream>
#include <cmath>
#include <gmp.h>
#include <list>
#include <set>
#include <ctime>

#include "qsieve.H"
#include "StaticFactorbase.H"



#include <vector>
#include <stack>
#include <algorithm>


#ifdef NOTIFY_PARENT
 #warning "undefining NOTIFY_PARENT, since this is not a networked server"
 #undef NOTIFY_PARENT
#endif


const string StaticRelationsFile  = "static_relations.dat";
const string SpecialRelationsFile = "special_relations.dat";


string communication_name;
ofstream communication_stream;


#include "mpqsPolynom.H"


mpz_t n, // number to factorize (will be reduced during factorization)
      kN; // input for MPQS (includes a suitable multiplier)


#include "StaticRelations.H"
#include "DynamicRelations.H"
#include "SpecialRelations.H"

#include "Sieving.H"
#include "ConfigFile.cc"


CmpqsPolynom Polynom; // object to manage polynomial computations for multipolynomial sieve (MPQS)
#include "mpqsStatistics.cc" // statistical stuff about found relations, client connections, etc.


#include "modulo.H" // modulo operations for unsigned int
using namespace numtheory;

#include "StaticRelations.cc"
#include "CRelation-inc.cc"


void cleanup_memory()
{
#ifdef VERBOSE_INFO
  cout << "cleanup allocated memory" << endl;
#endif
  StaticRelations::cleanup_memory();
#ifdef VERBOSE_INFO
  cout << "cleanup mpz numbers" << endl;
#endif
  mpz_clear(kN); mpz_clear(n);
#ifdef VERBOSE_INFO
  cout << "mpz numbers cleared." << endl;
#endif
}



int main(const int argc, const char* const argv[])
{

#ifdef USE_NCURSES
  new Cncursed(); // trigger activation of ncursed streams
#endif

  PrintHeader("Qsieve file-client");

  if (argc!=2)
    {
      cerr << "servername expected!" << endl;
      exit(1);
    }

  cout.setf(ios::fixed); // decimal representation, not scientific representation!

  mpz_init(n); // number to factorize
  mpz_init(kN);
  atexit(cleanup_memory); // on successful exit free allocated data

  Read_ConfigFile();

  communication_name = argv[1];

  // does the file exist?
  ifstream communication_stream2(communication_name.c_str());
  if (!communication_stream2)
    {
      cerr << "Unable to open " << communication_name << endl;
      exit (1);
    }
  communication_name+=".out";
  
  communication_stream2 >> n; 
  communication_stream2 >> StaticFactorbase::Size_StaticFactorbase;
  communication_stream2 >> Factor_Threshold;
  communication_stream2 >> LogicalSieveSize;
  communication_stream2.ignore(1,'\n');
#ifdef VERBOSE_NOTICE
  cout << "n=" << n << endl;
  cout << "Size of static factorbase: " << StaticFactorbase::Size() << endl;
  cout << "Factor-Threshold (Exponent): " << Factor_Threshold << endl;
  cout << "Sieveinterval per polynomial: [" << -LogicalSieveSize << "," << LogicalSieveSize << "]" << endl;
#endif
  communication_stream.open(communication_name.c_str(), ios::out|ios::trunc);
  if (!communication_stream)
   {
     MARK; cerr << "opening communication_stream \"" << communication_name << "\" failed!" << endl;
     exit(1);
   }
  determine_best_MPQS_Multiplier(n,kN,MPQS_Multiplier); // optimal multiplier value to sieve with
  
  if ( sqrt(mpz_get_d(kN)) < PhysicalSieveSize )
    {
      cerr << "Sieve size too big (you may want to reduce its size)!" << endl;
      exit(1);
    }

  // Set a threshold for Double-Large-Primes,
  // this is the square of the maximal Single Large Prime...
  mpz_init(CmpqsFactor::DLP_Threshold);
  mpz_set_ui(CmpqsFactor::DLP_Threshold,SingleLargePrime_Threshold);
  mpz_mul(CmpqsFactor::DLP_Threshold,CmpqsFactor::DLP_Threshold,CmpqsFactor::DLP_Threshold);
  
  StaticFactorbase::compute_StaticFactorbase();
  CSieveStaticFactors::initialize();
  SieveControl::compute_SieveThreshold(); // Threshold for detecting relations during sieving phase
  for (int i=0; i<64; ++i) SieveArray_[i]=-1; // initialize array underflow trigger values
  Polynom.compute_first_polynomial(kN,LogicalSieveSize); // compute the first MPQS polynomial to sieve with

  // file-client: write kN to file
  communication_stream << kN << endl;
  
  TDynamicFactorRelation relation;
  relation.fpos = 0; relation.factor=0;
  int diff; // we read the differences between factors instead the factors itself
  communication_stream2 >> diff; // read over filepos
  while (true)
    {
      communication_stream2 >> diff;
      relation.factor+=diff;
      if (diff<=0) break;
      relation.append_for_sieving();
      DynamicFactorRelations.insert(relation);
    }
  
  display_StatusLegend();

  mpz_t UpperBound_D;
  mpz_init(UpperBound_D);
    
  Polynom.load(communication_stream2);
  communication_stream2 >> UpperBound_D;
  if (communication_stream.fail())
   {
     cerr << "No more data from file??" << endl;
     exit(1);
   }
  
  // The ultimate loop:
  do // sieve relations
   {
     do_sieving();
     Polynom.compute_next_polynomial();
   } while (Polynom < UpperBound_D);
  mpz_clear (UpperBound_D);

#ifdef VERBOSE_INFO  
  cout << endl << "Session ended successfully." << endl;
#endif
  return 0;
}
