/*! @file
 * @brief
 * contains factoring algorithms which are related to Fermat's method
 */


void fermat_like_method()
{
  /* Fermat-hnliche Methode:
     Sei N=p*q (hier ausnahmsweise p,q nichtnotwendigerweise prim)
     und sei M=m1*m2 ein Multiplikator, so dass m1*q ungefhr m2*p entspricht.
     Dann ist (m1*q)*(m2*p)=(m1*m2)*(q*p)=M*N.

     Wir definieren R=floor(sqrt(M*N)).
     Der Ansatz ist nun, ein delta zu finden, so dass Q := (R+delta)^2 - M*N
     eine Quadratzahl ergibt, denn dann folgt (R+delta)^2 - Q = 0 (mod N),
     was gleichbedeutend ist zu (R+delta+sqrt(Q))*(R+delta-sqrt(Q)) = 0 (mod N).
     Mit gcd(R+delta+sqrt(Q),N) wre dann ein Teiler von N gefunden...
       
     Diese Methode ist zwar nicht besonders effizient, da sie eher einer
     Probedivision entspricht; sie hilft aber, solche Zahlen besonders
     schnell zu faktorisieren, die auf "allzu naive" Weise als
     "besonders schwer" zu faktorisierende Zahlen konstruiert sind.
     So knnen zum Beispiel Produkte der Form
       N:= p*nextprime(10*p+"kleiner Offset")
     effizient zerlegt werden.
  */
#ifdef VERBOSE_NOTICE
  cout << "starting Fermat factorization method" << endl;
#endif
  if (mpz_sizeinbase(n,10)<50)
   {
     cout << "number too small, it's not worthwhile do proceed; skipping fermat..." << endl;
     return;
   }
  
  mpz_t c,r,h;
  mpz_init(c); mpz_init(r); mpz_init(h);

  const unsigned int k_max = 0x100; // or 0x040 (to be less exhaustive)
  for (unsigned int k=0; k<k_max; ++k)
    {
#ifdef REACT_ON_SIGUSR
      if (USRSignalHandler.got_SIGUSR1()) break;
      if (USRSignalHandler.got_SIGUSR2()) break;
#endif
      mpz_set_ui(c,2*2*3*5*7*11*13*17*19); mpz_pow_ui(c,c,4);
      unsigned int m = 1;
      if (k&0x001) m*=2;
      if (k&0x002) m*=3;
      if (k&0x004) m*=5;
      if (k&0x008) m*=7;
      if (k&0x010) m*=11;
      if (k&0x020) m*=13;
      if (k&0x040) m*=17;
      if (k&0x080) m*=19;
      mpz_mul_ui(c,c,m);

      // Idea behind these multipliers given by example:
      // Let t be a (small) number, then
      //
      //              t^0   t^1   t^2   t^3   t^4
      //  t^4 covers  --- , --- , --- , --- , --- 
      //              t^4   t^3   t^2   t^1   t^0
      //
      //  which is identical to the sequence
      //
      //    1     1     
      //   --- , --- , 1 , t^2 , t^4
      //   t^4   t^2
      //
      // because p*q*t^4 = p*(q*t^4) = (p*t)*(q*t^3) = (p*t^2)*(q*t^2) = ...
      //
      //                        1     1    1
      // Similarly t^5 covers  --- , --- , - , t , t^3, t^5
      //                       t^5   t^3   t

#ifdef VERBOSE_INFO
      cout << "using Fermat prefactor " << c << "             \r" << flush;
#endif
      mpz_mul(c,c,n); mpz_sqrt(r,c); mpz_mul(h,r,r); mpz_sub(h,h,c);
      mpz_mul_ui(r,r,2);
      for (unsigned int delta=1; delta<50000; ++delta)
	{
	  //if (delta%10000==0) cout << "delta=" << delta << "\r" << flush;
	  mpz_add(h,h,r); mpz_add_ui(h,h,1); mpz_add_ui(r,r,2); // compute next square: h=(x+1)^2=x^2+2x+1=h+r+1 (using r=2x)
	  if (mpz_perfect_square_p(h))
	    {
#ifdef VERBOSE_INFO
	      cout << endl << "delta=" << delta << ": Square detected..." << endl;
#endif
	      mpz_div_ui(r,r,2); mpz_sqrt(h,h); mpz_add(h,h,r);
	      mpz_gcd(h,h,n);
	      if (mpz_cmp_ui(h,1)!=0) 
		{
                  mpz_divexact(n,n,h);
		  //cout << "t1=" << h << endl;
		  //cout << "t2=" << n << endl;
		  if (mpz_probab_prime_p(h,probab_prime_checks))
		    {
		      cout << h << " is factor." << endl;
                      std::ostringstream comment;
                      comment << " [fer]";
		      Factorization_to_file << MAL(h,comment) << flush;
		    }
		  else
		    {
		      fermat_like_method(); // try again...
		      mpz_swap(n,h);
		      if (mpz_probab_prime_p(h,probab_prime_checks))
		       {
                         cout << h << " is factor." << endl;
                         std::ostringstream comment;
                         comment << " [fer]";
		         Factorization_to_file << MAL(h,comment) << flush;
		       }
		      else
		       {
		         cout << h << " is a composite factor." << endl;
			 if (Potenztest(h)) Factorization_to_file << " [fer]" << flush;
			 else
                          {
                            std::ostringstream comment;
                            comment << " [fer] [composite]";
                            Factorization_to_file << MAL(h,comment) << flush;
                          }
		       }
		    }
		  if (!mpz_probab_prime_p(h,probab_prime_checks)) fermat_like_method(); // and try again once more...
		  k=k_max; break;
		}
	    }
	}
    }
#ifdef VERBOSE_INFO
  cout << endl;
#endif
  mpz_clear(h); mpz_clear(r); mpz_clear(c);
}


void phimat(mpz_t n)
{
  mpz_t x,y;
  mpz_init(x); mpz_init(y);
  
  mpz_mul_ui(x,n,4); mpz_sqrt(x,x); // x=trunc(2*sqrt(n));
  mpz_add_ui(y,n,1); mpz_sub(y,y,x); // y=n+1-x
  mpz_set_ui(x,2); mpz_powm(x,x,y,n); // x=2^y mod n;

  cout << "trying phimat..." << x << endl;

  //mpz_add_ui(y,n,1); mpz_div_ui(y,y,2);
  unsigned long int z=0;
  while (z<1000000000)
    {
      int p=mpz_scan1(x,0); z+=p; 
      mpz_tdiv_q_2exp(x,x,p);
      //cout << z << ": " << x << endl;
      if (mpz_cmp_ui(x,1)==0) break;
      mpz_add(x,x,n);
    }

  cout << z << ":" << x << endl;
  
  if (mpz_cmp_ui(x,1)==0)
    {
      cout << "x=1 ... " << endl;
      mpz_mul_ui(x,n,4); mpz_sqrt(x,x); mpz_add_ui(x,x,z); // x=p+q
      mpz_mul(y,x,x);
      mpz_sub(y,y,n); mpz_sub(y,y,n); mpz_sub(y,y,n); mpz_sub(y,y,n);
      mpz_sqrt(y,y); // y=p+q
      mpz_add(y,x,y); mpz_div_ui(y,y,2); // y=p
      mpz_div(x,n,y); // x=q
      mpz_mul(x,x,y);
      cout << x << endl; 
      if (mpz_cmp(x,n)==0) // wirklich echte Teiler ermittelt?
	{
	  mpz_div(x,n,y); // x=q
	  if (mpz_probab_prime_p(x,probab_prime_checks))
	    {
	      cout << x << " is factor." << endl;
	      Factorization_to_file << MAL(x) << flush;
	    }
	  else
	    {
	      cout << x << " is a composite factor." << endl;
	      Factorization_to_file << MAL(x) << " [composite]" << flush;
	    }
	  if (mpz_probab_prime_p(y,probab_prime_checks))
	    {
	      cout << y << " is factor." << endl;
	      Factorization_to_file << MAL(y) << flush;
	    }
	  else
	    {
	      cout << y << " is a composite factor." << endl;
	      Factorization_to_file << MAL(y) << " [composite]" << flush;
	    }      
	  mpz_set_ui(n,1);
	}
    }
  
  mpz_clear(x); mpz_clear(y);
}
