#ifndef CmpqsFactor_header
#define CmpqsFactor_header

/*! @file
 * @brief
 * header file for declaration of base class used in DLP-MPQS
 */


#include <iosfwd>
#include <gmp.h>
#include "utils.H"
#include "mpz_wrapper.H"
using namespace my_mpz_wrapper;

using std::ostream;
using std::istream;

using std::cout;
using std::cerr;

extern const int SingleLargePrime_Threshold;


/*!
 * @short
 *  base class to provide an enumeration of to offsprings
 *
 * This class is a base class that provides an enumeration of the different
 * factor types for detecting and transmitting these types.
 * A class providing a double large prime (DLP) for example can also be empty
 * and (therefore contain no factor at all) or it may contain a single large
 * prime or a static prime.
 */
class CmpqsFactortypes
{
 public:
  enum Factortype { empty, static_prime, single_large_prime, double_large_prime, Factortype_size };
};

/*!
 * @short
 *  provides DLP (double large primes) as a data type
 *
 * This class provides the data type DLP (double large prime).
 * It can also handle its subsets (empty, static prime, single large prime).
 * The DLP is represented by its two factors @p p1 and @p p2.
 * Methods to split (and check) a multiple precision number into (for) its DLP
 * representation are also provided.
 */ 
class CmpqsFactor : public CmpqsFactortypes
{
 private:

  /// prime components (constraint in normal representation: @p p1 <= @p p2 )
  unsigned int p1,p2; // with p1<=p2

  /// counter for rejected DLP (while trying to factor them)
  static double rejected_dlp_counter;

 public:

  /// threshold, when a multiple precision integer should be checked for DLP
  static mpz_t DLP_Threshold; // Double-Large-Prime-Threshold; will be initialized in main()

  /// returns, whether @p n is definitely rejected as a DLP
  static inline const bool DLP_rejected(const mpz_t n)
   {
     if (mpz_cmp(n,DLP_Threshold)>0)
      {
        rejected_dlp_counter+=1.0;
        return true;
      }
      // also reject probably primes (but do not count them)
      return mpz_probab_prime_p(n,10)!=0;
   }

  /// try to convert @p n into a DLP (using pollard rho algorithm)
  const bool DLP_get_using_pollard_rho(const mpz_t n); // is somewhat slower than SQUFOF!

  /*!
   * @short try to convert @p n into a DLP (using SQUFOF algorithm)
   *
   * @param n number to convert into DLP
   * @result true, if succeed and false if failed
   *
   * If the result is @p false, this does not mean necessarily that @p n is
   * no DLP. It means that either the given @p n isn't a DLP at all OR
   * that we were not able to factorize @p n in an acceptable amount of time.
   * If the result is @p true, then the object contains now a valid DLP.
   */
  const bool DLP_get(const mpz_t n); // using SQUFOF (with fallback to pollard rho)

  /// assign a single (large prime) integer value to this class
  inline const CmpqsFactor& operator= (const unsigned int x)
   { p1=0; p2=x; return *this; }

  /// if the class contains a single large prime or static prime, return it.
  inline int int_value() const
   {
     // return an integer value (or exit with a error message)
     if (p1)
      {
        cerr << "Cannot assign two integers to a single integer value!" << endl;
        exit(1);
        return 0;
      }
     return p2;
   }

  /// return first DLP factor
  inline int LP1() const { return p1; }

  /// return second DLP factor
  inline int LP2() const { return p2; }

  /*!
   * @short
   *  assign a single (large prime) integer value to this class (but switched representation)
   *
   * This functions inserts the integer value @p x into the object using its
   * switched (non-standard) representation, that is setting the second
   * component to 0 and the first component to @p x.  This makes it possible
   * to search in a set of DLP for elements containing this value as its
   * smaller component.
   */
  inline void set_for_search(const unsigned int x)
   { 
     // we insert it reverse sorted (to be able to search for x)
     p1=x; p2=0;
   }

  /// swaps the components @p p1 and @p p2 of the DLP
  inline void swap(void) { unsigned int h=p1; p1=p2; p2=h; }

  /// return the DLP as a multiple precision integer @p x
  inline void assign_to_mpz(mpz_t x) const
   {
     if (p1==0) { mpz_set_ui(x,p2); return; }
     if (p2==0) { mpz_set_ui(x,p1); return; }
     mpz_set_ui(x,p1); mpz_mul_ui(x,x,p2);
   }

  /*!
   *
   * returns, whether the object is empty,
   * or if it contains a static prime, a single large prime or
   * a double large prime.
   */
  inline Factortype Type() const
   {
     if (p1>p2)
      { cerr << "CmpqsFactor: unordered factors!" << endl; }
     if (p1==0 && p2==0) return empty;
     if (p1==0 && p2==1) return static_prime;
     if (p1==0) return single_large_prime;
     return double_large_prime;
   }

  inline const bool IsTypeOf(const Factortype givenType) const
   {
     return Type()==givenType;
   }

  /// returns, whether the object is divisible by @p x
  inline bool DLP_divisible_by(const unsigned int x) const
   { return p1==x || p2==x; }

  inline bool operator< (const CmpqsFactor &x) const
   {
     if (p1==x.p1) return p2<x.p2;
     else return p1<x.p1;
   }

  inline const bool operator== (const CmpqsFactor &x) const
   {
     return (p1==x.p1 && p2==x.p2); // componentwise identity
   }

  inline const bool operator!= (const CmpqsFactor &x) const
   {
     return !((*this)==x);
   }

  /// returns the quotient of this object and @p u
  inline const int operator/ (const unsigned int u) const
   {
     // return p1*p2/u, in other words: just the other factor...
     if (p1==u) return p2;
     if (p2==u) return p1;
     cerr << "CmpqsFactor: invalid division!" << endl;
     exit(1);
     return 0;
   }

  /// output operator for DLP
  friend ostream& operator<< (ostream &ostr, const CmpqsFactor &x);

  /// input operator for DLP
  friend istream& operator>> (istream &istr, CmpqsFactor &x);
};


inline ostream& operator<< (ostream &ostr, const CmpqsFactor &x)
{
   if (x.p1) ostr << x.p1 << "*" << x.p2;
   else ostr << x.p2;
   return ostr;
}

#endif
