/*! @file
 * @brief
 * implementation of class StaticRelations
 */

#include "StaticRelations.H"

CRelation* StaticRelations::GLS[StaticFactorbaseSettings::MaxSize] = {NULL};
int StaticRelations::Filling_GLS = 0;

filebuf StaticRelations::FileBuffer;
ostream StaticRelations::StaticRelations_to_file(&StaticRelations::FileBuffer);
istream StaticRelations::StaticRelations_from_file(&StaticRelations::FileBuffer);


void StaticRelations::Load()
{
  // reading static relations from file
#ifdef VERBOSE_NOTICE
  cout << "Reading static relations from file..." << endl;
#endif
  StaticRelations_from_file.seekg(0,ios::beg);
  while (StaticRelations_from_file.peek()=='G' || StaticRelations_from_file.peek()=='U')
   {
     if (StaticRelations_from_file.peek()=='U')
      {
#ifdef VERBOSE_WARN
        cerr << "skipping obsolete/invalid relation." << endl;
#endif
        StaticRelations_from_file.ignore(1000000,'\n');
        continue;
      }
     CRelation* GL = new CRelation();
     GL->combine(StaticRelations_from_file);
     GL->optisize(); // Relation optimieren (dense<->sparse, etc.)
     GLS[GL->relevant_factor]=GL; ++Filling_GLS; // insert relation into linear system of equations
     StaticRelations_from_file.ignore(1,'\n');
   }
  if (StaticRelations_from_file.peek()!=EOF)
   {
     cerr << "Static Relations: garbage detected. Giving up. Please validate." << endl;
     exit(1);
   }
  StaticRelations_from_file.clear();
#ifdef VERBOSE_NOTICE
  cout << "Recover: " << StaticRelations::Count() << " static relations read." << endl;
#endif
}

void StaticRelations::insert(CRelation *GL, const bool do_multi_combine_init /* =true */)
{
  /* Any new relation is inserted into the linear system of equations.
     If the relation is redundant or if it leads to a factorization,
     this will be detected automatically. */

#ifdef IS_CLIENT
  // the Client does not manage any system of equations (which is instead handled by the server!)
  // so: delete the relation, do stats and return
  if (GL->relevant_factor >= 0) ++Filling_GLS;
  delete GL;
  return;
#else  
  if (GL->relevant_factor==CRelation::dynamic_factor || GL->relevant_factor==CRelation::special_factor)
    { delete GL; return; } // save it to background storage and delete it in main memory

#ifdef IS_SERVER
  // if this function is called by multiple threads, it is a critical section!
  // therefore we have to protect it with a mutex...
  static CMutex my_mutex;
  CUnlockMutexAtDestruction Watcher(my_mutex); // let the destructor of Watcher unlock the mutex!
  my_mutex.lock();
#endif
  
  if (do_multi_combine_init)
   {
     CRelation::SMulticombineData *pMD=new(CRelation::SMulticombineData);
     GL->set_MulticombineData(pMD);
     GL->multi_combine_init();
   }
   
  while (!GL->empty())
    {
      if (GL->relevant_factor>=StaticFactorbase::Size())
       {
         MARK;
         cerr << "value too big!" << endl;
         exit(1);
       }
      CRelation *pos = GLS[GL->relevant_factor];
      if (pos!=NULL)
	{
          // an entry for this relation already exists -> we can combine it!!
#if 1 /* toggle minimization of relations (1=on/0=off) */
          if (
	       (GL->Relation_sparse) && (pos->Relation_sparse) // both are sparse
               && GL->SizeOfRelation()<=pos->SizeOfRelation()
	       && GL->second_largest_factor_in_Relation()<=pos->second_largest_factor_in_Relation()
             )
	    {
              /* weniger Primzahlen in der Relation bedeuten schnelleres Abarbeiten.
	  	 Da beide Relationen quivalent sind, knnen sie problemlos so
		 getauscht werden, dass die platzsparende im Gleichungssystem verbleibt.
		 Andererseits bestimmt das Maximum der zweitgrten Faktoren zweier
		 Relationen die nchste Verknpfung. Also ist auch dies zu minimieren.
                 Beim Zielkonflikt wren beide Kriterien also abzuwgen. Dies geschieht
		 hier nicht. Nur wenn beide Kriterien zugleich die Lage nicht
                 verschlechtern, werden die Relationen getauscht.

                 Please note, that swapping the two relations does not change the the result of
                 the combine operation. But it may help to speed-up later combinations and/or
                 and/or reduces memory consumption.
              */
	      GL->multi_combine_exit(); 
	      GLS[GL->relevant_factor]=GL;
	      std::swap(pos,GL); // swap relations
              GL->use_MulticombineData_from(*pos);
              pos->invalidate_MulticombineData();
              GL->multi_combine_init();
	    }
#endif
          // biggest factor has to be eliminated
 	  GL->multi_combine_main(*pos); 
	}
      else
	{
	  GL->multi_combine_exit();
          if (do_multi_combine_init) GL->dispose_MulticombineData();
          else GL->invalidate_MulticombineData();
	  GLS[GL->relevant_factor]=GL; // insert relation into linear system of equations
	  Filling_GLS++; // one more relation in the system of linear equations
          GL->save(StaticRelations_to_file);
          StatusReport();
	  return;
	}
    }

  GL->multi_combine_exit();
  if (do_multi_combine_init) GL->dispose_MulticombineData();
  else GL->invalidate_MulticombineData();

  // the relation is redundant:
  // -> we cannot insert it into the static factorbase,
  // -> it can yield a factor instead!
  // -> compute quadratic congruency!
  const bool complete = GL->ComputeQuadraticCongruence();
  delete GL;
  if (complete)
   {
     // Game over, we have won!
     // leave the program...
     ExitManager::StopFactorization(); // try to stop the program successfully...
     exit(0); // this line should not be reached!
   }
#endif
}
