/*! @file
 * @brief
 * implementation of class SpecialRelations; handling double large prime stuff
 */


#include "SpecialRelations.H"

SpecialRelations::TSpecialFactorRelations SpecialRelations::SpecialFactorRelations;
filebuf SpecialRelations::FileBuffer;
ostream SpecialRelations::SpecialRelations_to_file(&SpecialRelations::FileBuffer);
istream SpecialRelations::SpecialRelations_from_file(&SpecialRelations::FileBuffer);


void SpecialRelations::Load()
{
  // read in Special Relations
  // compactify special relations on file
  // (that is: remove relations which are marked redundant)
  cout << "Reading special relations and reorganizing file..." << endl;
  unsigned int u_hits = 0, g_hits = 0;
  streampos rpos=0, wpos=0;
  // Special Relations einlesen
  SpecialRelations_from_file.seekg(0,ios::beg);
  SpecialRelations_to_file.seekp(0,ios::beg);

  while (SpecialRelations_from_file.peek()=='G')
   {
     TSpecialFactorRelation relation;
     relation.fpos = SpecialRelations_from_file.tellg();
     string s; SpecialRelations_from_file >> s; // read "G "
     SpecialRelations_from_file >> relation.factor; // read special relation factor
     SpecialFactorRelations.insert(relation);
     relation.factor.swap();
     SpecialFactorRelations.insert(relation);
     relation.factor.swap();
     SpecialRelations_from_file.ignore(1000000,'\n'); // read over invalid/redundant lines
     ++g_hits;
    }
  wpos=rpos=SpecialRelations_from_file.tellg(); // write position = read position

  // quick sanity check:
  if (SpecialRelations_from_file.peek()!='U' && SpecialRelations_from_file.peek()!=EOF)
   {
     MARK;
     cerr << "Special Relations: garbage detected. Giving up. Please validate." << endl;
     exit(1);
   }

  while (true) 
    {
      SpecialRelations_from_file.seekg(rpos);
      while (SpecialRelations_from_file.peek()=='U')
       {
         SpecialRelations_from_file.ignore(1000000,'\n'); // read over invalid/redundant lines
         ++u_hits;
       }

      int cached_g = 0;
      const int max_cached_g = 1000;
      string sG[max_cached_g];
      while (SpecialRelations_from_file.peek()=='G')
       {
         while (SpecialRelations_from_file.peek()!='\n')
          {
            char buffer[2048];
            SpecialRelations_from_file.get(buffer, sizeof(buffer), '\n');
            sG[cached_g]+=buffer;
          }
         SpecialRelations_from_file.ignore(10,'\n');
         if (++cached_g>=max_cached_g) break;
         while (SpecialRelations_from_file.peek()=='U')
          {
            SpecialRelations_from_file.ignore(1000000,'\n'); // read over invalid/redundant lines
            ++u_hits;
          }
       }
      rpos=SpecialRelations_from_file.tellg();
      if (cached_g==0) break;
      SpecialRelations_to_file.seekp(wpos);

      for (int i=0; i<cached_g; ++i)
       {
         TSpecialFactorRelation relation;
         relation.fpos = SpecialRelations_to_file.tellp();
         {
           istringstream is(sG[i]);
           string s; is >> s;
           is >> relation.factor;
         }
         SpecialRelations_to_file << sG[i] << "\n";
         SpecialFactorRelations.insert(relation);
         relation.factor.swap();
         SpecialFactorRelations.insert(relation);
         relation.factor.swap();
       };
      SpecialRelations_to_file.flush();
      wpos=SpecialRelations_to_file.tellp();
      g_hits+=cached_g;
      cout << g_hits << " special relations written, "
           << u_hits << " removed." << "\r" << flush;
      //cout << (long long int)static_cast<streamoff>(rpos) << " , " << (long long int)static_cast<streamoff>(wpos) << endl;
    }
   cout << endl;
   
  // in the absence of a truncate-operation we fill up the
  // rest of the file with U's (u=ungltig=invalid)
  {
    streampos pos, ende;
    pos = wpos;
    SpecialRelations_to_file.seekp(0, ios::end);
    ende = SpecialRelations_to_file.tellp();
    SpecialRelations_to_file.seekp(wpos);
    //cout << (long long int)static_cast<streamoff>(pos) << " , " << (long long int)static_cast<streamoff>(ende) << endl;

    char buffer[4096];
    for (unsigned int i=0; i<sizeof(buffer); ++i) buffer[i]='U';
    while (static_cast<streamoff>(SpecialRelations_to_file.tellp()) < static_cast<streamoff>(ende))
      {
        SpecialRelations_to_file.write(buffer,sizeof(buffer));
        //cout << (long long int)static_cast<streamoff>(SpecialRelations_to_file.tellp()) << " , " << (long long int)static_cast<streamoff>(ende) << endl;
      }
    SpecialRelations_to_file << flush;
    SpecialRelations_to_file.seekp(pos);
  }
  SpecialRelations_from_file.clear();
  cout << "Recover: " << SpecialFactorRelations.size()
       << " special relations read, " << endl
       << u_hits << " invalid DLP-relations eliminated." << endl;
}


const int CycleSearchPeriod = 500000;
static int next_CycleSearch = CycleSearchPeriod;

void SpecialRelations::CycleSearch()
{
  // If we are able to find/construct a subset of Double-Large-Primes that
  // represents a cycle (p1,p2),(p2,p3),(p3,p4),...(pk,p1), then this
  // cycle neutralizes the Large Primes inside this subset, because
  // every Large Prime occurs twice (or as multiple of 2) inside the cycle.
  // -> we gain a static relation for each cycle.

  if (SpecialFactorRelations.empty()) return; // there is nothing to find
  cout << "Starting cycle find for Special-Factor-Set" << endl;

  TSpecialFactorRelations WorkingSet;
  cout << "Size: " << SpecialFactorRelations.size() << ", " << WorkingSet.size() << endl;

  WorkingSet.swap(SpecialFactorRelations);
  // Hint: to reduce memory usage, we move the Special-Factor-Relations
  // to the WorkingSet instead of copying them.
  // Therefore we must pay attention to move them back after cycle-find is done!
  cout << "Size: " << SpecialFactorRelations.size() << ", " << WorkingSet.size() << endl;

  intset NodeSet;
  typedef list<TSpecialFactorRelation> TEdgeList;
  TEdgeList EdgeList;

  TSpecialFactorRelations::iterator pos;
  TSpecialFactorRelation SearchRelation;
  int right_side,left_side;

  while (!WorkingSet.empty())
   {
     NodeSet.clear(); // important! (due to repeated iteration)
     pos=WorkingSet.begin();
     TSpecialFactorRelation relation = *pos;
     EdgeList.push_back(relation);
     left_side=relation.factor.LP1();
     right_side=relation.factor.LP2();
     NodeSet.insert(left_side);
     NodeSet.insert(right_side);
     WorkingSet.erase(pos); SpecialFactorRelations.insert(relation);
     relation.factor.swap();
     WorkingSet.erase(relation); // remove old Specialfactor (swapped copy)
     SpecialFactorRelations.insert(relation);
     relation.factor.swap();

     //cout << "Starting new round of cycle search." << endl;
     while (!EdgeList.empty())
      {
        //for (TEdgeList::iterator p=EdgeList.begin(); p!=EdgeList.end(); ++p)
        // cout << (*p).factor << "|";
        //cout << endl;
        //cout << "Searching for " << left_side << endl;
	//cout << "WorkingSet contains: " << endl;
        //for (pos=WorkingSet.begin(); pos!=WorkingSet.end(); ++pos)
        // cout << (*pos).factor << ",";
	//cout << endl;

        SearchRelation.factor.set_for_search(left_side);
        pos=WorkingSet.lower_bound(SearchRelation);
        if (pos!=WorkingSet.end() && (*pos).factor.DLP_divisible_by(left_side) )
         {
           // a node can be inserted at the left side
           relation = *pos;
           WorkingSet.erase(pos); // remove old Specialfactor
           SpecialFactorRelations.insert(relation);
           relation.factor.swap();
           WorkingSet.erase(relation); // remove old Specialfactor (swapped copy)
           SpecialFactorRelations.insert(relation);
           relation.factor.swap();
           left_side=relation.factor/left_side;
           if (NodeSet.find(left_side)!=NodeSet.end())
            { 
              // found a cycle!
	      statistical_data::DLP_cycle_hits++;
	      cout << "DLP-cycle found... " << endl;
              // now: 
              // evaluate cycle, construct the relation and insert it
              // into the system of equations
              CRelation *GL = new CRelation;
              GL->combine(SpecialRelations_from_file,relation.fpos);
              mpz_mul_ui(GL->Delta,GL->Delta,left_side);
	      mpz_mod(GL->Delta,GL->Delta,n);
              right_side=relation.factor/left_side;
	      //cout << relation.factor;

              // Walk through the EdgeList up to the position, where the
              // value of "left_side" occurs firstly. 
              //  -> cycle ends at this position.
	      // Don't forget to multiply the root of the product of the specialfactors!
              // As each part of a specialfactor occurs twice, we can use
              // the "right_side" values.
	      unsigned int CycleLength=1;
              for (TEdgeList::iterator p=EdgeList.begin(); p!=EdgeList.end(); ++p)
	       {
                 CycleLength++;
		 //cout << " - " << p->factor;
                 mpz_mul_ui(GL->Delta,GL->Delta,right_side);
	         mpz_mod(GL->Delta,GL->Delta,n);
                 GL->combine(SpecialRelations_from_file,(*p).fpos);
                 if ((*p).factor.DLP_divisible_by(left_side)) break;
		 right_side=(*p).factor/right_side;
               }
              //cout << endl;
              cout << "DLP-cycle size: " << CycleLength << endl;
	      GL->SanityCheck(); // as a precaution...
     	      StaticRelations::insert(GL);

              // The length of the cycle is (roughly) the number of large primes inside the cycle.
              statistical_data::DynamicFactorRating.increment_Hits_at_position(CycleLength);
	    
  	      // remove an edge (which should represent this cycle) out of the Special-Relation-Set,
              // we choose the last (=current) edge):
              SpecialFactorRelations.erase(relation); // remove old special factor
              relation.factor.swap();
              SpecialFactorRelations.erase(relation); // and the swapped copy, too!
              relation.factor.swap();
              {
	        // mark this relation as redundant/invalid on file!
                const streampos pos = SpecialRelations_to_file.tellp(); // memorize write position
                SpecialRelations_to_file.seekp(relation.fpos); SpecialRelations_to_file << "U " << flush; // mark as invalid
                SpecialRelations_to_file.seekp(pos); // restore fileposition
              }
 
              // important: continue cycle search, there may be more than just one!
	      left_side=relation.factor/left_side;
              continue;
            }
           else
            {
              // no cycle found -> insert edge and node (and retry)
              NodeSet.insert(left_side);
              EdgeList.push_front(relation);
	      //cout << "Insert node " << relation.factor << endl;
              continue;
            } 
         }
        // no edge found which can be appended at the left side.
        // -> alternative 1: insert right-wise (not necessary for cyclefind)
        // -> alternative 2: remove the current left-sided edge (as this edge gains no further connections)
	//cout << "removing edge " << EdgeList.begin()->factor << endl;
        left_side=EdgeList.begin()->factor/left_side;
        EdgeList.pop_front(); // remove left-sided edge
      }
   }
  cout << "Size: " << SpecialFactorRelations.size() << ", " << WorkingSet.size() << endl;
  cout << "cycle find finished." << endl;
}

bool SpecialRelations::insert(const CmpqsFactor &DLP, CRelation *GL, const short int HitCount /* =0 */)
{
  // insert Special-Factor into Special-Factor-Set. (This is done for both
  // factors of the DLP for fast access.) If the Special-Factor is already
  // present, then combine it to get a static relation.

  if (DLP.LP1()==DLP.LP2()) // special case: DoubleLargePrime=LargePrime^2
   {
     cout << "special hit: DLP is a square!" << endl;
     statistical_data::Special_hit++;
     // please note: Delta is not the square, but it is the part of the
     // relation which is to square. Therefore never forget to multiply
     // roots into Delta!
     mpz_mul_ui(GL->Delta,GL->Delta,DLP.LP1()); mpz_mod(GL->Delta,GL->Delta,n);
     return false;
   }
  
  TSpecialFactorRelation relation;
  relation.factor=DLP;

  const TSpecialFactorRelations::iterator pos = SpecialFactorRelations.find(relation);
  if (pos==SpecialFactorRelations.end())
   {
     relation.fpos=GL->save(SpecialRelations_to_file,relation.factor,HitCount);
     SpecialFactorRelations.insert(relation);
     relation.factor.swap();
     SpecialFactorRelations.insert(relation);

     // trigger cyclesearch on DLP at regular intervals
     if (!--next_CycleSearch)
      {
        next_CycleSearch=CycleSearchPeriod;
        SpecialRelations::CycleSearch();
      }
     return true;
   }
  else
   { 
     // Special-Factor already known
     cout << "Special Hit by already known Special Factor!" << endl;
     statistical_data::Special_hit++;
     mpz_t x;
     mpz_init(x); DLP.assign_to_mpz(x);
     mpz_mul(GL->Delta,GL->Delta,x); mpz_mod(GL->Delta,GL->Delta,n);
     mpz_clear(x);
     GL->combine(SpecialRelations_from_file,(*pos).fpos);
     //StaticRelations::insert(GL); // should be done elsewhere!
     return false;
   }
}


bool SpecialRelations::SpecialFactor_splitable(const CmpqsFactor &DLP)
{
  // check, whether Special-Factor can be split successfully
  // true -> yes, false -> no
  if (DLP.LP1()==DLP.LP2()) // special case: DoubleLargePrime=LargePrime^2
   {
     // cout << "Special case: DLP is a square!" << endl;
     return true;
   }
  TSpecialFactorRelation relation;
  relation.factor=DLP;
  TSpecialFactorRelations::iterator pos = SpecialFactorRelations.find(relation);
  return (pos!=SpecialFactorRelations.end());
}

void SpecialRelations::split_by_primefactor(const int Primfaktor)
{
  // Specialfactors are always outside the static factorbase. No sieving is
  // done for special factors.  Special factors are always composite factors
  // and (normally) consist of exactly two single large primes.  If one
  // SLP-part of a special factor is already inside the dynamic factorbase,
  // then we can split the special factor and we get a new SLP and possibly
  // a new dynamic factor, too.  Maybe we can gain even a hit inside the
  // static factorbase: This happens, when both parts of the special factor
  // are known SLPs. Then we can combine all the parts of the special
  // relation in conjunction with these single large primes to get a static
  // relation.

  // Whenever a new SLP gets detected, we should also check, whether this
  // SLP is part of any Specialfactor.  In the latter case, these
  // Specialfactors can be split as well. (And we should recurse the
  // procedure to get possibly even more of them!) This way we get new
  // dynamic factors to sieve with!

  // Any split primefactor (outside the static factorbase) will be
  // appended to the new relation as a "correction factor". The relations
  // that correspond with these "correction factors" need to be combined
  // later to get a valid static relation.  This is a kind of lazy
  // evaluation, because the evaluation of the relation will be delayed (or
  // even avoided) until it is known that the relation can be combined
  // (recursively) to gain a static relation!

  static int Trashcounter = 0; // number of superfluous relations (that contain already eliminated/combined factors)
  
  TSpecialFactorRelation SearchRelation;
  SearchRelation.factor.set_for_search(Primfaktor);

  while (true)
   {
    //cout << "splitting " << Primfaktor << endl;
    /* Because of recursion we need to restart here every time when a specialfactor got
       removed inside the loop.  -> get the next biggest Special-Factor */
    const TSpecialFactorRelations::iterator pos=SpecialFactorRelations.lower_bound(SearchRelation);
    if (pos==SpecialFactorRelations.end()) break;
    if (!(*pos).factor.DLP_divisible_by(Primfaktor)) break;

    TSpecialFactorRelation memorized_Relation = *pos;
    SpecialFactorRelations.erase(pos); // remove old specialfactor
    memorized_Relation.factor.swap();
    SpecialFactorRelations.erase(memorized_Relation); // and remove also its swapped version
    memorized_Relation.factor.swap(); // swap back to original
    
    TDynamicFactorRelation relation;
    relation.factor=Primfaktor;

    if (!is_dynamic_factor(relation))
      {
        MARK; 
        cerr << "Inconsistency in dynamic factorbase!" << endl;
        exit(1);
      }
    const TDynamicFactorRelation el=relation;

    relation.factor=memorized_Relation.factor/Primfaktor; // this is the new remaining factor
    // now we have a correctly computed dynamic factor at hand

    // but is it really a new one, or did it exist already the dynamic factorbase?
    if (!is_dynamic_factor(relation))
     { 
       // it is new!
       //cout << "Special Factor: Dynamic Factor detected!" << endl;
       statistical_data::Special_to_dynamic_Factor_hit++;
       relation.fpos = DynamicRelations_to_file.tellp();
       // transform the relation and store it:
       {
         const streampos rpos=SpecialRelations_from_file.tellg(); // memorize position
         SpecialRelations_from_file.seekg(memorized_Relation.fpos); // memorize position

         CStreamDecoder enc_in(SpecialRelations_from_file);
         CStreamEncoder enc_out(DynamicRelations_to_file);

         CmpqsFactor ret; string s;
         // memorize formatflags of file
         const ios::fmtflags oldFlagsR = SpecialRelations_from_file.flags(); // memorize old formatflags
         SpecialRelations_from_file.unsetf(ios::hex); SpecialRelations_from_file.setf(ios::dec); // assure decimal notation (note: recursive call can involve SLP correction factors!)
         SpecialRelations_from_file.unsetf(ios::showbase); // don't show base (eg. don't prepend "0x" for hexadecimal numbers)
         const ios::fmtflags oldFlagsW = DynamicRelations_to_file.flags(); // save old format flags
         DynamicRelations_to_file.unsetf(ios::hex); DynamicRelations_to_file.setf(ios::dec); // assure decimal notation (note: recursive call can involve SLP correction factors!)
         DynamicRelations_to_file.unsetf(ios::showbase); // don't show base
 
         SpecialRelations_from_file >> s; // relation start symbol 'G'
         if (s != "G")
          {
	   MARK;
           cerr << "error: wrong position while reading relation" << endl;
           exit(1);
          }
         SpecialRelations_from_file >> ret; // read in factor (dynamic-factor or special-factor or 1 (for static factor))
         SpecialRelations_from_file >> s; // delta
         DynamicRelations_to_file << "G " << setprecision(20) << relation.factor << " " << s << " ";

         // now we use hexadecimal notation:
         SpecialRelations_from_file.unsetf(ios::dec); SpecialRelations_from_file.setf(ios::hex); // hexadecimal notation
         SpecialRelations_from_file.unsetf(ios::showbase); // don't show base
         DynamicRelations_to_file.unsetf(ios::dec); DynamicRelations_to_file.setf(ios::hex); // hexadecimal notation
         DynamicRelations_to_file.unsetf(ios::showbase); // don't show base

         int distance=enc_in.GetValue();
         while (distance>=0)
          {
            enc_out.PutValue(distance);
            distance=enc_in.GetValue();
          }
         enc_out.PutValue(-2);
         enc_out.PutValue(Primfaktor);
         if (distance==-2)
          {
            distance=enc_in.GetValue();
            while (distance>=0)
             {
               enc_out.PutValue(distance);
               distance=enc_in.GetValue();
             }            
          }
         enc_out.PutValue(-1); DynamicRelations_to_file << endl;
         SpecialRelations_from_file.flags(oldFlagsR); // restore old Formatflags
         DynamicRelations_to_file.flags(oldFlagsW); // restore old Formatflags
         SpecialRelations_from_file.seekg(rpos); // restore position
       }
       relation.append_for_sieving();

       // not really important: for the standalone version we would also be
       // able to insert the factor immediately for sieving here.

#ifdef SAFEMODE
       // validate the newly generated dynamic factor relation.
       // also the new dynamic factor needs to be feed to the validation routine!
       {
        const streampos rpos=DynamicRelations_from_file.tellg();
        DynamicRelations_from_file.seekg(relation.fpos);
        CRelation::SanityCheck(DynamicRelations_from_file); // exit if relation is invalid (->this would be an internal error!)
        DynamicRelations_from_file.seekg(rpos);
       }
#endif

       DynamicFactorRelations.insert(relation); // insert Dynamicfactor
       SpecialRelations::split_by_primefactor(relation.factor);
     }
    else 
     { 
       // does it already exist in the dynamic FB? -> a new relation!
       //cout << "Special Factor: Special Hit!" << endl;
       CRelation *GL = new CRelation;
       GL->combine(SpecialRelations_from_file,memorized_Relation.fpos);
       // combine the dynamic factors which were found:
       mpz_mul_ui(GL->Delta,GL->Delta,relation.factor); mpz_mod(GL->Delta,GL->Delta,n);
       GL->combine(DynamicRelations_from_file,relation.fpos);
       // don't forget to eliminate the primefactors as well!
       mpz_mul_ui(GL->Delta,GL->Delta,el.factor); mpz_mod(GL->Delta,GL->Delta,n);
       GL->combine(DynamicRelations_from_file,el.fpos);
       statistical_data::Special_hit++; StaticRelations::insert(GL);
       statistical_data::DynamicFactorRating.increment_Hits_at_position(2); // two SLP are involved
     } 

    { 
      // remove old (and superfluous) special-relation
      const streampos pos = SpecialRelations_to_file.tellp(); // save position
      SpecialRelations_to_file.seekp(memorized_Relation.fpos); SpecialRelations_to_file << "U "; // mark as invalid
      SpecialRelations_to_file.seekp(pos); // restore position
    }

    ++Trashcounter; // count the redundant relations of this file
   }

 // If too much trash has been collected, we can do a "garbage collection"
 // or issue a message to restart the program to trigger a compactification. 
 // But today's harddisks are large enough, so this doesn't matter...
}


bool SpecialRelations::insert(const CmpqsFactor &DLP, const string &GL_String)
{
  // Insert Special-Factor into Special-Factor-Set. Do this for both parts
  // of the DLP to speed up access!
  // If the Special-Factor is already known, combine it and gain a new relation.

  if (DLP.LP1()==DLP.LP2()) // special case: DoubleLargePrime=LargePrime^2
   {
     cout << "special case: DLP is a square! (in LAZY)" << endl;
     istringstream is(GL_String);
     CRelation* GL = new CRelation();
     CRelation::SMulticombineData *pMD = new CRelation::SMulticombineData;
     GL->set_MulticombineData(pMD);
     GL->multi_combine_init();
     CmpqsFactor DLP_2 = GL->multi_combine_main(is);
     if (DLP!=DLP_2) { MARK; cerr << "DLP-Error somewhere in LAZY!" << endl; exit(1); }
     statistical_data::Special_hit++;
     // caution: Delta isn't the square, but it is to square!
     // therefore multiply only square roots!
     mpz_mul_ui(GL->Delta,GL->Delta,DLP.LP1()); mpz_mod(GL->Delta,GL->Delta,n);
     StaticRelations::insert(GL,without_multi_combine_init); // LAZY -> insert take place here
     delete pMD;
     return false;
   }
  
  TSpecialFactorRelation relation;
  relation.factor=DLP;

  const TSpecialFactorRelations::iterator pos = SpecialFactorRelations.find(relation);
  if (pos==SpecialFactorRelations.end())
   {
     //cout << "Speichere DLP-Relation (LAZY)" << endl;
     //relation.fpos=GL->save(SpecialRelations_to_file,relation.factor);
     relation.fpos=SpecialRelations_to_file.tellp();
     SpecialRelations_to_file << GL_String;

     SpecialFactorRelations.insert(relation);
     relation.factor.swap();
     SpecialFactorRelations.insert(relation);

     // trigger cyclesearch on DLP at regular intervals
     if (!--next_CycleSearch)
      {
        next_CycleSearch=CycleSearchPeriod;
        SpecialRelations::CycleSearch();
      }
     return true;
   }
  else
   { 
     // Special-Factor already known
     cout << "Special Hit by known Special Factor! (LAZY)" << endl;
     istringstream is(GL_String);
     CRelation* GL = new CRelation();
     CRelation::SMulticombineData *pMD = new CRelation::SMulticombineData;
     GL->set_MulticombineData(pMD);
     GL->multi_combine_init();
     CmpqsFactor DLP_2 = GL->multi_combine_main(is);
     if (DLP!=DLP_2) { MARK; cerr << "DLP-Error somewhere in LAZY!" << endl; exit(1); }
     statistical_data::Special_hit++;
     mpz_t x;
     mpz_init(x); DLP.assign_to_mpz(x);
     mpz_mul(GL->Delta,GL->Delta,x); mpz_mod(GL->Delta,GL->Delta,n);
     mpz_clear(x);
     GL->multi_combine_main(SpecialRelations_from_file,(*pos).fpos);
     //GL->multi_combine_exit() may be omitted, if StaticRelations::insert does no multi_combine_init
     StaticRelations::insert(GL,without_multi_combine_init); // LAZY -> inserting takes place here...
     delete pMD;
     return false;
   }
}
