#ifndef MY_SEMAPHORE_
#define MY_SEMAPHORE_

/*! @file
 * @brief
 * contains wrapper class for semaphore handling
 * in a C++ multithreaded environment
 */


#ifdef CYGWIN_COMPAT
 // strange: using "" instead of <> for cygwin
 #include "/usr/include/semaphore.h"
#else
extern "C"
{
  #include <semaphore.h>
}
#endif

//! This class provides basic exception handling
class CSemaphore_exception : public std::exception
{
 private:
  const std::string Message;

 public:
  explicit CSemaphore_exception(const std::string& Msg) throw()
   : Message(Msg)
   {
#if defined(VERBOSE) || defined(DEBUG)
     cerr << "throwing exception: " << Msg << endl;
#endif
   }
 
  virtual ~CSemaphore_exception() throw() { }
  
  virtual const char* what() const throw()
   {
     return Message.c_str();
   }  
};


//! wrapper class for semaphores
class CSemaphore
{
 private:
  sem_t resource_counter;
 public:
  CSemaphore(const unsigned int my_resource_counter)
   {
     if (my_resource_counter>static_cast<long unsigned int>(SEM_VALUE_MAX))
      throw CSemaphore_exception("CSemaphore::constructor: my_resource_counter>SEM_VALUE_MAX");
     if (sem_init(&resource_counter, 0, my_resource_counter))
      throw CSemaphore_exception("CSemaphore::constructor: initialization failed!");
   }
  ~CSemaphore()
   {
     if (sem_destroy(&resource_counter))
      throw CSemaphore_exception("CSemaphore::destructor: failed!");
   }
  void post()
   {
     if (sem_post(&resource_counter)) throw CSemaphore_exception("CSemaphore_post");
   }
  void wait()
   {
     if (sem_wait(&resource_counter)) throw CSemaphore_exception("CSemaphore_wait");
   }
  const bool trywait()
   {
     int retval=(sem_trywait(&resource_counter));
     if (retval==-1 && errno!=EAGAIN) throw CSemaphore_exception("CSemaphore_trywait");
     return (retval==0);
   }
  const int getvalue()
   {
     int retval;
     if (sem_getvalue(&resource_counter,&retval)) throw CSemaphore_exception("CSemaphore_getvalue");
     return retval;
   }
};

#endif /* MY_SEMAPHORE_ */
