#ifndef PERSISTENT_D_H
#define PERSISTENT_D_H

// written by Thorsten Reinecke (reinecke@thorstenreinecke.de)
// last change: 2004-10-06

/*! @file
 * @brief
 * set of classes to provide persistent storage of variables
 */


extern "C"
{
 #include <unistd.h>
}

#include <map>
#include <string>
#include <iostream>
#include <fstream>

//! abstract interface class
class CPersistentDataInterface
{
 public:
  virtual void Load(std::istream &is) = 0;
  virtual void Save(std::ostream &os) const = 0;
  virtual ~CPersistentDataInterface() = 0;
};
CPersistentDataInterface::~CPersistentDataInterface() { }

// gcc-4.0 (experimental) issued warnings about missing virtual destructors.
// Although we don't need destructors for our implementation, we *do* define
// a virtual destructor in the interface. That's because if the interface
// doesn't declare a virtual destructor (and also defines a default
// destructor), then no destructor will be called for the dynamically
// constructed items inside CPersistentDataCollection!


//! derived "typed" helper classes to load and store variables
template <typename T>
class CPersistentData : protected CPersistentDataInterface
{
 protected:
  T &data; // reference of the variable to load/save
  CPersistentData(T &mydata) : data(mydata) { }
  virtual ~CPersistentData() { }
  virtual void Load(std::istream &is)
   {
     is >> data;
#ifdef VERBOSE
     cout << data << endl;
#endif
   }
  virtual void Save(std::ostream &os) const { os << data << std::endl; }
  friend class CPersistentDataCollection;
};


/*!
 * @brief
 * provides a collection of references to load/save variables
 *
 * This class stores references to variables denoted by an identifier.
 * On Load/Save requests all these registered variables are read/written
 * to a file (persistent storage).
 *
 * Warning: Use this class with extreme caution!
 *  - Don't forget to unregister (local) variables before they're dead!
 *  - The compiler may optimize the access to variables; therefore
 *    access to the dereferenced data may remain undetected by the compiler;
 *    it is possible, that cached values in registers are used instead, if
 *    the compiler does very aggressive optimization!
 *  - Don't use any methods provided by this class within a different thread!
 *    Any access is not threadsafe!
 */
class CPersistentDataCollection : private ForbidAssignment
{
 protected:
  typedef std::map<std::string,CPersistentDataInterface*> TCollection;
  const std::string filename;
  TCollection collection;

 public:

  //! delete the assigned file
  void ClearStream()
   {
     // remove stream
     unlink(filename.c_str());
   }

  //! unregister all variables so that the collection becomes empty
  void ClearCollection()
   {
    while (!collection.empty())
     {
       delete collection.begin()->second;
       collection.erase(collection.begin());
     }
   }

  //! constructor needs a filename to provide persistent storage
  explicit CPersistentDataCollection(const std::string myFilename)
   : filename(myFilename) { }
  
  ~CPersistentDataCollection() { ClearCollection();  }

  //! register a variable to be persistent
  template<typename T> inline void RegisterVar(const std::string ID, T &data)
   {
     if (collection.find(ID)!=collection.end()) // check for presence
      {
        // was already present!
        exit(1); // for debugging (we could throw an exception instead)
      }
     else collection[ID] = new CPersistentData<T>(data);
   }

  //! unregister a variable to be no more persistent
  template<typename T> inline void UnregisterVar(const std::string ID)
   {
     if (collection.find(ID)==collection.end()) // check for presence
      {
        // wasn't present!
        exit(1); // for debugging (we could throw an exception instead)
      }
     else
      {
        delete collection[ID];
        collection.erase(ID);
      }
   }

  //! load the persistent values for the registered variables from file
  void Load()
   {
     std::ifstream istr(filename.c_str());
     while (istr)
      {
        while (isspace(istr.peek())) istr.get();

        // it seems that gcc-3.2.3 has some problems detecting eof,
        // I hope this helps...
        istr.get();
        if (istr.eof()) break;
        istr.unget();

        std::string ID;
        istr >> ID;
        if (collection.find(ID)==collection.end()) // check for presence
         {
           MARK;
           cout << "ID " << ID << " not found!" << endl;
           exit(1); // isn't present!
         }
        else
         {
#ifdef VERBOSE
           cout << "Loading ID " << ID << "= "; 
#endif
           collection[ID]->Load(istr);
         }
      }
     istr.close();
   }

  //! save the values for the registered variables to file (to make them persistent)
  void Save()
   {
     std::ofstream ostr(filename.c_str(),std::ios::out|std::ios::trunc);
     for (TCollection::const_iterator p=collection.begin(); p!=collection.end(); ++p)
      {
        ostr << p->first << " ";
        p->second->Save(ostr);
      }
     ostr.close();
   }

};

#endif /* PERSISTENT_D_H */
