/*! @file
 * @brief
 * Implementation of a helper class for storing factors.
 */


void TFoundFactor::output(std::ostream &ostr) const
{
  ostr << x;
  if (e!=1) ostr << "^" << e;
  if (comment!="") ostr << " " << comment;
  else
   {
     if (!mpz_probab_prime_p(x,probab_prime_checks)) ostr << " [composite]";
   }
}

void TFoundFactors::PrettyPrint(std::ostream &os) const
{
  os << endl << "Factorization of " << regarding << ": " << endl;
  if (mpz_sizeinbase(regarding_numeric_value,10)<200)
   os << "[= " << regarding_numeric_value << "]" << endl;

  mpz_t x,y;
  mpz_init_set_ui(x,1); mpz_init(y);

  bool flag = false;
  for (TFoundFactors::const_iterator p=begin(); p!=end(); ++p)
   {
     if (flag) os << "* "; else os << "= ";
     p->output(os); os << endl;
     unsigned int exponent;
     p->get_factor(y,exponent); mpz_pow_ui(y,y,exponent);
     mpz_mul(x,x,y);
     flag=true;
   }

  // perform a sanity check
  if (mpz_cmp(x,regarding_numeric_value)) os << "The factorization is incomplete!" << endl;

  mpz_clear(x); mpz_clear(y);
}

void TFoundFactors::AutoSave() const
{
  std::ofstream ofstr(FoundFactorsFile.c_str());
  ofstr << regarding << endl;
  ofstr << regarding_numeric_value << endl;
  Save(ofstr);
}

void TFoundFactors::AutoLoad()
{
#ifdef VERBOSE_INFO
  cout << "Autoload FoundFactors" << endl;
#endif
  std::ifstream ifstr(FoundFactorsFile.c_str());
  char buf [32768];
  ifstr.getline(buf,sizeof(buf),'\n'); regarding=buf;
  ifstr >> regarding_numeric_value;
  Load(ifstr);
}


const char* const myMAL(void)
{
  // returns empty string on first call, and  " * " otherwise.
  // Task: Prevent costly checks for suppressing the trivial factor 1
  // as the fist resp. last factor.

#ifdef NOTIFY_PARENT
  // send SIGUSR2 to the parent to notify that a new
  // factor has found
  kill(getppid(),SIGUSR2); // send SIGUSR2 to the parent process
#endif

  static bool first_call = true;
  if (first_call)
   {
     first_call=false;
     return "";
   }
  else return " * ";
}

std::string MAL(const mpz_t factor, const unsigned int exponent=1)
{
  FoundFactors.insert(TFoundFactor(factor,exponent));
  FoundFactors.AutoSave();
  ostringstream helper;
  helper << myMAL() << factor;
  if (exponent>1) helper << "^" << exponent;
  return helper.str();
}

std::string MAL(const mpz_t factor, const unsigned int exponent, const std::ostringstream &comment)
{
  FoundFactors.insert(TFoundFactor(factor,exponent,comment.str()));
  FoundFactors.AutoSave();
  ostringstream helper;
  helper << myMAL() << factor;
  if (exponent>1) helper << "^" << exponent;
  return helper.str()+comment.str();
}


std::string MAL(const mpz_t factor, const std::ostringstream &comment)
{
  FoundFactors.insert(TFoundFactor(factor,1,comment.str()));
  FoundFactors.AutoSave();
  ostringstream helper;
  helper << myMAL() << factor;
  return helper.str()+comment.str();
}

std::string MAL(const long unsigned int factor, const unsigned int exponent=1)
{
  FoundFactors.insert(TFoundFactor(factor,exponent));
  FoundFactors.AutoSave();
  ostringstream helper;
  helper << myMAL() << factor;
  if (exponent>1) helper << "^" << exponent;
  return helper.str();
}
