/*! @file
 * @brief
 * Declaration of a helper class for storing factors.
 */


class TFoundFactor
{
 private:
  mpz_t x;
  unsigned int e;
  string comment;
 public:
  explicit TFoundFactor(const mpz_t f, const unsigned int exp = 1, const string &xcomment="")
   : e(exp), comment(xcomment) { mpz_init_set(x,f); }
  explicit TFoundFactor(const long unsigned int f, const unsigned int exp = 1)
   : e(exp), comment() { mpz_init_set_ui(x,f); }
  TFoundFactor(const TFoundFactor &FF)
   : e(FF.e), comment(FF.comment) { mpz_init_set(x,FF.x); }
  explicit TFoundFactor(std::istream &is)
   {
     mpz_init(x);
     is.get();
     if (is.eof()) { is.setstate(std::ios::failbit); return; }
     is.unget();
     string s;
     is >> x;
     is >> s; if (s!="e") { MARK; is.setstate(std::ios::failbit); return; }
     is >> e;
     is >> s; if (s!="c") { MARK; is.setstate(std::ios::failbit); return; }
     char buf[4096];
     is.getline(buf,sizeof(buf),'\n'); comment=buf;
   }
  virtual ~TFoundFactor() { mpz_clear(x); }
  virtual bool operator< (const TFoundFactor &nf) const { return (mpz_cmp(x,nf.x)<0); }
  void get_factor(mpz_t f, unsigned int &exponent) const { mpz_set(f,x); exponent=e; }
  virtual void output(std::ostream &ostr) const;
  TFoundFactor& operator= (const TFoundFactor &rhs)
   {
     mpz_set(x,rhs.x); e=rhs.e; comment=rhs.comment;
     return *this;
   }
  virtual void Save(std::ostream &os) const
   {
     os << x << " e " << e << " c " << comment << endl;
   }
};



inline std::ostream& operator << (std::ostream& ostr, const TFoundFactor FF)
{
  FF.output(ostr);
  return ostr;
}


// quick and dirty:
class TFoundFactors : public set<TFoundFactor>
{
private:
 // helper class to provide access to a mpz_t number without explictly calling mpz_init()/mpz_clear()
 class autompz_t
  {
   private:
    mpz_t x;
   public:
    autompz_t() { mpz_init(x); }
    ~autompz_t() { mpz_clear(x); }
    operator mpz_t&() { return x; }
    operator const mpz_t&() const { return x; }
  };
public:
 string regarding; // argument (given number, can be symbolic)
 autompz_t regarding_numeric_value;

 void Load(std::istream &is)
  {
    clear();
    while (is)
     {
       TFoundFactor FF(is);
       if (is.good()) insert(FF);
     }
  }
 void Save(std::ostream &os) const
  {
    for (TFoundFactors::const_iterator p=begin(); p!=end(); ++p) p->Save(os);
  }

 void PrettyPrint(std::ostream &os) const;
 void AutoSave() const;
 void AutoLoad();
};
