/*! @file
 * @brief
 * controlled (successful) termination of program
 */

extern "C"
{
 #include <sys/types.h>
 #include <signal.h>
}


/*!
 * @short
 * manages the ordered exiting of the program
 *
 * This class manages the ordered exiting of the program after the
 * factorization has been ended or aborted by program flow.
 * For one reason, there are some data structures (files, etc.)
 * that should be closed and even deleted.
 * The other reason is, that in server mode there may be some
 * threads running unaware of the fact, that factorization has stopped.
 * So the general idea of this manager class is to notify or cancel
 * these threads, too...
 */
class ExitManager
{
public:
  typedef void (*cleanup_function)(void);
private:
  static stack<cleanup_function> cleanup_functions;
#ifdef USE_NETWORK
  static CMutex Mutex, AliveMutex;
public:
  static void register_cancel(const pthread_t tid = pthread_self());
  static void unregister_cancel(const pthread_t tid = pthread_self());
  static void cancel_registered_threads();
#endif

public:
  static void register_exithandler(cleanup_function a_cleanup_function);
  // the difference between atexit and register_exithandler is that
  // the exithandlers registered by exithandler are only called, when
  // StopFactorization is invoked (and not on other kinds of termination).

  static void StopFactorization();
};
stack<ExitManager::cleanup_function> ExitManager::cleanup_functions;


#ifdef USE_NETWORK

CMutex ExitManager::Mutex, ExitManager::AliveMutex;
set<pthread_t> pending_threads;


/// register a thread for possible cancellation
void ExitManager::register_cancel(const pthread_t tid /* = pthread_self() */)
{
  Mutex.lock();
  pending_threads.insert(tid);
  Mutex.unlock();
}

/// unregister a thread for cancellation
void ExitManager::unregister_cancel(const pthread_t tid /* = pthread_self() */)
{
  Mutex.lock();
  pending_threads.erase(tid);
  Mutex.unlock();
}

/// cancel all registered threads
void ExitManager::cancel_registered_threads()
{
  Mutex.lock(); // only one thread may proceed...
  pthread_t tid = pthread_self();
  for (set<pthread_t>::const_iterator p=pending_threads.begin(); p!=pending_threads.end(); ++p)
   if (*p != tid)
    {
#ifdef VERBOSE_INFO
      cout << "cancelling thread " << *p << endl;
#endif
      int result=pthread_cancel(*p);
#ifdef VERBOSE_WARN
      if (result<0) cerr << "cancel request failed for " << *p << endl;
#endif
    }
  usleep(100000); // wait a small amount of time for possibly deferred cancellation
  Mutex.unlock();
}

#endif


/// register an exithandler
void ExitManager::register_exithandler(cleanup_function a_cleanup_function)
{
  cleanup_functions.push(a_cleanup_function);
}


#ifdef USE_NETWORK
static pthread_t tid_caller = 0;
static void catch_sigint(int signr)
{
  cout << "Signal " << signr << " received. (ThreadId=" << pthread_self() << ")" << endl;
  if (pthread_self()!=tid_caller)
   {
     cout << "going down: ThreadId=" << pthread_self() << endl;
     exit(0);
   }
}
#endif


/// cancel all registered thread, clean-up resources and exit the program
void ExitManager::StopFactorization()
{
#ifdef VERBOSE_INFO
  MARK;
  cout << "ExitManager::StopFactorization() called." << endl;
#endif
  if (XML_StatusFile!="")
   {
     // write a (final) status report
#ifdef VERBOSE_NOTICE
     cout << "writing XML Status Report to \"" << XML_StatusFile << "\"" << endl;
#endif
     ofstream StatusReport;
     StatusReport.open(XML_StatusFile.c_str(),ios::in|ios::out|ios::ate);
     if (StatusReport) StatusReport.seekp(0,ios::end);
     else
      {
        StatusReport.close();
        StatusReport.clear();
        StatusReport.open(XML_StatusFile.c_str(),ios::in|ios::out|ios::trunc);
      }
     if ( (StatusReport) && StatusReport.tellp()<=std::streampos(0) )
      {
        StatusReport << "<?xml version=\"1.0\"?>" << endl
                     << "<?xml-stylesheet href=\"qsieve-status.xsl\" type=\"text/xsl\" ?>" << endl
                     << endl << "<LOGFILE>" << endl << "</LOGFILE>" << flush;
      }
     if (StatusReport)
      {
        StatusReport.seekp(-10,ios::end); // overwrite "</LOGFILE>"
        // remark: strange! seekp(-10,ios::end) will be ignored, if "ios::in" in open is missing!
        statistical_data::XML_StatusReport(StatusReport);
        StatusReport << "</LOGFILE>" << flush;
      }
     if (StatusReport)
      {
#ifdef VERBOSE_INFO
        cout << "Status report written." << endl;
#endif
      }
     else
      {
        cout << "Status report not written due to a failure!" << endl;
      }
   }
  if (PrintSummary) FoundFactors.PrettyPrint(cout); // print a summary on screen
#ifdef USE_NETWORK
  AliveMutex.lock(); // only one thread may proceed...
  // only one thread may proceed...
  cancel_registered_threads(); // does a Mutex.lock(),
#endif
#ifdef VERBOSE_INFO
  cout << "calling registered cleanup-functions...." << endl;
#endif
  while (!cleanup_functions.empty())
   {
     cleanup_function a_cleanup_function = cleanup_functions.top(); // pop next cleanup function
     a_cleanup_function(); // and call it
     cleanup_functions.pop();
   }
#ifdef VERBOSE_NOTICE
  cout << "going down..." << endl;
#endif

#ifdef USE_NETWORK
  // exit(0) does not work well sometimes:
  // at least for linuxthreads,  sometimes there are still threads
  // that simply ignore, that the program is going down.
  // for this reason we send a kill signal to all processes in the current group.
  tid_caller=pthread_self();
  signal(SIGINT,catch_sigint);
  kill(0,SIGINT); // desperately convincing the group to commit suicide :-(
#endif

  exit(0); // terminate program
}
