/*! @file
 * @brief
 * dynamic relation specific stuff, need to be inlined because of compile-time dependencies
 */

/* About Dynamic-Factors/Special-Factors:

   This program uses a modified Multi-Large-Prime-Variant:
   Dynamic-Factors are like Single-Large-Primes, except that we are
   sieving with these factors as they get detected.

   Special-Factors are composite rests (esp. Double-Large-Primes variant). 
   On the one hand, we try to split Special-Factors by new Dynamic-Factors
   to get other Dynamic-Factors.  On the other hand a cycle search is
   performed occasionally to gain relations.

   In future implementations the cycle search may be extended to hyper edges
   as well (for multi-Large-Primes). [But most probably this approach would
   end up in using a standard method for solving a linear system of equations.]
*/


#ifdef IS_SERVER
 #include "Semaphore.H"
#endif

extern std::string DynamicRelationsFile;

class DynamicRelations
{
 // this class encapsulates access to the functions
 // which are specific for handling dynamic relations
 // (SLP relations = SingleLargePrime relations;
 //  synonymic: dynamic factor relations, dynamic relations)

 protected:
  // provide streams to store SLP relations
  static filebuf FileBuffer;
  static ostream DynamicRelations_to_file;
  static istream DynamicRelations_from_file;

  class IstreamPool
  {
   private:
    static const int PoolSize=6;
#ifdef IS_SERVER
    static CCriticalSection::Mutex PoolMutex;
    static CSemaphore Semaphore; // #resources to be protected by a semaphore
#endif
    typedef istream* Pistream;
    static Pistream Pistream_slots[PoolSize]; // = { NULL };
    static bool slot_occupied[PoolSize]; // = { false };
   public:
    static istream* acquire_istream()
     {
        {
#ifdef IS_SERVER
          if (!Semaphore.trywait())
           {
             cout << "IStreamPool: thread " << pthread_self() << " waiting for free slot!" << endl;
             Semaphore.wait();
             cout << "IStreamPool: thread " << pthread_self() << " got free slot!" << endl;
           }
          CCriticalSection CriticalSection(PoolMutex);
#endif
          for (int i=0; i<PoolSize; ++i)
           if (!slot_occupied[i])
            {
              slot_occupied[i]=true;
              if (i>1) cout << "IStreamPool: using slot " << i << endl;
              if (!Pistream_slots[i]) Pistream_slots[i]=new ifstream(DynamicRelationsFile.c_str());
              return Pistream_slots[i];
            }
        }
        MARK; cout << "semaphore handling failed! all slots in use, wasn't able to acquire a slot." << endl;
        exit(1); // this is not supposed to happen!!
     }
    static void release_istream(std::istream *pis)
     {
#ifdef IS_SERVER
       CCriticalSection CriticalSection(PoolMutex);
#endif
       for (int i=0; i<PoolSize; ++i)
        if (pis==Pistream_slots[i])
         {
           slot_occupied[i]=false;
#ifdef IS_SERVER
           Semaphore.post();
#endif
           return;
         }
       throw std::runtime_error("cannot release istream due to invalid pointer");
     }
  };

 public:
  static void Load();
  static void cleanup_files()
   {
     FileBuffer.close();
     remove(DynamicRelationsFile.c_str());
   }

 friend int main(const int argc, const char* const argv[]);
 friend class CRelation;
 friend class CRelation::ProvideDynamicRelationsStream;
};

#include "DynamicFactorRelations.H"

#ifdef IS_SERVER
#include "Cprocess_clients.H"
class CServerDynamicFactorRelations : protected TDynamicFactorRelations
{
 private:
  mutable CMutex monitor_mutex, SLP_mutex;
  vector<int> DynamicFactorList;
  int size_active_SLP, size_passive_SLP;

 public:
  inline void insert(const TDynamicFactorRelation& x)
   {
     SLP_mutex.lock();
     const bool flag=TDynamicFactorRelations::insert(x).second;
     SLP_mutex.unlock();
     if (flag)
      {
        if (!x.sieveable())
         {
           // x is a Single Large Prime, but a passive one: not sieveable
           ++size_passive_SLP;
           return;
         }
        monitor_mutex.lock();
        ++size_active_SLP;
        DynamicFactorList.push_back(x.factor);
        monitor_mutex.unlock();
      }
     else
      {
        cerr << "CServerDynamicFactorRelations::insert duplicate (rejected)!" << endl;
        cerr << "This indicates an inconsistency in dynamic factor handling!" << endl;
        cerr << "(The inconsistency is harmless, but we abort nevertheless.)" << endl;
        cerr << "SLP: " << x.factor << endl;
        exit(1);
      }
   }
  inline const int monitoredSize() // const /* const attribute could be dangerous due to multithreading!? */
   {
     monitor_mutex.lock();
     const int s=DynamicFactorList.size();
     monitor_mutex.unlock();
     return s;
   }
  inline const int size_active() const { return size_active_SLP; }
  inline const int size_passive() const { return size_passive_SLP; }
  inline const size_t size() const { return TDynamicFactorRelations::size(); }
  inline const int operator[] (const int i) // const /* const attribute could be dangerous due to multithreading!? */
   {
     monitor_mutex.lock();
     const int x=DynamicFactorList[i];
     monitor_mutex.unlock();
     return x;
   }
 friend inline void fillin_streampos(TDynamicFactorRelation& x);
 friend bool is_dynamic_factor(const int number);
 friend bool is_dynamic_factor(TDynamicFactorRelation &FaRelSearch);
 friend class Cprocess_clients;
};
CServerDynamicFactorRelations DynamicFactorRelations;
#else
TDynamicFactorRelations DynamicFactorRelations;
#endif


// *************** implementation part ****************


inline void fillin_streampos(TDynamicFactorRelation &FaRelSearch)
{
#ifdef IS_SERVER
  CUnlockMutexAtDestruction CriticalSection(DynamicFactorRelations.SLP_mutex);
  DynamicFactorRelations.SLP_mutex.lock();
#endif
  const TDynamicFactorRelations::const_iterator p = DynamicFactorRelations.find(FaRelSearch);
  if (p==DynamicFactorRelations.end()) throw std::runtime_error("SLP not found in DynamicFactorRelations.");
  FaRelSearch.fpos=(*p).fpos;
}

// not explicit inline, because it must be linked!
bool is_dynamic_factor(const int number)
{
  // returns, whether this number is already registered as a dynamic factor;
  // (dynamic factor -> prime number outside the static factorbase, also known as Single Large Prime)
  TDynamicFactorRelation relation;
  relation.factor = number;
#ifdef IS_SERVER
  CUnlockMutexAtDestruction CriticalSection(DynamicFactorRelations.SLP_mutex);
  DynamicFactorRelations.SLP_mutex.lock();
#endif
  const TDynamicFactorRelations::const_iterator pos = DynamicFactorRelations.find(relation); // search dynamic factor
  return (pos != DynamicFactorRelations.end());
}

// not explicit inline, because it must be linked!
bool is_dynamic_factor(TDynamicFactorRelation &FaRelSearch)
{
  // returns, whether this number is already registered as a dynamic factor;
  // (dynamic factor -> prime number outside the static factorbase, also known as Single Large Prime)
#ifdef IS_SERVER
  CUnlockMutexAtDestruction CriticalSection(DynamicFactorRelations.SLP_mutex);
  DynamicFactorRelations.SLP_mutex.lock();
#endif
  const TDynamicFactorRelations::const_iterator pos = DynamicFactorRelations.find(FaRelSearch); // search dynamic factor
  if (pos!=DynamicFactorRelations.end()) { FaRelSearch=*pos; return true; }
  return false;
}


filebuf DynamicRelations::FileBuffer;
ostream DynamicRelations::DynamicRelations_to_file(&DynamicRelations::FileBuffer);
istream DynamicRelations::DynamicRelations_from_file(&DynamicRelations::FileBuffer);

#ifdef IS_SERVER
CCriticalSection::Mutex DynamicRelations::IstreamPool::PoolMutex;
CSemaphore DynamicRelations::IstreamPool::Semaphore( DynamicRelations::IstreamPool::PoolSize); // #resources to be protected by a semaphore
#endif
DynamicRelations::IstreamPool::Pistream DynamicRelations::IstreamPool::Pistream_slots[DynamicRelations::IstreamPool::PoolSize] = { NULL };
bool DynamicRelations::IstreamPool::slot_occupied[DynamicRelations::IstreamPool::PoolSize] = { false };


void DynamicRelations::Load()
{
  // read in Dynamic Relations
#ifdef VERBOSE_NOTICE
  cout << "Reading dynamic relations from file..." << endl;
#endif
  DynamicRelations_from_file.seekg(0,ios::beg);
  unsigned int line = 0;
  while (DynamicRelations_from_file.peek()=='G' || DynamicRelations_from_file.peek()=='U')
   {
     ++line;
     if (line%1000==0) cout << line << " relations processed.\r" << flush;
     if (DynamicRelations_from_file.peek()=='U')
      {
#ifdef VERBOSE_WARN
        cerr << "skipping obsolete/invalid relation." << endl;
#endif
        DynamicRelations_from_file.ignore(1000000,'\n');
        continue;
      }
     TDynamicFactorRelation relation;
     relation.fpos = DynamicRelations_from_file.tellg();
     string s; DynamicRelations_from_file >> s;
     DynamicRelations_from_file >> relation.factor;
     relation.append_for_sieving();
     DynamicRelations_from_file.ignore(1000000,'\n');
     DynamicFactorRelations.insert(relation);
   }
  cout << line << " relations processed." << endl;
  if (DynamicRelations_from_file.peek()!=EOF)
   {
     cerr << "Dynamic Relations: garbage detected. Giving up. Please validate." << endl;
     exit(1);
   }
  DynamicRelations_from_file.clear();
  DynamicRelations_to_file.seekp(0, ios::end); // as a precaution...
#ifdef VERBOSE_NOTICE
  cout << "Recover: " << DynamicFactorRelations.size()
       << " dynamic relations read." << endl;
#endif
}
