/*! @file
 * @brief
 * configfile related functions
 */

extern "C"
{
 #include <sys/types.h>
 #include <sys/stat.h>
 #include <unistd.h>
}


// a few parameters for easy-factorization (and their default values)
bool SkipFermat    = false; // try the fermat method?
bool SkipPhi       = false; // try Pollard-Phi-Method?
bool SkipFibonacci = false; // try Fibonacci-Method?
bool SkipEasyECM   = false; // try elliptic curve method im easy-mode?
bool SkipECM       = false; // try elliptic curve method?
bool PrintSummary  = false; // do we a summary when factorization is complete?
int rho_Phase      =   150000; // main initialization for Rho-Phase
int phi_Phase1     =   500000; // upper bound for Phase 1 in phi-Method
double phi_Phase2  = 10000000.0; // upper bound for phi-Method Phase 2 (=isolated factor)
double elcu_Phase1 =    20000.0; // upper bound for elliptic curve Phase 1
double elcu_Phase2 =  2000000.0; // upper bound for elliptic curve Phase 2
int elcu_Kurven    =      100; // number of elliptic curves to try
int ecm_curves_processed = 0; // number of curves done so far

double Factor_Threshold = 3.0; /* exponent, which allows bigger factors (which are not part of the static factorbase)
				  suggestive values: 1.0 up to 3.0
				  1.0: no additional factors outside the static factorbase
				  up to 2.0: prime factors outside the static factorbase
                                             up to the square of the biggest factor inside the static FB.
				  >2.0: factors up to (biggest factor inside the static FB) raised to the power of Factor_Threshold.
			       */


string XML_StatusFile;

string TempDir              = "/tmp"; // default directory for temporary files
string ConfigFile           = "qsieve.cfg"; // default, may be overwritten by getenv("QSIEVE_CFG")
string DynamicRelationsFile = "dynamic_relations.dat"; // default, may be overwritten by entry in configfile

string ClientAccountName    = ""; // can be set by ConfigFile. If empty, then socket info will be used instead of an account name

unsigned long int FFT_MAX_MEM_USAGE = 32*1024;


#ifndef IS_CLIENT
const string FactorizationsFile   = "factorizations.txt";
std::ofstream Factorization_to_file;
/* Factorizations are logged to this file. Numbers to factorize and any
   found factors are appended. The file can be deleted, if the logged data
   is of no use anymore.
*/
#endif


//! needed for security reasons to prevent escape sequences or similar bad things
const string read_restrictedName(istream &is)
{
  string s;
  while (is.good() && is.peek()!=EOF && isblank(is.peek())) is.get();
  while (is.good() && is.peek()!=EOF && s.length()<=40)
   {
     int i=is.get();
     if (isalnum(i) || i==' ' || i=='@' || i=='#' || i==',' || i=='.' || i=='+' || i=='-') s+=static_cast<char>(i);
     else return s;
   }
  return s;
}

void Get_ConfigFile()
{
  // resolve ConfigFile
  if (getenv("QSIEVE_CFG"))
   {
     ConfigFile=getenv("QSIEVE_CFG");
#ifdef VERBOSE_NOTICE
     cout << "Setting ConfigFile to \"" << ConfigFile << "\"" << endl;
#endif
   }
  // and convert it to real/absolute path
#ifdef PATH_MAX
  char resolved_path[PATH_MAX];
#else
  char resolved_path[4096];
#endif /* PATH_MAX */
  struct stat buf;
  // for Cygwin: realpath doesn't return an error, if file doesn't exist
  // we ignore file permission issues and use stat to check for existence
  if (realpath(ConfigFile.c_str(),resolved_path) && (stat(resolved_path,&buf)==0))
   ConfigFile=resolved_path;
  else
   {
     cerr << "Cannot find ConfigFile " << "\"" << ConfigFile << "\"" << endl;

#if defined(etc_ConfigFile)
     cout << "trying to use \"" << etc_ConfigFile << "\"" << endl;
     if (realpath(etc_ConfigFile,resolved_path) && (stat(resolved_path,&buf)==0))
       {
         ConfigFile=resolved_path;
         return;
       }
     else
       cerr << "no ConfigFile found!" << endl;
#endif

     // server & standalone versions need a configuration file,
     // clients do not need it (but they make use of it as well)
#ifndef IS_CLIENT
     exit(1);
#endif
   }
}


void Read_ConfigFile(void)
{
  Get_ConfigFile(); // get position of configfile (evaluate environment, etc.)

  ifstream in(ConfigFile.c_str());
  char InputLine[200];
  string zeile;
  if (in)
    {
#ifdef VERBOSE_INFO
      cout << "Reading configuration file" << endl;
#ifdef IS_CLIENT
      cout << "Some server related options may have no effect to clients." << endl;
#endif
#endif
      while (!in.eof())
	{
	  in.getline(InputLine,sizeof(InputLine),'\n');
	  if (InputLine[0]=='#') continue; // read over comment lines
	  zeile=InputLine;
          while (zeile[0]==' ') zeile.erase(0,1);
          if (zeile.empty() || zeile[0]=='#') continue; // read over empty lines and comment lines

          // process keywords
          istringstream is(zeile);
          string Keyword;
          is >> Keyword;
          if (Keyword=="DynamicRelationsFile")
           {
             string s;
             is >> s;
             if (s!="=")
              {
                cerr << "Error in " << ConfigFile  << "!" << endl;
                exit(1);
              }
             is >> DynamicRelationsFile;
             continue;
           }
          if (Keyword=="WorkDir")
           {
             string s;
             is >> s;
             if (s!="=")
              {
                cerr << "Error in " << ConfigFile  << "!" << endl;
                exit(1);
              }
             is >> s;
#ifdef VERBOSE_NOTICE
             cout << "Setting current working directory to \"" << s << "\"." << endl;
#endif
	     if (chdir(s.c_str())!=0)
              {
                cerr << "Changing working directory failed!" << endl;
                exit(2);
              };
             continue;
           }
          if (Keyword=="TempDir")
           {
             string s;
             is >> s;
             if (s!="=")
              {
                cerr << "Error in " << ConfigFile  << "!" << endl;
                exit(1);
              }
             is >> s;
#ifdef VERBOSE_NOTICE
             cout << "Setting TempDir to \"" << s << "\"." << endl;
#endif
             TempDir=s; 
             continue;
           }
          if (Keyword=="XMLStatusFile")
           {
             string s;
             is >> s;
             if (s!="=")
              {
                cerr << "Error in " << ConfigFile  << "!" << endl;
                exit(1);
              }
             is >> s;
#ifdef VERBOSE_NOTICE
             cout << "Setting current XMLStatusFile to \"" << s << "\"." << endl;
#endif
             XML_StatusFile=s; 
             continue;
           }
          if (Keyword=="ClientAccount")
           {
             string s;
             is >> s;
             if (s!="=")
              {
                cerr << "Error in " << ConfigFile  << "!" << endl;
                exit(1);
              }
             s=read_restrictedName(is);
#ifdef VERBOSE_NOTICE
             cout << "Setting current ClientAccount to \"" << s << "\"." << endl;
#endif
             ClientAccountName=s; 
             continue;
           }
          if (Keyword=="SkipFermat")
           {
	     SkipFermat=true;
             continue;
           }
          if (Keyword=="SkipPhi")
           {
	     SkipPhi=true;
             continue;
           }
          if (Keyword=="SkipFibonacci")
           {
	     SkipFibonacci=true;
             continue;
           }
          if (Keyword=="SkipEasyECM")
           {
	     SkipEasyECM=true;
             continue;
           }
          if (Keyword=="SkipECM")
           {
	     SkipECM=true;
             continue;
           }
          if (Keyword=="PrintSummary")
           {
	     PrintSummary=true;
             continue;
           }
          if (Keyword=="MemoryLimit")
           {
             string s;
             is >> s;
             if (s!="=")
              {
                cerr << "Error in " << ConfigFile  << "!" << endl;
                exit(1);
              }
             is >> FFT_MAX_MEM_USAGE;
             FFT_MAX_MEM_USAGE<<=10; // MB -> KB
#ifdef VERBOSE_NOTICE
             cout << "Setting FFT MemoryLimit to " << (FFT_MAX_MEM_USAGE>>10) << "MB." << endl;
#endif
             continue;
           }
        }
    }
  else
    {
       cout << ConfigFile << " not found! Using default values!" << endl;
#ifndef IS_CLIENT
       static char ch = 'N';
       if (ch!='Y')
         {
           cout << "Bitte mit [Y]es besttigen!" << endl;
           cout << "Please confirm with [Y]es." << endl;
           cin >> ch;
           if (ch!='Y') exit(1);
         }
#endif
    }

#ifdef VERBOSE_INFO
  char cwd[4096];
  cout << "Current working directory: " << getcwd(cwd,4096) << endl;
  cout << "Configuration file: " << ConfigFile << endl;
  cout << "temp directory: " << TempDir << endl;
  cout << "DynamicRelationsFile: " << DynamicRelationsFile << endl;
#endif
#if !defined(IS_CLIENT) && !defined(IS_VALIDATOR)
  // clients do report factors to the server and not to this file,
  // so only stand-alone versions and servers need this:
  if (!Factorization_to_file.is_open()) Factorization_to_file.open(FactorizationsFile.c_str(),ios::out|ios::app);
#endif
}


#ifndef IS_CLIENT
// clients will be configured by the server, so only stand-alone and server versions need this...
void tune_parameters(const unsigned int DecimalDigits)
{
  ifstream in(ConfigFile.c_str());
  char InputLine[200];
  string zeile;
  unsigned int Stellen, gewaehlte_Stellenzahl=0;
  if (in)
    {
#ifdef VERBOSE_INFO
      cout << "Reading configuration file" << endl; 
#endif
      while (!in.eof())
	{
	  in.getline(InputLine,sizeof(InputLine),'\n');
	  if (InputLine[0]=='#') continue; // read over comment lines
	  zeile=InputLine;
          while (zeile[0]==' ') zeile.erase(0,1);
          if (zeile.empty() || zeile[0]=='#') continue; // read over empty lines / comment lines

          // process keywords
          istringstream is(zeile);
          string Keyword;
          is >> Keyword;
	  if (Keyword=="SkipFermat") continue;
	  if (Keyword=="SkipPhi") continue;
	  if (Keyword=="SkipFibonacci") continue;
          if (Keyword=="SkipEasyECM") continue;
	  if (Keyword=="SkipECM") continue;
	  if (Keyword=="PrintSummary") continue;
	  if (Keyword=="MemoryLimit") continue;
          if (Keyword=="DynamicRelationsFile" || Keyword=="WorkDir" || Keyword=="TempDir" || Keyword=="XMLStatusFile" || Keyword=="ClientAccount")
           {
             string s;
             is >> s;
             if (s!="=")
              {
                cerr << "Error in " << ConfigFile  << "!" << endl;
                exit(1);
              }
             // is >> DynamicRelationsFile; // do not read anything here!
             continue;
           }

          // otherwise: line contains parameter for tuning
	  Stellen = atoi(zeile.substr(0, zeile.find(":")).c_str());

          if (Stellen==0)
           {
             cerr << "Error in " << ConfigFile  << "!" << endl;
             cerr << "-> " << zeile << endl;
             exit(1);
           }

	  if (Stellen > DecimalDigits || Stellen==0) break;
          gewaehlte_Stellenzahl=Stellen;
	  zeile.erase(0, zeile.find(":")+1);
	  rho_Phase=atoi(zeile.substr(0, zeile.find(",")).c_str());
	  zeile.erase(0, zeile.find(",")+1);
	  phi_Phase1=atoi(zeile.substr(0, zeile.find(",")).c_str());
	  zeile.erase(0, zeile.find(",")+1);
	  phi_Phase2=strtod(zeile.substr(0, zeile.find(",")).c_str(),NULL);
	  zeile.erase(0, zeile.find(",")+1);
	  elcu_Phase1=strtod(zeile.substr(0, zeile.find(",")).c_str(),NULL);
	  zeile.erase(0, zeile.find(",")+1);
	  elcu_Phase2=strtod(zeile.substr(0, zeile.find(",")).c_str(),NULL);
	  zeile.erase(0, zeile.find(",")+1);
	  elcu_Kurven=atoi(zeile.substr(0, zeile.find(",")).c_str());
	  zeile.erase(0, zeile.find(",")+1);
	  StaticFactorbase::Size_StaticFactorbase=atoi(zeile.substr(0, zeile.find(":")).c_str());
	  zeile.erase(0, zeile.find(":")+1);
          Factor_Threshold = atof(zeile.substr(0, zeile.find(":")).c_str());
          if (Factor_Threshold<=0.01 || Factor_Threshold>10.0)
           {
             MARK;
             cerr << "invalid value of Factor_Threshold: " << Factor_Threshold << endl;
             exit(1);
           }
	  zeile.erase(0, zeile.find(":")+1);

	  LogicalSieveSize = atoi(zeile.c_str());
	  LogicalSieveSize = MAX(LogicalSieveSize,PhysicalSieveSize);
	}
    }
  else
    {
       cout << ConfigFile << " not found! Using default values!" << endl;
       static char ch = 'N';
       if (ch!='Y')
         {
           cout << "Bitte mit [Y]es besttigen!" << endl;
           cout << "Please confirm with [Y]es." << endl;
           cin >> ch;
           if (ch!='Y') exit(1);
         }
    }

#ifdef VERBOSE_INFO  
  cout << "Configuration:" << endl;
  cout << "#decimal digits: " << DecimalDigits << endl;
  cout << "actual parameters for " << gewaehlte_Stellenzahl << " digits" << endl;
  cout << "Pollard-Rho: " << rho_Phase << endl;
  cout << "Pollard-Phi: phase 1: " << phi_Phase1 << " phase 2: " << setprecision(1) << phi_Phase2 << endl;
  cout << "elliptic curves: phase 1: " << elcu_Phase1
       << " phase 2: " << elcu_Phase2
       << " #curves: " << elcu_Kurven << endl; 
  cout << "Size of static factorbase: " << StaticFactorbase::Size() << endl;
  cout << "factor-threshold (exponent): " << Factor_Threshold << endl;
  cout << "sieve interval per polynomial: [" << -LogicalSieveSize << "," << LogicalSieveSize << "]" << endl;
#endif

  if (StaticFactorbase::Size()>StaticFactorbase::MaxSize)
    {
      cerr << "MaxSize_StaticFactorbase is too small (or Size_StaticFactorbase is to big)!" << endl;
      exit(1);
    }
}
#endif /* not defined(IS_CLIENT) */
