// Kurvenanpassung nach der Methode der kleinsten Quadrate
// vgl. [1] Robert Sedgewick: Algorithmen, Gau+Kurvenanpassung
//      [2] Stoer: Numerische Mathematik 1
//      [3] John Mandel: The Statistical Analysis Of Experimental Data

// written 2004-02 by Thorsten Reinecke

/*! @file
 * @brief
 * linear fitting of curves using the method of least squares
 */


#include <iostream>
#include <iomanip>
#include <sstream>
#include <utility>
#include <vector>
#include <cmath>


//! provides numerical linear algebra tools
namespace numeric_linalg
{

using namespace std;


class CLinearFitting
{
public:
 typedef double (*numeric_function)(double);
 typedef vector<numeric_function> numeric_functions;

protected:
 const int M;

private:

 // Funktionen f[0],...,f[M-1], welche als Linearkombination
 // die (nherungsweise) gesuchte Funktion f bilden
 // mit f(x):=c[0]*f[0](x)+c[1]*f[1](x)+...+c[M-1]*f[M-1](x)
 vector<double> c;
 vector<numeric_function> f;
 vector<vector<double> > a;

 // Anzahl der eingegebenen Wertepaare, an denen die gesuchte Funktion
 // als Linearkombination der bergebenen Funktionen angenhert werden soll,
 // so dass die Summe der Quadrate der Funktionswerte einen minimalen
 // Fehler ergibt
 int N;

 vector<vector<double> > F;

 bool is_ready; // Kurvenanpassung bereits erfolgt?

public:
 explicit CLinearFitting(const numeric_functions& fns)
  : M(fns.size()), c(M), f(fns), a(M), N(0), F(M+1), is_ready(false)
  {
    for (int i=0; i<M; ++i) a[i].resize(M+1);
  }

 CLinearFitting(const numeric_function fns[], const int size)
  : M(size), c(M), f(size), a(M), N(0), F(M+1), is_ready(false)
  {
    for (int i=0; i<M; ++i) f[i]=fns[i];
    for (int i=0; i<M; ++i) a[i].resize(M+1);
  }

 void add_point(const double x, const double y)
  {
    for (int i=0; i<M; ++i) F[i].push_back(f[i](x));
    F[M].push_back(y);
    ++N; // neue Anzahl an Vektoren
    is_ready=false; // neuer Wert macht neue Kurvenanpassung notwendig
  }

protected:

 void eliminate()
 {
  for (int i=0; i<M; ++i)
   {
    int max=i;
    for (int j=i+1; j<M; ++j)
      if (abs(a[j][i])>abs(a[max][i])) max=j;
    for (int k=i; k<=M; ++k) swap(a[i][k],a[max][k]);
    for (int j=i+1; j<M; ++j)
     for (int k=M; k>=i; --k)
      a[j][k]-=a[i][k]*a[j][i]/a[i][i];
   }
 }

 void substitute()
 {
   //for (int i=0; i<M; ++i) c[i]=0.0;
   for(int j=M-1; j>=0; --j)
    {
      double t=0.0;
      for (int k=j+1; k<M; ++k) t+=a[j][k]*c[k];
      c[j]=(a[j][M]-t)/a[j][j];
    }
 }

 void anpassen()
 {
  // Matrix vorbereiten
  for (int i=0; i<M; ++i)
   for (int j=0; j<=M; ++j)
    {
      double t = 0.0;
      for (int k=0; k<N; ++k) t+=F[i][k]*F[j][k];
      a[i][j]=t;
    }

  // Gauss
  eliminate();
  substitute();
  is_ready=true;
 }

public:

 //! Liefert (angenherten) Funktionswert zurck
 double operator() (const double x)
  {
    if (!is_ready) anpassen();
    double y=0;
    for (int i=0; i<M; ++i) y+=c[i]*f[i](x);
    return y;
  }

 //! liefert Koeffizienten der Linearkombination zurck
 double operator[] (int i)
  {
    if (!is_ready) anpassen();
    return c[i];
  }
};


class CLinearFitting_with_symbols : public CLinearFitting
{
private:
 vector<string> f_symbol;
public:
 explicit CLinearFitting_with_symbols(const numeric_functions& fns)
  : CLinearFitting(fns), f_symbol(fns.size())
  {
    char buf[16];
    for (int i=0; i<M; ++i)
     {
       sprintf(buf,"f%i",i);
       f_symbol[i]=buf;
     }
  }

 CLinearFitting_with_symbols(const numeric_function fns[], const int size)
  : CLinearFitting(fns,size), f_symbol(size)
  {
    char buf[16];
    for (int i=0; i<M; ++i)
     {
       sprintf(buf,"f%i",i);
       f_symbol[i]=buf;
     }
  }

 CLinearFitting_with_symbols(const numeric_function fns[], const char* const * const symb_names, const int size)
  : CLinearFitting(fns,size), f_symbol()
  {
    for (int i=0; i<size; ++i) f_symbol.push_back(symb_names[i]);
  }

 const string symbolic()
  {
    ostringstream os;
    bool flag = false;
    os << "f(x):=";
    for (int i=0; i<M; ++i)
     if (abs(operator[](i))>1e-12) // cut irrelevant coefficients
      {
        os << " " << operator[](i) << "*" << f_symbol[i];
        flag=true; os << showpos;
      }
    if (!flag) os << "0";
    return os.str();
  }

};


class CCubicFitting : public CLinearFitting_with_symbols
{
private:
 static double f0(double) { return 1; }
 static double f1(double x) { return x; }
 static double f2(double x) { return x*x; }
 static double f3(double x) { return x*x*x; }
 static const CLinearFitting::numeric_function fns[4];
 static const char* const fns_symb[4];
public:
  CCubicFitting() : CLinearFitting_with_symbols(fns,fns_symb,4) { }
};
const CCubicFitting::numeric_function CCubicFitting::fns[4] = { &f0, &f1, &f2, &f3 };
const char* const CCubicFitting::fns_symb[4] = { "x^0", "x^1", "x^2", "x^3" };


} // end of namespace numeric_linalg

#if 0

// example:

using namespace std;
using namespace numeric_linalg;

static double f0(double x) { return 1.0; }
static double f1(double x) { return x; }
static double f2(double x) { return x*x; }
static double f3(double x) { return x*x*x; }
static double f4(double x) { return sin(x); }
static double f5(double x) { return sqrt(x); }
static double f6(double x) { return exp(x); }

int main()
{
 CLinearFitting::numeric_function fns[] = { &f0, &f1, &f2, &f3, &f4, &f5, &f6 };
 const char* const symb_names[] = { "x^0", "x^1", "x^2", "x^3", "sin(x)", "sqrt(x)", "exp(x)" };

 const int M = sizeof(fns)/sizeof(CLinearFitting::numeric_function);
 const int N=10;

 CLinearFitting_with_symbols LA(fns,symb_names,M);
 double x[N],y[N];

 for (int i=0; i<N; ++i)
  {
    double w=i;
    x[i]=w;
    //y[i]=0.3*sqrt(w)+1*w*w+1*w+2+19*sin(w);
    //y[i]=0.3*w*w*w-12*w*w+13*w-4+drand48();
    y[i]=w-sqrt(w);
    LA.add_point(x[i],y[i]);
  }

 for (int i=0; i<N; ++i)
  {
    cout << x[i] << " gemessen: " << y[i] << " gegl.: " << LA(x[i]) << endl;
  }
 cout << LA.symbolic() << endl;


 CCubicFitting CF;
 for (int i=0; i<N; ++i) CF.add_point(x[i],y[i]);
 for (int i=0; i<N; ++i)
  {
    cout << x[i] << " gemessen: " << y[i] << " gegl.: " << CF(x[i]) << endl;
  }
 cout << CF.symbolic() << endl;

 return 0; 
}

#endif // example usage
