#ifndef UNIX_BUFFER_
#define UNIX_BUFFER_

// some abstraction layers for using UNIX-network services in C++ (mainly TCP)
// written by Thorsten Reinecke (reinecke@thorstenreinecke.de)
// last change: 2003-11-17

/*! @file
 * @brief
 * some abstraction layers for using UNIX-network services in C++ (mainly TCP)
 */


extern "C"
{
  #include <sys/socket.h>
  #include <netinet/in.h>
  #include <netdb.h>
  #include <unistd.h>
}

#include <cerrno>
#include <string>
#include <iostream>
#include <sstream>

using std::cerr;
using std::endl;
using std::flush;
using std::iostream;
using std::streambuf;


//! This class provides a streambuffer for handling network connections.
class unix_buffer : public streambuf {
protected:
  static const int buffersize = 4096;
  static const int pushbacksize = 10;
  int fd;
  bool is_open;
  char obuffer[buffersize];
  char ibuffer[buffersize+pushbacksize];

  const int empty_buffer(void)
   {
     if (fd<0) // invalid file descriptor?
      {
        cerr << "unix_buffer::empty buffer(): invalid descriptor! fd=" << fd << endl;
        return EOF;
      }
     int count = pptr()-pbase(); if (!count) return 0;
     int written=send(fd, obuffer, count, MSG_NOSIGNAL);
     if (written!=count)
      {
        if (written<=0)
         {
           cerr << "unix_buffer::empty_buffer():"; perror(NULL);
           return EOF;
         }
        cerr << "unix_buffer::empty_buffer(): "
             << written << " bytes of " << count << " sent." << endl;
      }
     pbump(-written);
     return written;
   }

  virtual int overflow (int c) 
   {
     if (c != EOF) { *pptr() = c; pbump(1); }
     if (empty_buffer() == EOF) return EOF;
     return c;
   }

  virtual int sync ()
   {
     return (empty_buffer() == EOF) ? -1 : 0;
   }

  virtual int underflow ()
   {
     if (gptr() < egptr()) return *gptr();
     int pbcount = gptr() - eback();
     if (pbcount > pushbacksize) pbcount = pushbacksize;
     if (pbcount > 0) memmove (ibuffer+(pushbacksize-pbcount), gptr()-pbcount, pbcount);

     int count;
     do {
       count = recv(fd, ibuffer+pushbacksize, buffersize, MSG_NOSIGNAL);
       if (count <= 0) // any unexpected behaviour?
        {
         if (count==0) return EOF;
         if (count==-1 && errno == EINTR) continue; // this should fix some problems...
 	 perror(NULL); return EOF;
        }
     } while (count<=0);
     setg(ibuffer+(pushbacksize-pbcount), ibuffer+pushbacksize,
	  ibuffer+pushbacksize+count);
     return *gptr();
   }
  
public:

  //! returns the socket descriptor
  inline const int get_descriptor() const { return fd; }

  //! set a connection timeout
  inline void set_timeout (const int secs, const int u_secs = 0)
   {
     timeval tv;
     tv.tv_sec = secs; // secs timeout
     tv.tv_usec = u_secs; // micro-secs timeout
     if (setsockopt(fd, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv))==-1)
       { cerr << "unix_buffer::set_timeout() "; perror(NULL); }
     if (setsockopt(fd, SOL_SOCKET, SO_SNDTIMEO, &tv, sizeof(tv))==-1)
       { cerr << "unix_buffer::set_timeout() "; perror(NULL); }
   }

  /*!
   *  open a network connection via the given socket descriptor
   *  @remark 
   *          - it is not possible to open more than one connetion for a buffer
   *          - it is not possible to close the connection without destructing the buffer
   */
  void open(const int _fd)
  {
    if (fd>0)
     {
       cerr << "unix_buffer::open(const int _fd): descriptor is already in use!" << endl;
       exit(1);
     }
    if (_fd<0)
     {
       cerr << "unix_buffer::open(const int _fd): descriptor is invalid!" << endl;
       exit(1);
     }
    fd = _fd;
    is_open = true;
    setp(obuffer, obuffer+(buffersize-1));
    setg(ibuffer+pushbacksize, ibuffer+pushbacksize, ibuffer+pushbacksize);
  }

  //! create an unix_buffer that is ready to be connected to a socket descriptor
  unix_buffer(void) // Default constructor
   : fd(0), // 0 denotes no descriptor defined yet
     is_open(true) // as no descriptor is defined, no descriptor need to be closed later
  {
    set_timeout(75); // Standard: 75 Sekunden
    setp(obuffer, obuffer+(buffersize-1));
    setg(ibuffer+pushbacksize, ibuffer+pushbacksize, ibuffer+pushbacksize);
  }

  //! create an unix_buffer that is connected to the given socket descriptor
  explicit unix_buffer(const int _fd, const int secs_timeout = 75)
   : fd(_fd), is_open(true)
  {
    set_timeout(secs_timeout);
    setp(obuffer, obuffer+(buffersize-1));
    setg(ibuffer+pushbacksize, ibuffer+pushbacksize, ibuffer+pushbacksize);
  }

 //! create an unix_buffer that connects to the given socket
 //! (and create a socket descriptor by itself)
 unix_buffer(const std::string &host, const int port)
  : fd(0), is_open(true)
  {
    // refer "man getaddrinfo" and "man socket" 
    struct addrinfo hints; // our wishes are placed here
    memset(&hints,0,sizeof(hints)); // must be zeroed for default options
    hints.ai_family=PF_INET; // we want IPv4 as protocol
    hints.ai_socktype=SOCK_STREAM; // and we need a stream, not datagram!
    addrinfo *addrinfo_res = NULL; // here the result will be delivered
    const int retval = getaddrinfo(host.c_str(),NULL,&hints,&addrinfo_res);
    if ( retval || addrinfo_res==NULL ) // any error?
     {
       cerr << "can't reach " << "\"" <<  host << "\"" << endl;
       cerr << "Error given by getaddrinfo: " << endl;
       cerr << gai_strerror(retval) << endl;
       exit(1);
     }
    if (addrinfo_res->ai_socktype!=SOCK_STREAM) // we got a "stream"-protocol?
     {
       cerr << "provided protocol doesn't support SOCK_STREAM" << endl;
       exit(1);
     }
    switch (addrinfo_res->ai_family)
     {
      case PF_INET:
       //cerr << "PF_INET: using IPv4!" << endl;
       fd = socket(AF_INET, SOCK_STREAM, 0);
       reinterpret_cast<sockaddr_in*>(addrinfo_res->ai_addr)->sin_port=htons(port);
       break;
      case PF_INET6:
       cerr << "PF_INET6: try using IPv6 (untested feature!)" << endl;
       fd = socket(AF_INET6, SOCK_STREAM, 0);
       reinterpret_cast<sockaddr_in6*>(addrinfo_res->ai_addr)->sin6_port=htons(port);
       break;
      default:
       cerr << "too bad! ai_family isn't supported by unix_buffer!" << endl;
       exit(1);
     }

    if (fd < 0) // is the descriptor valid?
     {
      cerr << "Can't open socket" << endl;
      exit(1);
     }

    {
      int retries_on_EAGAIN = 3;
      while ( connect(fd, addrinfo_res->ai_addr, addrinfo_res->ai_addrlen) <0 )
       {
        switch (int err_number=errno)
         {
           case ETIMEDOUT:
            cerr << "." << flush; sleep(10); break;
           case ENETUNREACH:
            cerr << "%" << flush; sleep(10); break;
           case EAGAIN:
            if (retries_on_EAGAIN--)
             {
               cerr << "Can't connect socket (" << strerror(err_number) << ")" << endl;
               if (err_number==EAGAIN)
                {
                  cerr << "Trying it again in 10 secs..." << endl;
                  sleep(10);
                }
               break;
             }
           default:
            cerr << "Can't connect socket (" << strerror(err_number) << ")" << endl;
            exit(1);
         }
       }
    }
    freeaddrinfo(addrinfo_res);
    is_open = false;
    setp(obuffer, obuffer+(buffersize-1));
    setg(ibuffer+pushbacksize, ibuffer+pushbacksize, ibuffer+pushbacksize);
  }

  //! destroy the unix_buffer and close network connection (if any was open)
  virtual ~unix_buffer()
   {
     if (fd<0) return;
     sync();
     if (!is_open)
      {
        shutdown(fd, SHUT_RDWR);
        close(fd);
      }
   }

};


//! a simple network connection stream
class unix_io_client : public iostream {
protected:
  unix_buffer buf;
public:
  unix_io_client(const std::string &host, const int port) : iostream(&buf), buf(host, port) { }
  /*!
   * @param host symbolic/numerical internet address
   * @port  port number to connect
   *
   * A tcp based network connection will be established and provided as stream.
   */
};


/*!
 * @param port port to listen for a tcp based network connection
 * @param qsize optional "backlog" parameter (refer "listen(2)" manpage)
 * @result socket descriptor for the connection, or -1 on error
 *
 * Use this function to wait for incoming tcp connection requests
 * and for handling the serversided part to establish the network connection.
 */ 
int open_internet_port(const int port, const int qsize = 5)
{
  int sd;
  int val = 1;
  struct sockaddr_in server;
  struct linger lingval;

  if (port <= 0) return -1;

  sd = socket(AF_INET, SOCK_STREAM, 0);
  if (sd < 0) return sd;
  if (setsockopt(sd, SOL_SOCKET, SO_REUSEADDR, reinterpret_cast<char*>(&val), sizeof(val)) <0)
    return -1;

  // nderung Thorsten: Linger-Option aktivieren
  lingval.l_onoff=1; // beim Schlieen des Sockets ggf. warten
  lingval.l_linger=10*100; // Timeout = 10 Sekunden
  if (setsockopt(sd, SOL_SOCKET, SO_LINGER, reinterpret_cast<char*>(&lingval), sizeof(lingval)) <0)
    return -1;

  server.sin_family = AF_INET;
  server.sin_addr.s_addr = INADDR_ANY;
  server.sin_port = htons(port);
  if (bind(sd, reinterpret_cast<struct sockaddr *>(&server), sizeof(server)) < 0) return -1;
  listen (sd, qsize);
  return sd;
}


// utils for verbose output of meta-data to streams

//! returns information about the specified socket to the given stream
inline std::ostream& operator<< (std::ostream& ostr, const sockaddr &sa)
{
  // outputs sockaddr info to given stream
  char host[1024];
  char service[1024];
  const int flags = NI_NOFQDN;
  const int retval = getnameinfo(&sa,sizeof(sa),host,sizeof(host),service,sizeof(service),flags);

  switch (retval)
  {
    case 0:
     ostr << host << " " << service;
     break;
    case EAI_AGAIN:
     ostr << "(The name could not be resolved at this time.)"; break;
    case EAI_BADFLAGS:
     ostr << "(The flags parameter has an invalid value.)"; break;
    case EAI_FAIL:
     ostr << "(A non-recoverable error occurred.)"; break;
    case EAI_FAMILY: 
     ostr << "(The address family was not recognized.)"; break;
    case EAI_MEMORY:
     ostr << "(Out of memory.)"; break;
    case EAI_NONAME:
     ostr << "(The  name  does  not  resolve.)"; break;
    case EAI_SYSTEM:
     ostr << "(A system error occurred.)"; break;
    default:
     ostr << "(unspecified error in getnameinfo.)"; 
   }
  return ostr;
}

/*!
 * @param socket_descriptor a socket descriptor
 * @result string that contains human readable information
 *         about the client who has initiated the network connection
 */
inline std::string peer_info(const int socket_descriptor)
{
 // outputs meta-data for socket_descriptor
 std::ostringstream ostr;

 struct sockaddr sa;
 socklen_t sl = sizeof(sa);

 const int retval = getpeername(socket_descriptor, &sa, &sl);
 if (retval==0) ostr << sa;
 return ostr.str();
}

/*!
 * @param socket_descriptor a socket descriptor
 * @result string that contains human readable information
 *         about the server who has "picked up" the network connection
 */
inline std::string socket_info(const int socket_descriptor)
{
 // outputs meta-data for socket_descriptor
 std::ostringstream ostr;

 struct sockaddr sa;
 socklen_t sl = sizeof(sa);

 const int retval = getsockname(socket_descriptor, &sa, &sl);
 if (retval==0) ostr << sa;
 return ostr.str();
}

/*!
 * @param socket_descriptor a socket descriptor
 * @result string that contains human readable information
 *         about the client who has initiated the network connection
 *         and the server who has "picked up" the network connection
 */
inline std::string connection_info(const int socket_descriptor)
{
 // outputs meta-data for socket_descriptor
 std::ostringstream ostr;

 struct sockaddr sa;
 socklen_t sl = sizeof(sa);

 const int retval1 = getpeername(socket_descriptor, &sa, &sl);
 ostr << "connection from ";
 switch (retval1)
  {
    case 0       : ostr << sa; break;
    case EBADF   : ostr << "(invalid descriptor)"; break;
    case ENOTSOCK: ostr << "(file, not a socket)"; break;
    default : ostr << "(unable to perform getpeername)";
  }

 const int retval2 = getsockname(socket_descriptor, &sa, &sl);
 ostr << " to ";
 switch (retval2)
  {
    case 0       : ostr << sa; break;
    case EBADF   : ostr << "(invalid descriptor)"; break;
    case ENOTSOCK: ostr << "(file, not a socket)"; break;
    default : ostr << "(unable to perform getsockname)";
  }
 return ostr.str();
}

/*!
 * @param ub a unix_buffer (handling a network connection)
 * @result string that contains human readable information
 *         about the client who has initiated the network connection
 */
inline std::string peer_info(const unix_buffer& ub)
{
 // outputs meta-data for unix_buffer to string
 return peer_info(ub.get_descriptor());
}

/*!
 * @param ub a unix_buffer (handling a network connection)
 * @result string that contains human readable information
 *         about the server who has "picked up" the network connection
 */
inline std::string socket_info(const unix_buffer& ub)
{
 // outputs meta-data for unix_buffer to string
 return socket_info(ub.get_descriptor());
}

/*!
 * @param ub a unix_buffer (handling a network connection)
 * @result string that contains human readable information
 *         about the client who has initiated the network connection
 *         and the server who has "picked up" the network connection
 */
inline std::string connection_info(const unix_buffer& ub)
{
 // outputs meta-data for unix_buffer to string
 return connection_info(ub.get_descriptor());
}

#endif
