/* Faktorisierungsprogramm
   written 1997-2000, 2002-2004 by Thorsten Reinecke
   version 2.94: 2004-01 by Thorsten Reinecke
   
   Fr den mathematischen Hintergrund siehe:
   Riesel, Hans: "Prime Numbers and computer methods for factorization", 2nd ed.,
   Boston; Basel; Stuttgart: Birkhuser, 1994
    
   Das Programm setzt einige der dort beschriebenen Algorithmen (in freier
   Variation) um.
   Zunchst findet eine Probedivision fr kleine Primzahlen statt.
   Anschlieend wird mit der Pollard-rho-Methode nach weiteren kleineren
   Faktoren gesucht.
   Falls um eins verminderte Faktoren ausschlielich in kleinere Teiler
   zerfallen, so werden diese mittels der Pollard-Phi-Methode entdeckt.
   
   Wenn die bis dahin verbleibende Zahl nachweislich zusammengesetzt
   (also nicht pseudoprim) ist, wird eine Faktorisierung durch eine
   Variation des multipolynomialen Quadratischen Siebs versucht.
   
   (vgl. Robert D. Silverman: The Multiple Polynomial Quadratic Sieve,
    Mathematics Of Computation, Vol.48, January 1987, p. 329-339)

   Die dort beschriebene Large Prime Variante verwende ich ebenfalls.
   Zustzlich benutze ich aber Large Primes als dynamische Faktoren zur
   dynamischen Erweiterung der (usrprnglich statischen) Faktorbasis, d.h.
   ich siebe zustzlich mit ihnen. Damit das trotz ihrer Gre effizient
   geschehen kann, benutze ich Priority Queues (z.B. Fibonacci-Heaps).
   Diese Methode ist auf eigenem Mist gewachsen; in der von mir studierten
   Literatur konnte ich keinen Hinweis auf hnliche Methoden finden, die
   die Faktorbasis dynamisch erweitern. -- Zustzlich zu diesem Verfahren
   habe ich mich durch ein als "Double Large Prime"-Variante bekanntes
   Verfahren inspirieren lassen. Dieses ist teilweise implementiert und an
   meine dynamische Erweiterung angepat.
   Eine Zyklensuche zur Konstruktion von vollstndig in der Faktorbasis
   aufgehenden Relationen aus Double Large Primes (LP1,LP2 - LP2,LP3 - ...
   ... - LPk,LP1) ist nunmehr ebenfalls implementiert.
   Weitergehende Verbesserungen sind allerdings noch nicht implementiert:
   Eine Absplittung dynamischer Faktoren aus "triple large primes"
   unter Zuhilfenahme von "double large primes"-Zyklen gerade bei der
   Faktorisierung sehr groer Zahlen drfte z.B. ebenfalls sehr ntzlich
   sein, da sie die dynamische Faktorbasis vergrert...

   Das durch die gesiebten Relationen entstehende lineare Gleichungssystem
   wird dabei simultan mit den eingefgten Relationen gelst.
   Um Speicherplatz zu sparen, die Korrektheit bertragener Relationen beim
   verteilten Sieben einfacher berprfen zu knnen sowie triviale Lsungen
   schneller zu entdecken, habe ich die Relationen normiert:
   Jede Relation hat bei mir nun die Form 
    "enthaltene Faktorbasiselemente mit ungeradem Exponenten" * x^2 = 1 (mod n),
   wobei x den bereits ermittelbaren Quadratwurzelanteil darstellt.
   Mehrfachpotenzen der enthaltenen Faktorbasiselemente werden bei
   (ganzzahliger) Halbierung des Exponenten zu x hinzumultipliziert. -- Wenn
   schlielich eine Relation der Form x^2=1 entstanden ist, so kann ein
   (hoffentlich nichttrivialer) Faktor ber gcd(x-1,n) gefunden werden.

   Es ist auch eine Implementierung elliptischer Kurven zur Ermittlung von
   Faktoren "gemigter Gre" enthalten. Dies ist das einzige Verfahren,
   mit dem sich momentan auch bei sehr groen Zahlen nichttriviale Faktoren
   entdecken lassen (da seine Laufzeit von der Gre des entdeckten Faktors
   abhngig ist).

   In seiner jetzigen Fassung sollte man mit dem Programm
   Zahlen bis um 100 Dezimalstellen in ertrglicher Zeit faktorisieren knnen.
  
   [ 
     Nachtrag Juni 2003:

     Bei greren Zahlen lassen sich mit den elliptischen Kurven immerhin
     nichttriviale Faktoren absplitten, z.B. als Dauerproze im Hintergrund:
     P190169 (480 Dezimalstellen); entdeckter Faktor:
     9820062895352727932801465820946503529 (37) [ecm,sigma=272319096]

     Fr die Version 2.90 habe ich einige Testlufe mit folgender
     Konfiguration gestartet: 1x Athlon 1.2 Ghz + 1x Athlon 1.75 Ghz
      + 1x Pentium III 1.0 Ghz + Pentium MMX 166 Mhz.
     Einige dieser Rechner waren zeitweise ber WAN (DSL-Anschlu) angebunden;
     elliptische Kurven wurden fr den Testlauf deaktiviert.
     10^93+3 wurde in knapp 20 Stunden faktorisiert.
     10^95+3 bentigte rund 3 Tage.

     Nachtrag September 2003:
     Im kleinen Netzwerk bentigte die Version 2.92 fr die Faktorisierung einer
     107-stelligen Lucaszahl mit dem Siebverfahren ca. 5 Wochen Rechenzeit.
     (Die Faktorisierung wurde dabei als reiner Hintergrundprozess betrieben.)
   ]
   
   Es fehlt allerdings noch eine ganze Reihe an Optimierungen und Algorithmen.
   Fr ein effizienteres Sieben wre vielleicht noch das "Number-Field-Sieve"
   sinnvoll. (Siebverfahren faktorisieren grundstzlich in Abhngigkeit von
   der Gre der zu faktorisierenden Zahl; sie sind besonders effizient,
   wenn die Zahl lediglich zwei Primteiler hat und diese von der Stelligkeit
   her nicht allzuweit voneinander entfernt liegen.)

   Ob das simultane Lsen des linearen Gleichungssystems sinnvoll ist, oder
   ob man vielleicht effizientere Methoden (z.B. Wiedemann-Verfahren) nach
   Abschlu der Siebphase anwendet, hngt im wesentlichen vom Speicherbedarf
   des Gleichungssystems ab. Beim verteilten Sieben hat das simultane Lsen
   des Gleichungssystems zumindest den Vorteil, parallel zur Siebphase abzu-
   laufen. 
*/

/*! @file
 * @brief
 * main implementation file of "Faktorisierungsprogramm"
 */


#ifdef AUTODETECTION
 // autodetection for gcc-compiler
 #warning "autodetection enabled..."
 #ifdef __GNUG__
  #if __GNUC__ > 2
  #else
    #error "Please compile again with a newer version of GCC (version >= 3.2.x, better >=3.3.x)"
  #endif

  #if #cpu (i386)
   #ifndef ASM386
    #warning "activating i386-inline assembler code"
    #define ASM386
    #ifdef ASMATHLON
     #warning "using athlon-inline assembler code"
     #ifndef ASM386_CMOV
     #warning "activating ASM386-CMOV usage, because ATHLON supports it :)"
     #define ASM386_CMOV
     #endif
    #endif
   #endif
  #else
   #ifdef ASM386
    #warning "deactivating i386-inline assembler code"
    #undef ASM386
    #undef ASMATHLON
   #endif
  #endif
  #if !defined (unix) && defined(USE_NETWORK)
    #error "network support only for Unix-systems" 
  #endif
 #endif
#endif /* autodetection */

#if (__GNUC__ == 3) && (__GNUC_MINOR__ == 4)
 #warning "enabling gcc-3.4-experimental workarounds"
 #define gcc3_4_workaround
 // Have to wait for he stable gcc-3.4...
 // The old code seems to work, so: unless it is necessary, we do not change it.
 // Workaround will be changed to default, if gcc-3.4 is stable...
#endif


using namespace std;


#ifndef USE_NETWORK
 #ifdef IS_SERVER
  #error "Illegal configuration: SERVER and no USE_NETWORK"
 #endif
#else
 #ifndef IS_STANDALONE
  const int server_port = 12364;
  const int server_port_tries = 5;
  #ifdef IS_SERVER
   int server_socket;
  #endif
  #include "unix_buffer.H"
 #endif
#endif

#ifndef IS_SERVER
 #ifndef IS_CLIENT
  #undef IS_STANDALONE
  #define IS_STANDALONE
 #endif
#endif

#ifdef IS_SERVER
 #ifdef IS_CLIENT
  #error "Illegal Configuration SERVER-CLIENT"
 #else
  #ifdef IS_STANDALONE
   #error "Illegal Configuration SERVER-STANDALONE"
  #endif
 #endif
#endif

#ifdef IS_CLIENT
 #ifdef IS_STANDALONE
  #error "Illegal Configuration CLIENT-STANDALONE"
 #endif
#endif

#if defined(USE_FIBHEAP) && defined(USE_FAKEHEAP)
 #error "defines USE_FIBHEAP and USE_FAKEHEAP exclude each other!"
#endif

#ifdef REACT_ON_SIGUSR
 // a little class that catches signal SIGUSR1 and SIGUSR2
 // and stores these flags...
 // can be useful to abort some long-running algorithms
 // without aborting the program or altering configuration parameters.
 #include "usr_signals.cc"
 Cusr_signal_proxy USRSignalHandler;
#endif

extern "C"
{
 #include <unistd.h>
}

#include <cstdio>
#include <cstdlib>
#include <string>
#include <iostream>
#include <iomanip>
#include <fstream>
#include <sstream>
#include <cmath>
#include <gmp.h>
#include "myBitString.H"
#include <list>
#include <set>
#include "utils.H"

inline unsigned long int mpz_remainder_ui(const mpz_t n, const unsigned long int d)
{
  // refer gmp manual:
  // "unsigned long int mpz_fdiv_ui(mpz_t n, unsigned long int d)" returns
  // the remainder of the division; except for not needing any mpz_t result
  // value it is identical to
  // "unsigned long int mpz_mod_ui(mpz_t t, mpz_t n, unsigned long int d)".
  
  // I find the name "mpz_fdiv_ui" too much ambiguous.
  // Defining "unsigned long int mpz_mod_ui(mpz_t n, unsigned long int d)"
  // would be somewhat more straight forward, but it is also risky, because
  // one might assume a real gmp-function and not a wrapper function.
  // mpz_remainder_ui can easily be detected as a wrapper function just
  // by grepping the sources...
  return mpz_fdiv_ui(n,d);
}


#if defined(mpz_odd_p)
 // replace mpz_odd_p / mpz_even_p macros with inline-C++
 // (better type checking + more explicite casts) 

#undef mpz_odd_p
#undef mpz_even_p

//! equivalent to mpz_odd_p macro, but more strict
inline int mpz_odd_p(const mpz_t z)
{
  return static_cast<int>(z->_mp_size != 0) & static_cast<int>(z->_mp_d[0]);
}

//! equivalent to mpz_even_p macro, but more strict
inline int mpz_even_p(const mpz_t z)
{
  return !mpz_odd_p(z);
}

#endif


struct myintcompare {
  inline const bool operator() (const int i1, const int i2) const
   {
     return i1 < i2;
   }
};
typedef set<int, myintcompare> intset;


#include <vector>
#include <stack>

#ifdef USE_NETWORK
 extern "C"
 {
  #include <sys/utsname.h>
  #include <pthread.h>
 }
 #include <csignal>
 #include "mutex.H"
 #ifdef IS_CLIENT
  extern "C"
  {
   #include <sys/wait.h>
   #include <unistd.h>
  }
 #endif
#endif


const string RecoveryFile         = "recovery.dat";
const string FactorizationsFile   = "factorizations.txt";
const string StaticRelationsFile  = "static_relations.dat";
const string SpecialRelationsFile = "special_relations.dat";
      string DynamicRelationsFile = "dynamic_relations.dat"; // default, may be overwritten by entry in configfile
      string ConfigFile           = "qsieve.cfg"; // default, may be overwritten by getenv("QSIEVE_CFG")


// Streams fr die Speicherung der Relationen und sonstiger Daten
// zur Verfgung stellen
// *************************************************************************
// **** wichtig:
// **** Wenn ostream und istream durch denselben filebuf fest miteinander
// **** gekoppelt sind, darf/kann man sich *nicht* darauf verlassen, da
// **** tellp() und tellg() voneinander unabhngige Funktionen sind!
// **** *Jede* Schreiboperation kann dann *auch* den Wert von tellg()
// **** verndern und *jede* Leseoperation den von tellp()!
// ************************************************************************* 

// Fr die Relationen zu den Specialfactors
// streams fr das Datenfile zur Verfgung stellen:
filebuf SpecialRelations_buffer;
ostream SpecialRelations_to_file (&SpecialRelations_buffer);
istream SpecialRelations_from_file (&SpecialRelations_buffer);

// Fr die Relationen der dynamischen Faktorbasis
// streams fr das Datenfile zur Verfgung stellen:
filebuf DynamicRelations_buffer;
ostream DynamicRelations_to_file (&DynamicRelations_buffer);
istream DynamicRelations_from_file (&DynamicRelations_buffer);

// Fr die Relationen der statischen Faktorbasis
// streams fr das Datenfile zur Verfgung stellen:
filebuf StaticRelations_buffer;
ostream StaticRelations_to_file (&StaticRelations_buffer);
istream StaticRelations_from_file (&StaticRelations_buffer);

// Fr Recovery
filebuf Recovery_buffer;
ostream Recovery_to_file (&Recovery_buffer);
istream Recovery_from_file (&Recovery_buffer);

#ifdef IS_CLIENT
string communication_name;
ofstream communication_stream;
#endif

#ifndef IS_CLIENT
// Fr das Speichern der gemachten Faktorisierungen
ofstream Factorization_to_file;
/* Bemerkung:
   Auf diesem Stream werden alle mit diesem Programm gemachten Faktorisierungen
   protokolliert. Die zu faktorisierenden Zahlen sowie alle gefundenen Faktoren
   werden angehngt.
   Das zugehrige File sollte im Bedarfsfall gelscht werden, wenn die Daten
   nicht mehr bentigt werden.
*/
#endif


typedef unsigned short int FBsizetype; // Typ der Faktorbasisgre
  // Wenn die Faktorbasis maximal 65535 Elemente haben soll,
  // sollte man "unsigned short int" whlen, um Speicherplatz zu sparen.
  // Wenn Speicherplatz keine Rolle spielt, ist "int" eine gute Wahl (denn
  // dann gibt es u.U. weniger implizite Typkonvertierungen).
  // Hinweis:
  // "FBsizetype" hat *nur* mit der Anzahl der Primzahlen in der statischen
  // Faktorbasis (also den Indices auf die darin enthaltenen Primzahlen),
  // aber *nichts* mit dem Typ (und der Gre) der Primzahlen aus der
  // statischen Faktorbasis selbst zu tun!

const FBsizetype MAX_Primbasis_size = 25000; /* Maximale Gre der Primzahlbasis */ 
/* Hinweis: Je grer die zu faktorisierende Zahl ist, desto grer
   sollte prinzipiell auch die Primzahlbasis sein, da dann die Relationen
   einfacher gefunden werden knnen.
   Aber: Das lineare Gleichungssystem sprengt bei groer Primzahlbasis
   den Speicher!
   Fr kleinere zu faktorisierende Zahlen wirkt sich dagegen eine grere
   Primzahlbasis verlangsamend aus, da fr ein greres Gleichungssystem
   natrlich auch mehr Relationen gesiebt werden mssen.
   Je nach Geschwindigkeit des Rechners, des verfgbaren Haupt- und
   Hintergrundspeichers (und auch dessen Zugriffszeiten) knnen die
   ttschlich zu verwendenden Werte stark variieren. Sie sind deshalb
   experimentell zu ermitteln.
   Die Maximalgre wird durch MAX_Primbasis_size festgelegt und steht
   damit bereits zur Kompilierzeit fest.
   Bei einer Speicherkapazitt von 32 MB (realem) Hauptspeicher
   hat sich 20000 als brauchbarer Wert erwiesen.
*/

#include "Tfactor.H"
#include "qsieve.H"
#include "mpqsPolynom.H"

const int Specialfactor_Sieb_Threshold  = 1000000000; /* bis hierhin wird auch mit
							 Dynamic-Factoren gesiebt */


struct TSpecialFactorRelation {
  CmpqsFactor factor;
  streampos fpos;

  inline bool operator() (const TSpecialFactorRelation &t1, const TSpecialFactorRelation &t2) const
    { 
      return t1.factor<t2.factor;
    }
};


class TDynamicFactorRelation
{
  /* Bemerkung:
     Falls factor nicht allzugro ist (im Augenblick ist dies als Obergrenze der
     int-Bereich), dann kann auch mit den Dynamic-Factors gesiebt werden.
     Nachfolgend die dazu erforderlichen Daten...
  */
public:
  int factor; // factor in int-Darstellung
  int SQRT_kN;
  streampos fpos;
  inline bool operator() (const TDynamicFactorRelation &t1, const TDynamicFactorRelation &t2) const
    { 
      return t1.factor<t2.factor;
    }
};


struct TSieb_Delta
{
  int delta;
  int factor;
  inline bool operator< (const TSieb_Delta &v) const
    {
#if defined (USE_FIBHEAP)
      return (delta < v.delta);
#else
      return (delta > v.delta);
#endif
      // delta > v.delta , if queue is used
      // delta < v.delta , if fibonacci-heap is used
    }
};

#ifndef IS_SERVER
#ifdef USE_FIBHEAP
 #include "fibheap.H"
 FibHeap <TSieb_Delta> Sieb_Delta_Heap, Sieb_Delta_QuadrateHeap;
 #include <queue>
#elif defined(USE_FAKEHEAP)
 #include "fakeheap.H"
 FakeHeap <TSieb_Delta> Sieb_Delta_Heap;
 #include <queue>
 priority_queue<TSieb_Delta> Sieb_Delta_QuadrateHeap;    
#else
 #include <queue>
 priority_queue<TSieb_Delta> Sieb_Delta_Heap, Sieb_Delta_QuadrateHeap;
#endif
#endif

mpz_t n, /* zu faktorisierende Zahl */
      kN, /* das mit einem gnstigen Vorfaktor multiplizierte n
	     (nachdem n um einfach zu findende Faktoren reduziert wurde) */
      r; /* Quadratwurzel von n (oder bei optimierter Fassung k*n) */

int Vorfaktor_von_kN; // der noch zu ermittelnde Vorfaktor fr kN (kN=Vorfaktor_von_kN*n)


FBsizetype Primbasis_size = 5000; // default fr zu verwendende Gre der Primzahlbasis
/*  Sinnvolle Werte fr Faktorbasen liegen zwischen 1000 und 50000
    (je nach verfgbarem Speicher und der zu faktorisierenden Zahl).
    Der Wert wird normalerweise spter durch die Prozedur "tune_parameters"
    angepat.
*/

int Primzahlen[MAX_Primbasis_size]; /* fr eine effiziente Probedivision
				       alle Primzahlen der Faktorbasis hier
				       akkumulieren */


int SQRT_kN_der_Primzahlen[MAX_Primbasis_size];
int Delta_der_Primzahlen[MAX_Primbasis_size][2]; /* fr den effizienten Zugriff
						    werden die pro Siebintervall berechneten
						    Werte hier abgelegt */
 
const int FB_max_weitere_Potenzen = 1000; // maximale Anzahl weiterer Primzahlpotenzen, mit denen gesiebt wird
int anz_weitere_Primzahlpotenzen = 0; // tatschliche Anzahl (wird im Programm ermittelt)
int FB_maxQuadrate = 0; /* tatschliche Anzahl der Quadrate,
			   mit denen normal gesiebt wird (wird im Programm ermittelt) */
int Primzahlpotenzen[FB_max_weitere_Potenzen];
int SQRT_kN_der_Primzahlpotenzen[FB_max_weitere_Potenzen];
int Delta_der_Primzahlpotenzen[FB_max_weitere_Potenzen][2];


CRelation* GLS[MAX_Primbasis_size] = {NULL};
/* Gleichungssystem besteht aus Zeigern auf Relationen (Gleichungszeilen),
   das mit Null-Pointern initialisiert ist, da anfangs noch keine Relationen
   vorliegen */

CmpqsPolynom Polynom; // Objekt zur Verwaltung der Polynomberechnung fr multipolynomiales Sieb



/* Zur Erluterung der Begrifflichkeit von Dynamic-Factors/Special-Factors:
   Ich verwende momentan eine modifizierte Multi-Large-Prime-Variante:
   Dynamic-Factors entsprechen der Large-Prime-Variante. Aber ich siebe mit jedem neu entdeckten
   dynamischen Faktor zustzlich...
   Special-Factors sind zusammengesetzte Restfaktoren (wie sie in der Double-Large-Prime-Variante
   auftreten). Es wird einerseits versucht, Special-Factors durch neu aufgenommene
   Dynamic-Factors zu splitten, um daraus weitere Dynamic-Factors zu gewinnen.
   Andererseits wird eine Zyklensuche auf diesen durchgefhrt, um weitere
   Relationen zu gewinnen.
   In einer spteren Implementierung knnte man die Zyklensuche noch
   auf Hyperkantenzyklen hherer Large-Primes erweitern.
*/
typedef set<TSpecialFactorRelation,TSpecialFactorRelation> TSpecial_Factor_Relations;
TSpecial_Factor_Relations Special_Factor_Relations;
typedef set<TDynamicFactorRelation,TDynamicFactorRelation> TDynamic_Factor_Relations;

#ifdef IS_SERVER
class CServerDynamic_Factor_Relations : public TDynamic_Factor_Relations
{
 private:
  mutable CMutex monitor_mutex;
  vector<int> DynamicFactorList;
 public:
  inline void insert(const TDynamicFactorRelation& x)
   {
     if (TDynamic_Factor_Relations::insert(x).second)
      {
        monitor_mutex.lock();
        DynamicFactorList.push_back(x.factor);
        monitor_mutex.unlock();
      }
     else
      {
        cerr << "CServerDynamic_Factor_Relations::insert duplicate (rejected)!" << endl;
        exit(1);
      }
   }
  inline const int monitoredSize() // const /* const attribute could be dangerous due to multihreading!? */
   {
     monitor_mutex.lock();
     const int s=DynamicFactorList.size();
     monitor_mutex.unlock();
     return s;
   }
  inline const int operator[] (const int i) // const /* const attribute could be dangerous due to multihreading!? */
   {
     monitor_mutex.lock();
     const int x=DynamicFactorList[i];
     monitor_mutex.unlock();
     return x;
   }
};
CServerDynamic_Factor_Relations Dynamic_Factor_Relations;
#else
TDynamic_Factor_Relations Dynamic_Factor_Relations;
#endif

double Factor_Threshold = 3.0; /* Exponent, der grere Faktoren ber die
				  Faktorbasis hinaus erlaubt.
				  Sinnvolle Werte: 1.0 bis 3.0
				  1.0: keine zustzlichen Faktoren auerhalb der Faktorbasis 
				  bis 2.0: Primfaktoren auerhalb der Faktorbasis bis
				  zum Quadrat des grten Faktors der FB
				  ber 2.0: Faktoren bis zu grter Faktor der Faktorbasis
				  hoch Factor_Threshold.
			       */


const int Siebgroesse = 100000;
/* In (physikalischen) Teilintervallen von "Siebgroesse" werden die Relationen
   im (logischen) Siebintervall [-M,M] (siehe auch Deklaration von M) gesiebt.
   Es handelt sich deshalb um eine bereits zur Compilierungszeit feststehenden Konstante. 
   Hinweis: Je grer "Siebgroesse", desto effizienter geht das Aussieben
   der Relationen. Gleichzeitig erhht sich aber auch der Speicherbedarf.
   Der Einflu der Siebgre ist allerdings geringer als der der gewhlten
   Primzahlbasis. Ein Wert um 1 Million drfte daher vollkommen ausreichen.
   Andererseits kann mglicherweise im Cachespeicher des Rechners gesiebt
   werden, wenn die Siebgre klein gewhlt wird. Kleine Siebgren erhhen
   zwar den rechnerischen Overhead, aber dafr ist der Zugriff auf das Sieb
   mglicherweise deutlich schneller. Deshalb sollte man auch eine Siebgre
   um 100000 (bei 512K Cache) ausprobieren.
   Fr eventuellen optimierten Zugriff ber Assemblerbefehle sollte
   Siebgroesse*TSIEBELEMENTSIZE ein Vielfaches der Maschinenwortgre
   (bei 32-Bit-Rechnern also 4, bei 64-Bit-Rechnern 8) sein. (Das stellt
   keine wirkliche Einschrnkung dar, ermglicht es aber, das Sieb
   effizient zu initialisieren ohne Sonderflle beachten zu mssen. --
   Wenn Assembler-Code aktiviert wird, verlasse ich mich auf diesen Contraint!)
*/

int M = 1000000; // das zu jedem Polynom gehrige Siebintervall ist [-M,M]
/* M ist ein im Unterschied zu "Siebgroesse" ein logischer Wert, der weniger stark
   von der verwendeten Hardware abhngig ist. Die logische Intervallgre
   ist vor allem abhngig von der Gre der zu faktorisierenden Zahl zu whlen.
   Der hier festgelegte Wert ist also lediglich ein Defaultwert, der von der
   Konfiguration vor Beginn eines konkreten Faktorisierungslaufs (zur Laufzeit)
   modifiziert werden kann (und soll).
   Bemerkung: In multiple-polynomial-quadratic-sieve-Varianten sollte man
   mglichst kleine Siebintervalle whlen, da die relative Trefferquote von
   Relationen mit der Intervallgre abnimmt. (In 10 kleinen Intervallen ist
   die Chance, eine Relation zu finden, hher als in einem 10mal so groen
   Intervall).
   Gegen kleine Intervalle spricht jedoch, da Kosten fr das Wechseln
   der Polynome, Neuinitialisierungen und Wurzelberechnungen anfallen.
*/


int Sieboffset = -M; /* Im allgemeinen wird das fr ein Polynom zu siebende Siebintervall 
			grer sein als die (physikalisch optimale) Siebgre.
			Das Siebintervall ist daher in Teilstcken zu sieben. Hierfr wird der
			Sieboffset bentigt, der den Startpunkt des jeweiligen Teilsiebs angibt. */

// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
// !!! WICHTIG !!! #define und typedef mssen einander entsprechen !!!
// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
#define TSIEBELEMENTSIZE 1 /* 1 = signed char, 2 = short signed int, 4 = signed int, 0 = kein assembler inline code */
// obiges define sorgt bei Verwendung der ASM386-Direktive fr inline-assembler-code in der Siebroutine

#if TSIEBELEMENTSIZE == 0
 typedef signed char TSiebElement; // must be signed!
 #define log_Multiplikator 1 /* Siebwert wird nach damit nach TSiebelement konvertiert */
 #ifdef ASM386
  #warning "assembly code is disabled for sieving!"
 #endif
#elif TSIEBELEMENTSIZE == 1
 typedef signed char TSiebElement; // must be signed!
 #define log_Multiplikator 1 /* Siebwert wird nach damit nach TSiebelement konvertiert */
#elif TSIEBELEMENTSIZE == 2
 typedef signed short int TSiebElement; // must be signed!
 #define log_Multiplikator 16 /* Siebwert wird nach damit nach TSiebelement konvertiert */
#elif TSIEBELEMENTSIZE == 4
 typedef signed int TSiebElement; // must be signed!
 #define log_Multiplikator 256 /* Siebwert wird nach damit nach TSiebelement konvertiert */
#else
 #error "invalid value for TSIEBELEMENTSIZE"
#endif

/* Je kleiner der Typ von TSiebElement ist, desto 
    a) kleiner ist der fr das Sieb bentigte Speicher,
    b) schneller kann deshalb gesiebt werden.
   Allerdings ist die Granularitt zum Erkennen von Relationen bei
   kleineren Datentypen grber und deshalb schlechter.
   Auerdem sind prozessorspezifische Effekte (z.B. Alignment-Penalties)
   zu bercksichtigen!
   In Frage kommen:
    - signed char (1 Byte)
    - short signed int (2 Byte)
    - signed int (4 Byte)
    - float, double (Fliekommatypen, mehrere (>3) Bytes)
   Die Bytegren knnen zwischen Systemen variieren. Ich habe aber nicht
   geprft, ob das Programm berhaupt auf mit einer Systemeinstellung von
   int < 4 Byte, short int < 2 Byte lauffhig ist.

   Wichtiger Hinweis: Wenn man Fliekommatypen als TSiebElement verwenden will (was ich nicht mehr empfehle),
   dann sind im Programm die "ceil"-Aufrufe fr TSiebElement kontraproduktiv und sollten entfernt werden.
   Aber es gibt keinen vernnftigen Grund mehr, mit Fliekommatypen zu sieben.
*/

// #define log_Multiplikator <VALUE> /* Siebwert wird nach damit nach TSiebelement konvertiert */
/*
  -> refer TSIEBELEMENTSIZE
  Da die Vorkommastellen der natrliche Logarithmen von Primzahlen
  zu hohe Ungenauigkeiten beim Sieben verursachen, werden sie noch
  mit log_Multiplikator multipliziert, um eine hhere Genauigkeit
  zu erreichen.
  Der Multiplikator kann auf 1 gesetzt sein, wenn TSiebElement Fliekommatyp
  ist. Ansonsten sollte man ihn mglichst gro whlen, und zwar so, da
  der Threshold der blicherweise zu faktorisierenden Zahlen noch in den
  Datentyp TSiebElement pat.
  Damit durch Abrundungen keine Relationen durch die Lappen gehen, werden
  aber beim Sieben grundstzlich aufgerundete Werte abgezogen, so da ein
  kleiner Multiplikator hier allenfalls ein zu engmaschiges Sieb erzeugt.
  Die exakten Auswirkungen verschiedener Einstellungen wren allerdings
  noch genauer zu untersuchen.
  Wenn man nicht genau wei, was man tut, sollte man eher TSiebElement
  "hhertypen". Der Multiplikator sollte dann deutlich ber 1 liegen (10 - 1000).
*/

TSiebElement* Sieb; // Sieb (noch zu allokierendes Feld von TSiebelementen)

TSiebElement log_Primzahlen[MAX_Primbasis_size]; /* Hier stehen die konvertierten Logarithmen
						    der Primzahlen aus der Faktorbasis */
TSiebElement log_Primzahl_der_Primzahlpotenzen[FB_max_weitere_Potenzen];
/* fr weitere Primzahlpotenzen stehen hier die konvertierten Logarithmen der Primzahlen */

bool collecting_phase_beendet = false; /* switch, der die "aktive" und "passive" Phase des Siebens
					  bestimmt:
					  Wenn gengend Special-Relations und Dynamic-Relations gesammelt
					  sind, um ausreichend viele Static-Relations zu erhalten, wird
					  das Aufsammeln weiterer Special- und Dynamic-Relations durch
					  das Setzen von collecting_phase_beendet=true unterbunden. */

int Fuellung_GLS = 0; // Fllung des Gleichungssystems mit Relationen
int Erfolgsaufteilung[7] = {0,0,0,0,0,0,0}; /* fr statistische Auswertung, inwieweit die dynamische
					   Factorbase fr das Finden der Relationen
					   verantwortlich ist:
					   [0]->gar nicht beteiligt,
					   [1]-> mit einem Faktor beteiligt
					   [2]-> mit zwei Faktoren beteiligt
					   [3]-> mit drei Faktoren beteiligt
					   [4]-> mit 4 Faktoren beteiligt
					   [5]-> mit 5 Faktoren beteiligt
					   [6]-> mit mehr als 5 Faktoren beteiligt */

int Special_hit = 0;
int Special_to_dynamic_Factor_hit = 0;
int gefundene_DLP_Zyklen = 0;

int biggest_Prime_in_Factorbase;



class SiebRegler
{
 private:
  static const int PrimzahlStartIndex = 5;
/* ab diesem Primzahlindex wird gesiebt.
   Der Wert mu grer als 1 sein, da die ersten beiden Faktoren
   der Primzahlbasis -1 und mglicherweise die 2 sind. Fr diese
   beiden Zahlen sowie dem Vorfaktor von kN kann nicht (jedenfalls nicht
   so einfach) gesiebt werden.
   Fr ein schnelles, aber unsauberes dirty-sieving kann auch ein
   grerer Wert gewhlt werden. Aber zu hohe Werte lassen dann auch
   viele Relationen durchs Sieb fallen... */

  static TSiebElement log_Threshold; // wird aber erst durch "Grundinitialisierung_des_Siebs" initialisiert
  static int _sieben_ab_Primzahlindex;

  static void IncreaseDirtyness(void)
   {
     // Threshold absenken und Index fr erste zu siebende Zahl inkrementieren
     if (_sieben_ab_Primzahlindex>(Primbasis_size>>2)) return; // das geht leider nicht...
     log_Threshold-=log_Primzahlen[_sieben_ab_Primzahlindex];
     ++_sieben_ab_Primzahlindex;
     cout << "Dirtyness now " << _sieben_ab_Primzahlindex << endl;
   }

  static void DecreaseDirtyness(void)
   {
     // Threshold hochsetzen und Index fr erste zu siebende Zahl dekrementieren
     if (_sieben_ab_Primzahlindex<=PrimzahlStartIndex) return; // das geht leider/wollen wir nicht...
     log_Threshold+=log_Primzahlen[--_sieben_ab_Primzahlindex];
     cout << "Dirtyness now " << _sieben_ab_Primzahlindex << endl;
   }

  static void RequestDirtyness(const signed int Level)
   {
     static signed int Pegel = 0;
     Pegel+=Level;
     if (Pegel>1000000)
      {
        Pegel=0;
        IncreaseDirtyness();
      } else
     if (Pegel<-1000000)
      {
        Pegel=0;
        DecreaseDirtyness();
      }
   }

 public:
  static void SiebSchwellwert_kalkulieren(void);
  static const TSiebElement GetLogThreshold(void) { return log_Threshold; }
  static const int sieben_ab_Primzahlindex(void) { return _sieben_ab_Primzahlindex; }
  static const bool Hit_after_DirtyKorrektur(const signed int Siebpos, TSiebElement Value);

  static inline void RequestMoreDirtyness(const signed int Level = 1000)
   {
     RequestDirtyness(Level);
   }
  static inline void RequestLessDirtyness(const signed int Level = 50)
   {
     RequestDirtyness(-Level);
   }

  friend void Sieb_initialisieren(void);
};

int SiebRegler::_sieben_ab_Primzahlindex = SiebRegler::PrimzahlStartIndex;


#include "modulo.cc" // Modulo-Operationen fr unsigned int
                     // als .cc included fr optimierende Compiler
using namespace numtheory;


//! contains wrapped I/O for mpz_t (multiple precision numbers)
namespace my_mpz_wrapper
{

const int mpzbase_f = 36; // mpz-Ein/Ausgabeformat fr Fileoperationen
// Fr lesbare Ein- und Ausgaben wird weiterhin Dezimalsystem verwendet  

inline ostream& operator << (ostream& ostr, const mpz_t value)
{ // Ausgabefunktion fr mpz_t - Zahlen (um die hassenswerten expliziten Konvertierungen zu minimieren...)
  char* str = new char [mpz_sizeinbase(value,10)+2]; mpz_get_str(str, 10, value);
  ostr << str << " (" << strlen(str) << ")";
  delete [] str;              
  return ostr;
}

inline istream& operator >> (istream& istr, mpz_t value)
{ // Eingabefunktion fr mpz_t - Zahlen
  string s;
  istr >> s;
  if (mpz_set_str(value, s.c_str(), 10))
   {
     MARK;
     cerr << "invalid number read." << endl;
     cerr << "given: *" << s << "*" << endl;
     cerr << "mpz_t: *" << value << "*" << endl;
     istr.setstate(ios::failbit);
     exit(1);
   } else  istr >> s; // Sizeinbase berlesen
  return istr;
}

} // end of namespace my_mpz_wrapper

using namespace my_mpz_wrapper;

// einige Parameter fr easy-Faktorisierung (defaults)
bool SkipFermat    = false; // Fermatmethode versuchen
bool SkipPhi       = false; // Pollard-Phi-Methode versuchen
bool SkipFibonacci = false; // Fibonacci-Methode versuchen
bool SkipECM       = false; // elliptic curve method versuchen
int rho_Phase      =   150000; // Grundinitialisierung fr Rho-Phase
int phi_Phase1     =   500000; // fr phi-Methode Phase 1
double phi_Phase2  = 10000000.0; // fr phi-Methode Phase 2 (=isolierter Faktor)
int elcu_Phase1    =    20000; // fr elliptische Kurven Phase 1
double elcu_Phase2 =  2000000.0; // fr elliptische Kurven Phase 2
int elcu_Kurven    =      100; // Anzahl der zu probierenden elliptischen Kurven 

const int probab_prime_checks = 25; // #rounds for mpz_probab_prime


#include "Tfactor.cc"


void SpecialFactors_Zyklensuche()
{
  // Wenn man einen Zyklus von Large-Primes durch Double-Large-Primes
  // erzeugen kann, so neutralisieren sich die beteiligten Large-Primes und
  // man erhlt eine vollstndig in der statischen Faktorbasis aufgehende
  // Relation.

  if (Special_Factor_Relations.empty()) return; // es gibt nichts zu finden
  cout << "Starting cycle find for Special-Factor-Set" << endl;

  TSpecial_Factor_Relations arbeitsmenge;
  cout << "Size: " << Special_Factor_Relations.size() << ", " << arbeitsmenge.size() << endl;

  arbeitsmenge.swap(Special_Factor_Relations);
  // Hinweis: um nicht doppelten Speicherbedarf zu haben, werden die
  // Special-Factor-Relations in die Arbeitsmenge geschoben (und nicht kopiert)
  // es ist dann darauf zu achten, da sie von der Arbeitsmenge auch wieder
  // zurckgeschoben (und nicht nur gelscht) werden...
  cout << "Size: " << Special_Factor_Relations.size() << ", " << arbeitsmenge.size() << endl;

  intset knotenmenge;
  typedef list<TSpecialFactorRelation> Tkantenliste;
  Tkantenliste kantenliste;

  TSpecial_Factor_Relations::iterator pos;
  TSpecialFactorRelation SearchRelation;
  int rechts,links;

  while (!arbeitsmenge.empty())
   {
     knotenmenge.clear(); // wichtig! (wegen Schleifenwiederholungen)
     pos=arbeitsmenge.begin();
     TSpecialFactorRelation relation = *pos;
     kantenliste.push_back(relation);
     links=relation.factor.LP1();
     rechts=relation.factor.LP2();
     knotenmenge.insert(links);
     knotenmenge.insert(rechts);
     arbeitsmenge.erase(pos); Special_Factor_Relations.insert(relation);
     relation.factor.swap();
     arbeitsmenge.erase(relation); // alten Specialfactor entfernen (geswapte Kopie)
     Special_Factor_Relations.insert(relation);
     relation.factor.swap();

     //cout << "Starte neue Runde der Zyklensuche." << endl;
     while (!kantenliste.empty())
      {
        //for (Tkantenliste::iterator p=kantenliste.begin(); p!=kantenliste.end(); ++p)
        // cout << (*p).factor << "|";
        //cout << endl;
        //cout << "Suche nach " << links << endl;
	//cout << "Arbeitsmenge enthlt: " << endl;
        //for (pos=arbeitsmenge.begin(); pos!=arbeitsmenge.end(); ++pos)
        // cout << (*pos).factor << ",";
	//cout << endl;

        SearchRelation.factor.set_for_search(links);
        pos=arbeitsmenge.lower_bound(SearchRelation);
        if (pos!=arbeitsmenge.end() && (*pos).factor.DLP_enthaelt_Teiler(links) )
         {
           // links kann eine Kante eingefgt werden
           relation = *pos;
           arbeitsmenge.erase(pos); // alten Specialfactor entfernen
           Special_Factor_Relations.insert(relation);
           relation.factor.swap();
           arbeitsmenge.erase(relation); // alten Specialfactor entfernen (geswapte Kopie)
           Special_Factor_Relations.insert(relation);
           relation.factor.swap();
           links=relation.factor/links;
           if (knotenmenge.find(links)!=knotenmenge.end())
            { 
              // Zyklus gefunden!
	      ++gefundene_DLP_Zyklen;
	      cout << "DLP-cycle found... " << endl;
              // nun noch: 
              // Zyklus auswerten und erhaltene Relation ins Gleichungssystem
              // einfgen.
              CRelation *GL = new CRelation;
              GL->kombinieren(SpecialRelations_from_file,relation.fpos);
              mpz_mul_ui(GL->Delta,GL->Delta,links);
	      mpz_mod(GL->Delta,GL->Delta,n);
              rechts=relation.factor/links;
	      //cout << relation.factor;

              // Kantenliste bis zu der Position, wo der Wert "links"
              // erstmalig enthalten ist, durchgehen -> dort endet der Zyklus.
	      // dabei noch die Wurzel des Produkts der Specialfaktoren
              // hinzumultiplizieren (beim Zyklus sind dies alle beteiligten
              // Large Primes in einfacher Potenz), also die Werte "rechts"
	      unsigned int Zyklenlaenge=1;
              for (Tkantenliste::iterator p=kantenliste.begin(); p!=kantenliste.end(); ++p)
	       {
                 Zyklenlaenge++;
		 //cout << " - " << p->factor;
                 mpz_mul_ui(GL->Delta,GL->Delta,rechts);
	         mpz_mod(GL->Delta,GL->Delta,n);
                 GL->kombinieren(SpecialRelations_from_file,(*p).fpos);
                 if ((*p).factor.DLP_enthaelt_Teiler(links)) break;
		 rechts=(*p).factor/rechts;
               }
              //cout << endl;
              cout << "DLP-cycle size: " << Zyklenlaenge << endl;
	      GL->Kongruenzcheck(); // aus Sicherheitsgrnden...
     	      Relation_einfuegen(GL);
	    
  	      // Anschlieend eine verantwortliche (wir whlen die letzte Kante)
              // aus der Special-Relation-Menge entfernen:
              Special_Factor_Relations.erase(relation); // alten Specialfactor entfernen
              relation.factor.swap();
              Special_Factor_Relations.erase(relation); // alten Specialfactor entfernen (geswapte Kopie)
              relation.factor.swap();
              {
	        // nicht vergessen, die Relation auf File zu markieren!
                const streampos pos = SpecialRelations_to_file.tellp(); // Schreibposition merken
                SpecialRelations_to_file.seekp(relation.fpos); SpecialRelations_to_file << "U " << flush; // ungltig markieren
                SpecialRelations_to_file.seekp(pos); // wieder an die alte position
              }
 
              // Schlielich: Zyklensuche fortsetzen.
	      links=relation.factor/links;
              continue;
            }
           else
            {
              // kein Zyklus gefunden -> Kante und Knoten einfgen
              knotenmenge.insert(links);
              kantenliste.push_front(relation);
	      //cout << "Fge Kante " << relation.factor << " hinzu " << endl;
              continue;
            } 
         }
        // keine Kante zum linksseitigen Einfgen gefunden
        // -> Alternative 1: rechts einfgen (zum Zyklusfinden nicht notwendig)
        // -> Alternative 2: linksseitig eine Kante entfernen
	//cout << "Entferne Kante " << kantenliste.begin()->factor << endl;
        links=kantenliste.begin()->factor/links;
        kantenliste.pop_front(); // also linksseitig eine Kante entfernen
      }
   }
  cout << "Size: " << Special_Factor_Relations.size() << ", " << arbeitsmenge.size() << endl;
  cout << "cycle find finished." << endl;
}

bool SpecialFactorRelation_vermerken(const CmpqsFactor &DLP, CRelation *GL, const short int HitCount /* =0 */)
{
  // Special-Factor in die Special-Factor-Set eintragen (doppelt, damit
  // beide Large-Primes schnell zugreifbar sind)
  // Falls der Special-Factor schon vorhanden sein sollte, kombinieren,
  // so da man GL anschlieend ins Gleichungssystem eintragen kann.

  if (DLP.LP1()==DLP.LP2()) // Sonderfall: DoubleLargePrime=LargePrime^2
   {
     cout << "Spezialfall: DLP ist Quadrat!" << endl;
     Special_hit++;
     // immer beachten: Delta ist nicht das Quadrat, sondern wird quadriert,
     // daher zu Delta immer nur Wurzeln multiplizieren!
     mpz_mul_ui(GL->Delta,GL->Delta,DLP.LP1()); mpz_mod(GL->Delta,GL->Delta,n);
     return false;
   }
  
  TSpecialFactorRelation relation;
  relation.factor=DLP;

  const TSpecial_Factor_Relations::iterator pos = Special_Factor_Relations.find(relation);
  if (pos==Special_Factor_Relations.end())
   {
     relation.fpos=GL->speichern(SpecialRelations_to_file,relation.factor,HitCount);
     Special_Factor_Relations.insert(relation);
     relation.factor.swap();
     Special_Factor_Relations.insert(relation);

     // regelmig Zyklensuche auf den Special-Relations durchfhren:
     static int naechste_Zyklensuche = 50000;
     if (!--naechste_Zyklensuche)
      {
        naechste_Zyklensuche=50000;
        SpecialFactors_Zyklensuche();
      }
     return true;
   }
  else
   { 
     // Special-Factor ist bereits bekannt
     cout << "Special Hit durch bekannten Special Factor!" << endl;
     Special_hit++;
     mpz_t x;
     mpz_init(x); DLP.in_mpz_ablegen(x);
     mpz_mul(GL->Delta,GL->Delta,x); mpz_mod(GL->Delta,GL->Delta,n);
     mpz_clear(x);
     GL->kombinieren(SpecialRelations_from_file,(*pos).fpos);
     //Relation_einfuegen(GL); // sollte an anderer Stelle gemacht werden!
     return false;
   }
}


bool SpecialFactorRelation_testen(const CmpqsFactor &DLP)
{
  // Testen, ob Special-Factor erfolgreich gesplittet werden kann
  // true -> ja, false -> nein
  if (DLP.LP1()==DLP.LP2()) // Sonderfall: DoubleLargePrime=LargePrime^2
   {
     // cout << "Spezialfall: DLP ist Quadrat!" << endl;
     return true;
   }
  TSpecialFactorRelation relation;
  relation.factor=DLP;
  TSpecial_Factor_Relations::iterator pos = Special_Factor_Relations.find(relation);
  return (pos!=Special_Factor_Relations.end());
}


#ifndef IS_CLIENT

void Read_ConfigFile(void)
{
  ifstream in(ConfigFile.c_str());
  char eingabezeile[200];
  string zeile;
  if (in)
    {
      cout << "Reading configuration file" << endl; 
      while (!in.eof())
	{
	  in.getline(eingabezeile,sizeof(eingabezeile),'\n');
	  if (eingabezeile[0]=='#') continue; // Kommentarzeilen berlesen
	  zeile=eingabezeile;
          while (zeile[0]==' ') zeile.erase(0,1);
          if (zeile.empty() || zeile[0]=='#') continue; // leere Zeile/Kommentarzeile berlesen

          // Keywords verarbeiten
          istringstream is(zeile);
          string Keyword;
          is >> Keyword;
          if (Keyword=="DynamicRelationsFile")
           {
             string s;
             is >> s;
             if (s!="=")
              {
                cerr << "Error in " << ConfigFile  << "!" << endl;
                exit(1);
              }
             is >> DynamicRelationsFile;
             continue;
           }
          if (Keyword=="WorkDir")
           {
             string s;
             is >> s;
             if (s!="=")
              {
                cerr << "Error in " << ConfigFile  << "!" << endl;
                exit(1);
              }
             is >> s;
             cout << "Setting current working directory to \"" << s << "\"." << endl;
	     if (chdir(s.c_str())!=0)
              {
                cerr << "Changing working dircetory failed!" << endl;
                exit(2);
              };
             continue;
           }
          if (Keyword=="SkipFermat")
           {
	     SkipFermat=true;
             continue;
           }
          if (Keyword=="SkipPhi")
           {
	     SkipPhi=true;
             continue;
           }
          if (Keyword=="SkipFibonacci")
           {
	     SkipFibonacci=true;
             continue;
           }
          if (Keyword=="SkipECM")
           {
	     SkipECM=true;
             continue;
           }

        }
    }
  else
    {
       static char ch = 'N';
       cout << ConfigFile << " not found! Using default values!" << endl;

       if (ch!='Y')
         {
           cout << "Bitte mit [Y]es besttigen!" << endl;
           cout << "Please confirm with [Y]es." << endl;
           cin >> ch;
           if (ch!='Y') exit(1);
         }
    }

  char cwd[4096];
  cout << "Current working directory: " << getcwd(cwd,4096) << endl;
  cout << "Configuration file: " << ConfigFile << endl;
  cout << "DynamicRelationsFile: " << DynamicRelationsFile << endl;
  if (!Factorization_to_file.is_open()) Factorization_to_file.open(FactorizationsFile.c_str(),ios::out|ios::app);
}

void tune_parameters(const unsigned int Dezimalstellen)
{
  ifstream in(ConfigFile.c_str());
  char eingabezeile[200];
  string zeile;
  unsigned int Stellen, gewaehlte_Stellenzahl=0;
  if (in)
    {
      cout << "Reading configuration file" << endl; 
      while (!in.eof())
	{
	  in.getline(eingabezeile,sizeof(eingabezeile),'\n');
	  if (eingabezeile[0]=='#') continue; // Kommentarzeilen berlesen
	  zeile=eingabezeile;
          while (zeile[0]==' ') zeile.erase(0,1);
          if (zeile.empty() || zeile[0]=='#') continue; // leere Zeile/Kommentarzeile berlesen

          // Keywords verarbeiten
          istringstream is(zeile);
          string Keyword;
          is >> Keyword;
	  if (Keyword=="SkipFermat") continue;
	  if (Keyword=="SkipPhi") continue;
	  if (Keyword=="SkipFibonacci") continue;
	  if (Keyword=="SkipECM") continue;
          if (Keyword=="DynamicRelationsFile" || Keyword=="WorkDir")
           {
             string s;
             is >> s;
             if (s!="=")
              {
                cerr << "Error in " << ConfigFile  << "!" << endl;
                exit(1);
              }
             // is >> DynamicRelationsFile; // hier nicht (mehr) einlesen!
             continue;
           }

          // ansonsten: Zeile enthlt Tuning-Parameter
	  Stellen = atoi(zeile.substr(0, zeile.find(":")).c_str());
	  if (Stellen > Dezimalstellen || Stellen==0) break;
          gewaehlte_Stellenzahl=Stellen;
	  zeile.erase(0, zeile.find(":")+1);
	  rho_Phase=atoi(zeile.substr(0, zeile.find(",")).c_str());
	  zeile.erase(0, zeile.find(",")+1);
	  phi_Phase1=atoi(zeile.substr(0, zeile.find(",")).c_str());
	  zeile.erase(0, zeile.find(",")+1);
	  phi_Phase2=strtod(zeile.substr(0, zeile.find(",")).c_str(),NULL);
	  zeile.erase(0, zeile.find(",")+1);
	  elcu_Phase1=atoi(zeile.substr(0, zeile.find(",")).c_str());
	  zeile.erase(0, zeile.find(",")+1);
	  elcu_Phase2=strtod(zeile.substr(0, zeile.find(",")).c_str(),NULL);
	  zeile.erase(0, zeile.find(",")+1);
	  elcu_Kurven=atoi(zeile.substr(0, zeile.find(",")).c_str());
	  zeile.erase(0, zeile.find(",")+1);
	  Primbasis_size=atoi(zeile.substr(0, zeile.find(":")).c_str());
	  zeile.erase(0, zeile.find(":")+1);
	  sscanf(zeile.substr(0, zeile.find(":")).c_str(), "%lf", &Factor_Threshold);
	  zeile.erase(0, zeile.find(":")+1);

	  M = atoi(zeile.c_str());
	  M = (M>Siebgroesse) ? M : Siebgroesse;
	}
    }
  else
    {
       static char ch = 'N';
       cout << ConfigFile << " not found! Using default values!" << endl;
       if (ch!='Y')
         {
           cout << "Bitte mit [Y]es besttigen!" << endl;
           cout << "Please confirm with [Y]es." << endl;
           cin >> ch;
           if (ch!='Y') exit(1);
         }
    }
  
  cout << "Configuration:" << endl;
  cout << "#decimal digits: " << Dezimalstellen << endl;
  cout << "actual parameters for " << gewaehlte_Stellenzahl << " digits" << endl;
  cout << "Pollard-Rho: " << rho_Phase << endl;
  cout << "Pollard-Phi: phase 1: " << phi_Phase1 << " phase 2: " << setprecision(1) << phi_Phase2 << endl;
  cout << "elliptic curves: phase 1: " << elcu_Phase1
       << " phase 2: " << elcu_Phase2
       << " #curves: " << elcu_Kurven << endl; 
  cout << "Size of static factorbase: " << Primbasis_size << endl;
  cout << "factor-threshold (exponent): " << Factor_Threshold << endl;
  cout << "sieve interval per polynomial: [" << -M << "," << M << "]" << endl;

  if (Primbasis_size>MAX_Primbasis_size)
    {
      cerr << "MAX_Primbasis_size is too small (or Primbasis_size is to big)!" << endl;
      exit(1);
    }
}
#endif /* von #ifndef IS_CLIENT */


const char* MAL(void)
{
  // liefert Leerstring bei erstem Aufruf, " * " sonst.
  // Aufgabe: Verhinderung aufwendiger Tests, um den Faktor 1 als
  // ersten bzw. letzten Faktor unterdrcken zu knnen.
  static bool erster_Aufruf = true;
  if (erster_Aufruf)
   {
     erster_Aufruf=false;
     return "";
   }
  else return " * ";
}

string MAL(const mpz_t factor, const unsigned int exponent=1)
{
  ostringstream helper;
  helper << MAL() << factor;
  if (exponent>1) helper << "^" << exponent;
  return helper.str();
}


#include "polynomial.cc" /* for fft-continuation */
#include "fft_param.cc"  /* dicrete fast fourier transform */ 
#include "elliptic_curve.cc" /* elliptic curve method (ecm) */


#ifndef IS_CLIENT
 // this stuff may only be needed by server or stand-alone version
 #include "easy_factor.cc"
#endif /* of #ifndef IS_CLIENT */



#include "sqrt_modulo.cc"
#include "mpz_sqrtmod.cc"

inline int SQRT_kN_mod_Primzahl(const unsigned int Primzahl)
{
  return sqrtmod(mpz_remainder_ui(kN,Primzahl),Primzahl);
}


int check_SQRT_kN_mod_Primzahl(const int Primzahl)
{
  // kann zu Debugging-Zwecken aufgerufen werden, um zu testen,
  // inwieweit die "SQRT_kN_mod_Primzahl"-Funktion korrekt arbeitet
  int w1 = SQRT_kN_mod_Primzahl(Primzahl);
  if (w1>Primzahl/2) w1=Primzahl-w1;

  mpz_t x;
  mpz_init_set_ui(x,w1); mpz_mul(x,x,x); mpz_sub(x,x,kN);
  if (mpz_mod_ui(x,x,Primzahl)!=0)
    {
      cerr << "WURZELALARM! " << w1 << " mod " << Primzahl << endl;
      exit(1);
    }
  mpz_clear(x);
  return w1;
}


void Statische_Faktorbasis_ermitteln()
{ 
  cout << "Calculating static factorbase." << endl;
  Primzahlen[0]=-1; /* nullte "Primzahl" ist -1 (Bercksichtigung des
		       Vorzeichens bei den Relationen! */
  log_Primzahlen[0]=0; // Vorzeichen hat keinen Logarithmuswert

  int Primzahl=2;
  // die Primzahl 2 ist ein Sonderfall. Ich gehe davon aus, da sie
  // auf jeden Fall in der Faktorbasis vorhanden sein soll.
  Primzahlen[1]=2; log_Primzahlen[1]=static_cast<TSiebElement>(ceil(log_Multiplikator*log(Primzahl)));

  Primzahl=1; // Initialwert 1 nur, wenn in der Schleife +=2 gezhlt wird!
  mpz_t x,y,wuqu;
  mpz_init(x); mpz_init(y); mpz_init(wuqu); // Fr Berechnung der Quadratwurzeln von Primzahlpotenzen
  for (int Primzahl_nr=2; Primzahl_nr<Primbasis_size; Primzahl_nr++)
    {
      do // nchsthhere Primzahl der statischen Faktorbasis ermitteln 
	{
	  do Primzahl+=2; while(!is_prime(Primzahl)); /* nchste Primzahl */
	  mpz_set_ui(x,Primzahl);
	}
      while (mpz_legendre(kN,x)<0); /* Primzahl gltig fr Primzahlbasis? */
      
      //printf("%i. Primzahl der Faktorbasis ist %i\r",Primzahl_nr,Primzahl);
      Primzahlen[Primzahl_nr]=Primzahl;
      
      log_Primzahlen[Primzahl_nr]=static_cast<TSiebElement>(ceil(log_Multiplikator*log(Primzahl)));
      // fr effizientes Sieben erforderlich (erspart dauernde Neuberechnung)
      
      SQRT_kN_der_Primzahlen[Primzahl_nr]=SQRT_kN_mod_Primzahl(Primzahl);
      // fr sptere Ermittlung der Deltawerte zum Sieben erforderlich
      
      // ggf. Prfung, ob Wurzel korrekt ist:
      mpz_set_ui(x,SQRT_kN_der_Primzahlen[Primzahl_nr]);
      mpz_mul(x,x,x); mpz_sub(x,kN,x); 
      if (mpz_remainder_ui(x,Primzahl)!=0)
       {
         cerr << "Wurzel nicht korrekt!" << endl;
         cerr << SQRT_kN_der_Primzahlen[Primzahl_nr] << "^2 <> "
              << mpz_remainder_ui(kN,Primzahl) << " (mod " << Primzahl << ")" << endl;
         exit(1);
       }

      if (Primzahl<=2*M/Primzahl && Primzahl>2 && mpz_remainder_ui(kN,Primzahl)!=0)
	// Primzahl darf nicht zu gro sein, ungleich 2 und kein Teiler von kN, um mit Potenzen gesiebt zu werden
	{
	  FB_maxQuadrate=Primzahl_nr; /* nach Abschlu der Schleife ist dies der Wert, bis zu dem
					 Quadrate auf normale Weise gesiebt werden. */

	  // Wurzeln von kN modulo Primzahl^pot ermitteln
	  // Initialwerte:
	  int P_Potenz = Primzahl*Primzahl;
	  int Wu       = SQRT_kN_der_Primzahlen[Primzahl_nr];

	  for (short int pot=2; ;pot++)
	    {
	      //cout << "berechne Wurzel fr " << Primzahl << "^" << pot << endl;

	      mpz_set_ui(y,2*Wu); mpz_set_ui(x,P_Potenz);
	      if (mpz_invert(y,y,x)==0)
		{ 
		  cerr << "PPot-Wurzel fr " << Primzahl  << ": Inverses existiert nicht!" << endl;
		  break;
		}

              if (anz_weitere_Primzahlpotenzen>=FB_max_weitere_Potenzen)
               { 
                 static char weitermachen = 'n';
                 if (weitermachen!='y')
                  {
	           cerr << "berlauf ab " << Primzahl << "^" << pot << endl;
                   cerr << "Es kann nicht optimal mit Primzahlpotenzen gesiebt werden," << endl
                        << "da der hierfr reservierte Speicherplatz nicht ausreicht." << endl;
                   cerr << "Sie sollten \"FB_max_weitere_Potenzen\" vergroessern!" << endl;
                   cerr << "Reserved memory for storing prime powers is to small..." << endl;
                   cerr << "sieving will be less efficient..." << endl;
                   cerr << "continue? (y/n) " << flush;
                   cin >> weitermachen;
                   if (weitermachen=='n') exit(1);
                  }
                 break; // berlauf durch Schleifenabbruch verhindern
               }

	      mpz_mod_ui(x,kN,P_Potenz); mpz_set_ui(wuqu,Wu); mpz_mul_ui(wuqu,wuqu,Wu); mpz_sub(x,x,wuqu);
	      mpz_mul(x,x,y); mpz_add_ui(x,x,Wu); mpz_mod_ui(x,x,P_Potenz);
	      register int h = mpz_get_ui(x); if (h>P_Potenz-h) h=P_Potenz-h; // auf die "kleinste" Wurzel normieren
	      Primzahlpotenzen[anz_weitere_Primzahlpotenzen]=P_Potenz;
	      SQRT_kN_der_Primzahlpotenzen[anz_weitere_Primzahlpotenzen]=h;
	      if ( pot==2 && Primzahl_nr<SiebRegler::sieben_ab_Primzahlindex() )
		// Exponent 2, weil erstes Auftreten des Faktors
		log_Primzahl_der_Primzahlpotenzen[anz_weitere_Primzahlpotenzen]
                 =static_cast<TSiebElement>(ceil(log_Multiplikator*2*log(Primzahl)));
	      else
		// Exponent 1, weil nochmaliges Auftreten des Faktors
		// beachte: wenn man schon einmal durch p geteilt hat, ergibt ein weiteres Teilen den Teiler p^2
		log_Primzahl_der_Primzahlpotenzen[anz_weitere_Primzahlpotenzen]
                 =static_cast<TSiebElement>(ceil(log_Multiplikator*log(Primzahl)));
	      
	      // ggf. Prfung, ob Wurzel korrekt ist:
	      mpz_set_ui(x,SQRT_kN_der_Primzahlpotenzen[anz_weitere_Primzahlpotenzen]);
	      mpz_mul(x,x,x); mpz_sub(x,kN,x); 
	      if (mpz_mod_ui(x,x,P_Potenz)!=0)
		{
		  cerr << "PPot-Wurzel nicht korrekt!" << endl;
		  cerr << "Primzahl=" << Primzahlen[Primzahl_nr] << " Nr. " << Primzahl_nr << endl;
		  cerr << "kN= " << kN << endl;
		  cerr << "SQRT(kN) mod p=" << SQRT_kN_der_Primzahlen[Primzahl_nr] << endl;
		  cerr << "SQRT(kN) mod p^i=" << SQRT_kN_der_Primzahlpotenzen[anz_weitere_Primzahlpotenzen] << endl;
		  cerr << "aber falsch: " << x << endl;
		  exit(1);
		}
	      
	      anz_weitere_Primzahlpotenzen++; // dies war eine weitere aufgenommene Primzahlpotenz

	      if (P_Potenz>2*M/Primzahl) break; // nchste Primzahlpotenz wird zu gro...
	      P_Potenz*=Primzahl; Wu=SQRT_kN_der_Primzahlpotenzen[anz_weitere_Primzahlpotenzen-1];
	    }
	  
	}
    }
  mpz_clear(x); mpz_clear(y); mpz_clear(wuqu);

  biggest_Prime_in_Factorbase=Primzahl;
  cout << "prime numbers in static factorbase have been computed." << endl;
  
  if (Primzahl>=Siebgroesse)
    cout << "Remark: Some members of the static factorbase are bigger than the sieve size!" << endl;
}


#ifndef IS_SERVER
void Deltas_berechnen_fuer(const TDynamicFactorRelation &DynRel, const int offset)
{
  TSieb_Delta sieb_delta;
  int d1,d2;
  
  sieb_delta.factor=DynRel.factor;

  const unsigned int inv_A2 = invmod(mpz_remainder_ui(Polynom.get_A2(),DynRel.factor),DynRel.factor);
  const unsigned int PolyB = mpz_remainder_ui(Polynom.get_B(),DynRel.factor);

  d1=DynRel.SQRT_kN-PolyB; if (d1<0) d1+=DynRel.factor;
  d1=mulmod(d1,inv_A2,DynRel.factor);
  sieb_delta.delta=normalized_signed_mod(d1-offset,DynRel.factor);
  sieb_delta.delta+=offset;
#ifdef USE_FAKEHEAP
  while (sieb_delta.delta<=M) // shortcut: nur dann eintragen, wenn die Relation noch im Siebintervall relevant wird
   {
     Sieb_Delta_Heap.push(sieb_delta);
     sieb_delta.delta+=sieb_delta.factor;
   }
#else
  if (sieb_delta.delta<=M) // shortcut: nur dann eintragen, wenn die Relation noch im Siebintervall relevant wird 
    Sieb_Delta_Heap.push(sieb_delta);
#endif
  
  d2=DynRel.factor-DynRel.SQRT_kN-PolyB; if (d2<0) d2+=DynRel.factor;
  d2=mulmod(d2,inv_A2,DynRel.factor);
  sieb_delta.delta=normalized_signed_mod(d2-offset,DynRel.factor);
  sieb_delta.delta+=offset;
  if ( (sieb_delta.delta<=M)  && (d1!=d2) ) // shortcut: nur dann eintragen, wenn die Relation noch im Siebintervall relevant wird 
   {
     Sieb_Delta_Heap.push(sieb_delta);
#ifdef USE_FAKEHEAP
     sieb_delta.delta+=sieb_delta.factor;
     while (sieb_delta.delta<=M)
      {
        Sieb_Delta_Heap.push(sieb_delta);
        sieb_delta.delta+=sieb_delta.factor;
      }
#endif
   }
}
#endif


//Vorwrtsdeklarationen:
int hole_Wert (istream &in);
void schreibe_Wert (ostream &out, const int w);

void Special_Factoren_durch_Primfaktor_splitten_LAZY(const int Primfaktor)
{
  // Wenn ein neuer dynamischer Primfaktor entdeckt wird, so empfiehlt es sich,
  // in der Menge der Specialfaktoren (welches zusammengesetzte Faktoren sind) einen
  // Teilbarkeitstest auf diesen Primfaktor durchzufhren. Auf diese Weise knnen
  // weitere Primfaktoren isoliert werden, die als dynamische Faktoren zur Verfgung stehen.
  // Im Gegensatz zur bisherigen BUSY-Methode verwandelt die LAZY-Methode
  // den abgesplitteten Primfaktor in einen Korrekturfaktor. Die eigentliche
  // Auswertung der Relation wird damit auf einen spteren Zeitpunkt verschoben
  // und mglicherweise ganz verhindert, falls die betroffene Relation nicht
  // bentigt wird.

  static int Trashcounter = 0; /* Die (nach Elimination eines Faktors) nutzlos auf dem File
				  befindlichen Relationen */
  
  TSpecial_Factor_Relations::iterator pos;
  TSpecialFactorRelation SearchRelation;
  SearchRelation.factor.set_for_search(Primfaktor);

  while (true)
   {
    //cout << "splitten " << Primfaktor << endl;
    /* Wegen Rekursion etc. immer wieder korrekt neu aufsetzen, wenn
       ein Specialfaktor gekegelt wurde -> nchsthherer Special-Factor */
    pos=Special_Factor_Relations.lower_bound(SearchRelation);
    if (pos==Special_Factor_Relations.end()) break;
    if (!(*pos).factor.DLP_enthaelt_Teiler(Primfaktor)) break;

    TSpecialFactorRelation gemerkte_Relation = *pos;
    Special_Factor_Relations.erase(pos); // alten Specialfactor entfernen
    gemerkte_Relation.factor.swap();
    Special_Factor_Relations.erase(gemerkte_Relation); // alten Specialfactor entfernen (geswapte Kopie)
    gemerkte_Relation.factor.swap();
    
    TDynamicFactorRelation relation;
    relation.factor=Primfaktor;
    TDynamic_Factor_Relations::iterator p_el,p;
    p_el = Dynamic_Factor_Relations.find(relation);
    if (p_el==Dynamic_Factor_Relations.end()) 
      {
        MARK; 
        cerr << "Fehler in dynamic factorbase! (Specialfaktoren splitten)" << endl;
        exit(1);
      }

    relation.factor=(*pos).factor/Primfaktor; // Dies ist der neue Restfaktor
    // es liegt nun der ermittelte dynamic Factor korrekt vor

    // aber ist er neu oder existierte er schon in der dyn. FB?
    p = Dynamic_Factor_Relations.find(relation);
    if (p==Dynamic_Factor_Relations.end())
     { 
       // er ist neu!
       //cout << "Special Factor: Dynamischer Faktor erzeugt!" << endl;
       Special_to_dynamic_Factor_hit++;
       relation.fpos = DynamicRelations_to_file.tellp();
       // nun die Relation modifiziert umkopieren:
       {
         const streampos rpos=SpecialRelations_from_file.tellg(); // Position merken
         SpecialRelations_from_file.seekg(gemerkte_Relation.fpos); // Position merken
         CmpqsFactor ret; string s;
         // Dateiformatflags retten
         const ios::fmtflags oldFlagsR = SpecialRelations_from_file.flags(); // alte Formatflags retten
         SpecialRelations_from_file.unsetf(ios::hex); SpecialRelations_from_file.setf(ios::dec); // Dezimalausgabe sicherstellen (beachte: Rekursiver Aufruf in Lazy-Variante wegen Korrekturfaktoren mglich!)
         SpecialRelations_from_file.unsetf(ios::showbase); // ohne Basiskennzeichnung
         const ios::fmtflags oldFlagsW = DynamicRelations_to_file.flags(); // alte Formatflags retten
         DynamicRelations_to_file.unsetf(ios::hex); DynamicRelations_to_file.setf(ios::dec); // Dezimalausgabe sicherstellen (beachte: Rekursiver Aufruf in Lazy-Variante wegen Korrekturfaktoren mglich!)
         DynamicRelations_to_file.unsetf(ios::showbase); // ohne Basiskennzeichnung
 
         SpecialRelations_from_file >> s; // Startzeichen
         if (s != "G")
          {
	   MARK;
           cerr << "error: wrong position while reading relation" << endl;
           exit(1);
          }
         SpecialRelations_from_file >> ret; // Faktor einlesen (dynamic-factor oder special-factor der Relation oder 1)
         SpecialRelations_from_file >> s; // delta
         DynamicRelations_to_file << "G " << setprecision(20) << relation.factor << " " << s << " ";

         // ab jetzt wird hexadezimal gearbeitet:
         SpecialRelations_from_file.unsetf(ios::dec); SpecialRelations_from_file.setf(ios::hex); // Hexadezimalausgabe
         SpecialRelations_from_file.unsetf(ios::showbase); // ohne Basiskennzeichnung
         DynamicRelations_to_file.unsetf(ios::dec); DynamicRelations_to_file.setf(ios::hex); // Hexadezimalausgabe
         DynamicRelations_to_file.unsetf(ios::showbase); // ohne Basiskennzeichnung

         int distance=hole_Wert(SpecialRelations_from_file);
         while (distance>=0)
          {
            schreibe_Wert(DynamicRelations_to_file,distance);
            distance=hole_Wert(SpecialRelations_from_file);
          }
         schreibe_Wert(DynamicRelations_to_file,-2);
         schreibe_Wert(DynamicRelations_to_file,Primfaktor);
         if (distance==-2)
          {
            distance=hole_Wert(SpecialRelations_from_file);
            while (distance>=0)
             {
               schreibe_Wert(DynamicRelations_to_file,distance);
               distance=hole_Wert(SpecialRelations_from_file);
             }            
          }
         schreibe_Wert(DynamicRelations_to_file,-1); DynamicRelations_to_file << endl;
         SpecialRelations_from_file.flags(oldFlagsR); // alte Formatflags wiederherstellen
         DynamicRelations_to_file.flags(oldFlagsW); // alte Formatflags wiederherstellen
         SpecialRelations_from_file.seekg(rpos); // position restaurieren
       }
       relation.SQRT_kN=SQRT_kN_mod_Primzahl(relation.factor);
#if !defined(IS_SERVER) && !defined(USE_FAKEHEAP)
       Deltas_berechnen_fuer(relation, Sieboffset+Siebgroesse);
#endif
       Dynamic_Factor_Relations.insert(relation); // Dynamicfactor einfgen
       Special_Factoren_durch_Primfaktor_splitten_LAZY(relation.factor);
     }
    else 
     { 
       // er existierte schon in der dyn. FB -> er fhrt zu einem Treffer im GLS!
       cout << "Special Factor: Special Hit!" << endl;
       CRelation *GL = new CRelation;
       GL->kombinieren(SpecialRelations_from_file,gemerkte_Relation.fpos);
       // gefundene dynamische Faktoren kombinieren:
       mpz_mul_ui(GL->Delta,GL->Delta,(*p).factor); mpz_mod(GL->Delta,GL->Delta,n);
       GL->kombinieren(DynamicRelations_from_file,(*p).fpos);
       // nicht vergessen: auch Primfaktor eliminieren!
       mpz_mul_ui(GL->Delta,GL->Delta,(*p_el).factor); mpz_mod(GL->Delta,GL->Delta,n);
       GL->kombinieren(DynamicRelations_from_file,(*p_el).fpos);
       Special_hit++; Relation_einfuegen(GL);
     } 

    { // alte Special-Relation streichen
      streampos pos;
      pos = SpecialRelations_to_file.tellp(); // position merken
      SpecialRelations_to_file.seekp(gemerkte_Relation.fpos); SpecialRelations_to_file << "U "; // ungltig markieren
      SpecialRelations_to_file.seekp(pos); // position restaurieren
    }

    Trashcounter++; // Die auf dem File verbleibenden sinnlosen Relationen zhlen 
   }

 // Wenn sich sehr viel Mll angesammelt hat: Meldung erstatten
 if (Trashcounter>200000)
   {
     // Hier knnte man nun eine Kompaktifizierung des Special-Relation-Files
     // durchfhren. Das wrde aber fr den laufenden Serverbetrieb zu lange
     // dauern. Stattdessen wird ab jetzt lediglich eine Meldung ausgegeben,
     // in welcher ein Runterfahren des Programms und ein anschlieender
     // Neustart desselben empfohlen wird. (Beim Recovery wird eine
     // Kompaktifizierung automatisch durchgefhrt.)
     clog << "Das Special-Relations-File hat einen hohen Anteil an redundanten" << endl;
     clog << "Relationen, die eliminiert werden knnten." << endl;
     clog << "Ein Runterfahren (Abbruch) und erneutes Starten (Recovery) des" << endl;
     clog << "Programms wrde eine Kompaktifizierung des Files initiieren." << endl;
     Trashcounter-=100; // in Krze die Meldung wiederholen...
   }
}


// fr die Speicherung der gefundenen dynamischen Faktoren.
// Bemerkung: Um die dynamischen Faktoren bereits in der Siebphase
//            aufzusammeln, ist das Array global.
//             (Dafr wird die Anzahl der gefundenen dynamischen Faktoren
//              fr die aktuelle Relation dem Konstruktor bergeben.)
const short int MaxHits=64; // bei Bedarf erhhen...
static struct {int Faktor, Exponent;} Hits[MaxHits]; // Dynamische Special-Factoren der Relation aufsammeln
// wichtiger Hinweis: Der Exponent zum Faktor wird nicht beim Sieben, sondern erst bei der Refaktorisierung
// ermittelt. Falls also mit Primzahlpotenzen gesiebt werden sollte, braucht der Exponent nicht eingetragen
// zu werden. Als Faktor sollte in jedem Falle die Primzahl und nicht die Primzahlpotenz eingetragen werden!

CRelation::CRelation(const signed int Siebpos, const short int HitCount /* =0 */)
: CProvideHilfsvariablen(), relevanter_Faktor(no_factor)
  /* Konstruktor: neue Relation einfgen.
     "Deltas" mit dem Wert Delta initialisieren,
     "Relationen" mit den Primfaktoren des reduzierten Quadrates fllen,
     deren Exponenten ungerade sind. */
{
  // cout << "initialisiere Gleichungszeile" << endl;
  Relation_sparse = new CTinyFBsizetypeVector;
  Relation_dense=NULL;
  mpz_init(Delta);
  Polynom.Werte_holen(Siebpos,Delta,y);
  // in y: reduziertes Quadrat (ist nun zu faktorisieren...) 
  mpz_invert(Delta,Delta,n); /* wir verwenden den Kehrwert von Delta,
				das erspart eine zustzliche Variable;
				gleichzeitig ist damit die zu findende
				quadratische Kongruenz auf Delta^2=1 (mod n)
				normiert. */
  
  if (mpz_sgn(y)<0) // Vorzeichenbehandlung 
    {
      Relation_sparse->append(0); 
      /* "nullte Primzahl" ist -1 wegen Bercksichtigung
         des Vorzeichens. Also die 0 fr Position von  -1 einfgen */
      mpz_neg(y,y); /* -1 war Teiler -> positiv machen */
    }
  
  // ersten Faktor der statischen Faktorbasis ebenfalls gesondert behandeln,
  // denn die generierten Polynome haben Probleme, die Primzahl 2 zu sieben...
  int z=0;
  while (mpz_div_ui(x,y,Primzahlen[1])==0) { mpz_set(y,x); z++; }
  if (z&1) Relation_sparse->append(1); // ungerade Potenz der Primzahl markieren
  if (z>1) // Delta mit den "gltigen Wurzeln" multiplizieren
  {
    if (z<4) mpz_mul_ui(Delta,Delta,Primzahlen[1]);
    else
      {
	mpz_set_ui(x,Primzahlen[1]); mpz_powm_ui(x,x,z/2,n);
	mpz_mul(Delta,Delta,x);
      }
  }
  
  
  // find factors in dynamic factorbase
  short int HitCount2 = HitCount;
  while (HitCount2--)
    {
      int Exponent=0; // Initialisierung des Exponenten
      while (mpz_div_ui(x,y,Hits[HitCount2].Faktor)==0)
	{
	  mpz_set(y,x); // getestete Division verwirklichen 
	  Exponent++; // Exponent fr Faktor inkrementieren
	}
      //if (Exponent==0)
      //{ // nur testweise fr debuggingzwecke
      //  cerr << "Falscher Teiler in Hits[]: " << Hits[HitCount2].Faktor << endl;
      //  exit(1);
      //}
      if (Exponent>1)
	{
	  mpz_set_ui(x, Hits[HitCount2].Faktor);
	  mpz_powm_ui(x,x,Exponent/2, n);
	  mpz_mul(Delta,Delta,x); mpz_mod(Delta,Delta,n);
	  Exponent%=2;
	}
      Hits[HitCount2].Exponent = Exponent;
    }


  {
    // find factors in stactic factorbase
    // (beginnend mit der 3. Primzahl der FB (s.o) )
    bool* const DivFlag = new bool[Primbasis_size];
    for (int i=2; i<Primbasis_size; i++)
     {
       // shortcut:
       /* keine aufwendige Division durchfhren, wenn diese
          Primzahl gar nicht ausgesiebt wurde */ 
  
       register int d = normalized_signed_mod(Siebpos,Primzahlen[i]);
       // same as: Siebpos%Primzahlen[i]; if (d<0) d+=Primzahlen[i];
       // Bemerkung: %-operation does *not* eliminate the sign!
       //            ugly: -1=1 (mod 2), but (-1)%2=-1, 1%2=1
       // -> so we have to eliminate the sign!!!
       // and don't try to use unsigned int d, this won't work!
  
  #if 0
       // for debugging reasons:
       bool b1=(mpz_div_ui(x,y,Primzahlen[i])==0);
       bool b2=(d==Delta_der_Primzahlen[i][0] || d==Delta_der_Primzahlen[i][1]);
       if (b1!=b2) 
        {
          cerr << "error in shortcut: " << Siebpos << "; "
               << Primzahlen[i] << ": " << d << ", "
               << Delta_der_Primzahlen[i][0] << ", "
               << Delta_der_Primzahlen[i][1] << endl;
          exit(1);
        }
  #endif
  
       DivFlag[i] = (d==Delta_der_Primzahlen[i][0] || d==Delta_der_Primzahlen[i][1]);
     }
   
    // Faktoren zusammenmultiplizieren und dann in einem Rutsch abdividieren
    mpz_set_ui(x,1);
    for (int i=2; i<Primbasis_size; i++) if (DivFlag[i]) mpz_mul_ui(x,x,Primzahlen[i]);
    mpz_divexact(y,y,x);
  
    // nun die Potenzen der Faktoren abdividieren...
    for (int i=2; i<Primbasis_size; i++) if (DivFlag[i])
      {
        z=1; // einmal ist dieser Faktor bereits vorhanden (gewesen)
        //while (mpz_div_ui(x,y,Primzahlen[i])==0) { mpz_set(y,x); z++; }
        while (mpz_divisible_ui_p(y,Primzahlen[i])) { mpz_divexact_ui(y,y,Primzahlen[i]); z++; } // should be faster...
  
        // Relation updaten:
        if (z&1) Relation_sparse->append(i); /* ungerade Potenz der i. Primzahl markieren */
        if (z>1) // Delta mit den "gltigen Wurzeln" multiplizieren
          {
            if (z<4) mpz_mul_ui(Delta,Delta,Primzahlen[i]);
            else
              {
                mpz_set_ui(x,Primzahlen[i]); mpz_powm_ui(x,x,z/2,n);
                mpz_mul(Delta,Delta,x);
              }
            mpz_mod(Delta,Delta,n); // Wert klein halten...
          }
      }
   delete [] DivFlag;
  }
  
  // Restfactor in static factorbase...
  const bool existiert_grosser_Restfaktor = (mpz_cmp_ui(y,biggest_Prime_in_Factorbase)>0);

  if (existiert_grosser_Restfaktor && collecting_phase_beendet)
    { relevanter_Faktor=special_factor; return; } // vorzeitiges Abbrechen

  if (!existiert_grosser_Restfaktor)
    {
      if (mpz_cmp_ui(y,1)!=0) 
        {
          cout << "CRelation::Konstruktor:" << endl;
          cout << "Fehler in Relation / BUG im Programm:" << endl;
          cout << y << " sollte 1 sein!" << endl;
          exit(1);
        }
      relevanter_Faktor=groesster_Faktor_in_Relation();
      // mit der aufgenommenen Relation lt sich dieser Faktor eliminieren
    }

#if 1
  // dynamic factor sanity check
  HitCount2 = HitCount;
  while (HitCount2--) if(Hits[HitCount2].Exponent!=0)
    {
      TDynamicFactorRelation FaRelSearch;
      FaRelSearch.factor=Hits[HitCount2].Faktor;
      TDynamic_Factor_Relations::iterator p = Dynamic_Factor_Relations.find(FaRelSearch);
      if (p==Dynamic_Factor_Relations.end())
	{
	  cerr << "Fehler in dynamic factorbase! " << FaRelSearch.factor << " not found!" << endl;
          // entweder tatschlich ein Fehler ODER
          // denkbarer (unwahrscheinlicher) Sonderfall:
          // ein gepushtes Primzahl-Quadrat auerhalb der dynamischen
          // Faktorbasis geht (sogar) mit ungerader Potenz ein...
          // ...dann liegt zwar kein Fehler vor, aber die Relation ist
          // trotzdem ungltig. Der Fall erscheint mir aber derart
          // unwahrscheinlich, da ich dann trotzdem abbreche.
          // Hinzu kommt: Wenn ein Client einen solchen Faktor sieben sollte,
          // dann wird diese Bearbeitung hier gar nicht aufgerufen und
          // der Server wird im Fehlerfall die Relation nicht akzeptieren.
          // Dann gibt es zwar eine mgliche dynamische Relation weniger,
          // aber das sollte bei der Seltenheit dieses Falles nicht stren.
	  exit(1); // not necessary, because it should also be safe just to skip this relation
          relevanter_Faktor=dynamic_factor; return; // skip the relation
	}
    }
#endif

  if (existiert_grosser_Restfaktor)
    {
      const bool is_possible_dynamic = (mpz_cmp_ui(y, Specialfactor_Sieb_Threshold)<=0);
      if ( !is_possible_dynamic && CmpqsFactor::DLP_rejected(y) )
        {
          // We should first filter out invalid big numbers, especially the
          // primes! It doesn't harm (except for performance) if some composite
          // numbers are filtered out as primes.
          // Important is that "DLP_reject(y)" is not too expensive!
	  relevanter_Faktor=special_factor; // ignored factor (because we do not insert any relation)
	  return;
        }
        
      if (is_possible_dynamic && probab_prime(mpz_get_ui(y)) )
        {
          // es liegt ein neuer dynamic Factor vor
          //cout << "neuer dynamic-factor..." << endl;
          relevanter_Faktor=dynamic_factor;
          TDynamicFactorRelation relation;
          relation.factor=mpz_get_ui(y);
          relation.SQRT_kN=SQRT_kN_mod_Primzahl(relation.factor);
#if !defined(IS_SERVER) && !defined(USE_FAKEHEAP)
          Deltas_berechnen_fuer(relation, Sieboffset+Siebgroesse);
#endif
      
#ifndef IS_CLIENT
          relation.fpos = speichern(DynamicRelations_to_file,relation.factor, HitCount);
#else
          // an das Ausgabefile
          relation.fpos=speichern(communication_stream,relation.factor, HitCount);
#endif
          Dynamic_Factor_Relations.insert(relation);
#ifndef IS_CLIENT
          Special_Factoren_durch_Primfaktor_splitten_LAZY(relation.factor);
#endif
          return;
        }
        
      // es liegt ein Specialfactor vor, der nicht prim ist
      {
        relevanter_Faktor=special_factor;
        TSpecialFactorRelation relation;
        if (!relation.factor.DLP_get(y))
          {
            //cerr << "Hinweis: Special Factor " << y << " ungltig." << endl;
            return; /* Treffer nicht bernehmen, da er ungltig ist. */
          }
#ifndef IS_CLIENT
        if (!SpecialFactorRelation_vermerken(relation.factor,this,HitCount))
          {
            relevanter_Faktor=groesster_Faktor_in_Relation();
            // wenn Factor nicht vermerkt wurde, heit das, da er
            // neutralisiert werden konnte. -> normale Relation liegt vor.

            // kombiniere Faktoren der dynamic Factorbase
            multi_kombinieren_init();
            HitCount2 = HitCount;
            while (HitCount2--) if(Hits[HitCount2].Exponent!=0)
             {
               //cout << "kombiniere dynamischen Factor " << Hits[HitCount2].Faktor << endl;
               TDynamicFactorRelation FaRelSearch;
               FaRelSearch.factor=Hits[HitCount2].Faktor;
               TDynamic_Factor_Relations::iterator p = Dynamic_Factor_Relations.find(FaRelSearch);
               mpz_mul_ui(Delta,Delta,(*p).factor); mpz_mod(Delta,Delta,n);
               multi_kombinieren_main(DynamicRelations_from_file,(*p).fpos);
               //cout << "fertig kombiniert von " << *pos << endl;
             }
            multi_kombinieren_exit();
            return;
          }
#else
        // an das Ausgabefile
        relation.fpos=speichern(communication_stream, relation.factor, HitCount);
#endif
        return;
      }
    } // end of "if (existiert_grosser_Restfaktor)"

#ifndef IS_CLIENT
  // kombiniere Faktoren der dynamic Factorbase
  multi_kombinieren_init();
  HitCount2 = HitCount;
  while (HitCount2--) if(Hits[HitCount2].Exponent!=0)
   {
     //cout << "kombiniere dynamischen Factor " << Hits[HitCount2].Faktor << endl;
     TDynamicFactorRelation FaRelSearch;
     FaRelSearch.factor=Hits[HitCount2].Faktor;
     TDynamic_Factor_Relations::iterator p = Dynamic_Factor_Relations.find(FaRelSearch);
     mpz_mul_ui(Delta,Delta,(*p).factor); mpz_mod(Delta,Delta,n);
     multi_kombinieren_main(DynamicRelations_from_file,(*p).fpos);
     //cout << "fertig kombiniert von " << *pos << endl;
   }
  multi_kombinieren_exit();
#else
  // Ausgabe an File
  speichern(communication_stream,1, HitCount);
#endif
      
  if (HitCount<6) Erfolgsaufteilung[HitCount]++; else Erfolgsaufteilung[6]++;
  // bei Erfolgsaufteilung[0]: Relation vollstndig durch statische FB entdeckt
  // bei Erfolgsaufteilung[0<k<6]: Relation durch Eliminierung von (genau) k Faktoren aus dyn. FB entdeckt
  // bei Erfolgsaufteilung[6]: Relation durch Eliminierung von mindestens 6 Faktoren aus dyn. FB entdeckt
  
  //cout << "GL: " << *this << endl;
  //cout << "relevanter Faktor: " << relevanter_Faktor << endl;
  //cout << "Initialisierung der Gleichungszeile abgeschlossen!" << endl;
}


ostream& operator<< (ostream& ostr, const CRelation& GL)
{
  if (GL.Relation_dense) ostr << "dense"; else ostr << "sparse";
  ostr << "1=";
  
  char *str;
  str = new char [mpz_sizeinbase(GL.Delta,10)+2]; mpz_get_str(str, 10, GL.Delta);
  ostr << str << "^2";
  delete [] str; 
  
  if (GL.Relation_dense)
    { 
      for (int i=GL.Relation_dense->first(); i>=0; i=GL.Relation_dense->next(i))
	ostr << "*p" << i;
    }
  else
    {
      const CTinyFBsizetypeVector& R = *GL.Relation_sparse;
      for (int pos=0; pos<R.size(); ++pos)
	ostr << "*p" << R[pos];
    }
  
  return ostr;
}


void CRelation::convert_Relation_to_dense()
{
  //cout << "Konvertiere von Sparsedarstellung nach Densedarstellung." << endl;
  if (Relation_dense)
    { cerr << "Relation ist bereits dense!" << endl; return; }
  Relation_dense = new myBitString;
  for (int pos=Relation_sparse->size()-1; pos>=0; --pos)
    Relation_dense->set((*Relation_sparse)[pos]);
  //Relation_sparse->erase(Relation_sparse->begin(),Relation_sparse->end());
  delete Relation_sparse; Relation_sparse=NULL; // Sparsedarstellung lschen
}

void CRelation::convert_Relation_to_sparse()
{
  //cout << "Konvertiere von Densedarstellung nach Sparsedarstellung." << endl;
  if (Relation_sparse)
    { cerr << "Relation ist bereits sparse!" << endl; return; }
  Relation_sparse = new CTinyFBsizetypeVector(Relation_dense->count(1));
  for (int i=Relation_dense->first(); i>=0; i=Relation_dense->next(i))
    Relation_sparse->fast_append(i);
  //Relation_dense->clear();
  delete Relation_dense; Relation_dense=NULL; // Densedarstellung lschen
}


void CRelation::kombinieren(const CRelation& GL2)
  // bergabe von GL2 ber Referenz, um effizienter zu sein...
  /* symmetrische Differenzen ermitteln */
{
  //cout << "Kombiniere Gleichungszeilen..." << endl;
  //cout << "Gl.1: "; Kongruenzcheck();
  //cout << "Gl.2: "; GL2.Kongruenzcheck();
  
  /*
    Symmetrische Differenz der Factors in Relation bilden.
    Gemeinsamen Anteil in einfacher(!!) Potenz in Relationswurzel aufnehmen
  */
  mpz_set_ui(x,1); // zur Beschleunigung Faktoren in x aufsammeln, bis sie grer als n werden

  if ( GL2.Relation_dense ) /* GL2.Relation is dense */
    {
      if (!Relation_dense) // Relation ist nicht dense
	convert_Relation_to_dense(); // Relation in dichte Darstellung bringen
      
      for (int i=GL2.Relation_dense->last(); i>=0; i=GL2.Relation_dense->prev(i))
	if (Relation_dense->test(i))
	  if (i>0)
	    {
	      mpz_mul_ui(x,x,Primzahlen[i]);
	      if (mpz_cmp(x,n)>=0)
		{
		  mpz_mod(x,x,n); mpz_mul(Delta,Delta,x);
		  mpz_mod(Delta,Delta,n); mpz_set_ui(x,1);
		}
	    } //else mpz_neg(Delta,Delta);
      
      (*Relation_dense)._xor(*GL2.Relation_dense); // symmetrische Differenz bilden
    }
  else /* GL2.Relation is sparse */
    if (Relation_dense) /* and Relation is dense */
      { /* symmetrische Differenz in dense-Darstellung bilden,
	   wobei der andere Vektor sparse ist */
	for (int pos=0; pos<GL2.Relation_sparse->size(); ++pos)
	  {
	    if (Relation_dense->test((*GL2.Relation_sparse)[pos]))
	      if ((*GL2.Relation_sparse)[pos]>0)
		{
		  mpz_mul_ui(x,x,Primzahlen[(*GL2.Relation_sparse)[pos]]);
		  if (mpz_cmp(x,n)>=0)
		    {
		      mpz_mod(x,x,n); mpz_mul(Delta,Delta,x);
		      mpz_mod(Delta,Delta,n); mpz_set_ui(x,1);
		    }
		} //else mpz_neg(Delta,Delta);
	    Relation_dense->invert((*GL2.Relation_sparse)[pos]);
	  }
      }
    else
      { /* symmetrische Differenz in sparse-Darstellung bilden */
	int p1 = 0;
	int p2 = 0;
	static CTinyFBsizetypeVector neueMenge(Primbasis_size);
	neueMenge.reset();
        const CTinyFBsizetypeVector& R = *Relation_sparse;
	const CTinyFBsizetypeVector& R2 = *GL2.Relation_sparse;
	while (p2<R2.size() && p1<R.size())
	  {
	    while (R2[p2]<R[p1])
	      {
		neueMenge.fast_append(R2[p2]); ++p2;
		if (p2==R2.size()) goto fertig;
	      }
	    while (R[p1]<R2[p2])
	      {
		neueMenge.fast_append(R[p1]); ++p1;
		if (p1==R.size()) goto fertig;
	      }
	    while (R2[p2]==R[p1])
	      {
		if (R[p1]>0)
		  {
		    mpz_mul_ui(x,x,Primzahlen[R[p1]]);
		    if (mpz_cmp(x,n)>=0)
		      {
			mpz_mod(x,x,n); mpz_mul(Delta,Delta,x);
			mpz_mod(Delta,Delta,n); mpz_set_ui(x,1);
		      }
		  } //else mpz_neg(Delta,Delta);
		++p1; ++p2;
		if (p2==R2.size()) goto fertig;
		if (p1==R.size()) goto fertig;      
	      }
	  }
      fertig:
	while (p2!=R2.size()) { neueMenge.fast_append(R2[p2]); ++p2; }
	while (p1!=R.size()) { neueMenge.fast_append(R[p1]); ++p1; }
	Relation_sparse->optisized_copy_from(neueMenge); // neue Menge bernehmen
        //optisize(); // Speicherbedarf optimieren (sparse<->dense, etc.)
      }
  
  // x beinhaltet (mglicherweise) noch einen Restwert, der zu Delta multipliziert werden mu:
  mpz_mul(Delta,Delta,x); mpz_mod(Delta,Delta,n);
  
  // nicht vergessen: Das neue Delta beinhaltet natrlich auch das
  // alte Delta der anderen Relation; dieses also dazumultiplizieren!
  mpz_mul(Delta,Delta,GL2.Delta); mpz_mod(Delta,Delta,n);
  
  relevanter_Faktor=groesster_Faktor_in_Relation();
  //cout << "Kombinierung beendet." << endl;
}

// Routinen, die ein mehrfaches Kombinieren beschleunigen:
// NEU: Multikombinationsroutinen sollten das Kombinieren einer
//      Relation mit vielen anderen Relationen beschleunigen, da
//      das Update der Faktorbasis-Exponenten der Relation
//      an das Ende aller Verknpfungen geschoben werden kann und
//      nicht mehr simultan durchgefhrt werden mu.

unsigned int CRelation::Exponentenfeld[MAX_Primbasis_size] = {0};
unsigned int CRelation::multi_kombinieren_Zaehler = 0;
 
void CRelation::multi_kombinieren_init()
{
  // Initialisierung des Exponentenvektors
  // Hinweis: Der Aufruf dieser Routine darf auch entfallen,
  //          sofern sichergestellt ist, da der letzte
  //          multi_kombinieren_...-Aufruf ein  multi_kombinieren_exit() war.
  if (!multi_kombinieren_Zaehler) return;
  multi_kombinieren_Zaehler=0;
  for (int i=0; i<Primbasis_size; i++) Exponentenfeld[i]=0;
}

void CRelation::multi_kombinieren_exit()
{
  if (!multi_kombinieren_Zaehler) return; // gibt nichts zu tun...
#ifdef VERBOSE
  cout << "Multikombinieren_exit nach " << multi_kombinieren_Zaehler
       << " Aufrufen." << endl;
#endif

  optisize(); // optimize Size of vector (sparse<->dense, etc.)

  // nun die Faktor-Exponenten in das Delta ziehen:
  mpz_set_ui(y,1); // Produkt hierin sammeln
  Exponentenfeld[0]=0; // Primzahl "-1" immer unbercksichtigt lassen

  /* 
     Idee zur Beschleunigung:
     Exponentenfeld absteigend nach Exponenten sortieren, anschlieend
     das Produkt analog zum "Horner-Schema" auswerten.
     Damit das Sortieren nicht zu teuer wird, machen wir eine Art Bucketsort,
     wobei das "Flag[i]" angibt, ob es zu i einen Exponenten gibt.
     Alle zu einem Exponenten e gehrenden Basen werden dann in "Basis[e]"
     multiplikativ zusammengefat.
  */

  bool* const Flag = new bool[Primbasis_size];
  for (int i=0; i<Primbasis_size; ++i) Flag[i]=false; // initialisieren
  mpz_t* const Basis = new mpz_t[Primbasis_size];

  for (unsigned int i=1; i<Primbasis_size; ++i)
   if (Exponentenfeld[i])
    {
      const unsigned int e=Exponentenfeld[i];
      Exponentenfeld[i]=0; // Exponenten wieder zurcksetzen

      if (e>=Primbasis_size)
       {
         cout << "Unexpected exponent " << e << "! Use normal exponentiation..." << endl;
         mpz_set_ui(y,Primzahlen[i]); mpz_powm_ui(y,y,e,n);
         mpz_mul(Delta,Delta,y);
         continue;
       }

      if (Flag[e])
       {
         // Basis erweitern
         mpz_mul_ui(Basis[e],Basis[e],Primzahlen[i]);
         if (mpz_cmp(Basis[e],n)>=0)
          { 
            //cout << "Basis erweitern, mod sinnvoll..." << endl;
            mpz_mod(Basis[e],Basis[e],n);
          }
       }
      else 
       { 
         // Basis initialisieren
         Flag[e]=true;
         mpz_init(Basis[e]); mpz_set_ui(Basis[e],Primzahlen[i]);
       }
    }

 /*
  Produkt der Basen analog zum Horner-Schema auswerten,
  Beispiel: a^7 * b^4 * c^3 * d 
            = a^(7-4) * (a*b)^(4-3) * (a*b*c)^(3-1) * (a*b*c*d)
 */     

  unsigned int e2 = Primbasis_size-1;
  Flag[0]=true; // sicherheitshalber fr nachfolgende Schleife...
  while (!Flag[e2]) --e2; // hchsten gesetzten Exponenten holen

  mpz_set_ui(y,1);
  if (e2>1)
   for (unsigned int e=e2-1; e>0; --e)
    if (Flag[e])
     {
       mpz_mul(Basis[e],Basis[e],Basis[e2]);
       if (mpz_cmp(Basis[e],n)>=0)
        {
          //cout << "Auswertung A, mod sinnvoll" << endl; 
          mpz_mod(Basis[e],Basis[e],n);
        }
       if (e2-e>1) mpz_powm_ui(Basis[e2],Basis[e2],e2-e,n);
       mpz_mul(y,y,Basis[e2]);
       if (mpz_cmp(y,n)>=0)
        {
          //cout << "Auswertung B, mod sinnvoll" << endl; 
          mpz_mod(y,y,n);
        }
       mpz_clear(Basis[e2]); // do not forget: mpz-Zahl wieder freigeben
       e2=e;
     }

  if (e2>1) mpz_powm_ui(Basis[e2],Basis[e2],e2,n);
  if (e2>0)
   {
      mpz_mul(y,y,Basis[e2]); mpz_mod(y,y,n);
      mpz_clear(Basis[e2]); // do not forget: mpz-Zahl wieder freigeben
   }

  delete [] Basis; delete [] Flag;
  mpz_mul(Delta,Delta,y); mpz_mod(Delta,Delta,n);
  multi_kombinieren_Zaehler=0;
#ifdef VERBOSE
  cout << "Multikombinieren_exit beendet." << endl;
#endif
}

void CRelation::multi_kombinieren_main(const CRelation& GL2)
  // bergabe von GL2 ber Referenz, um effizienter zu sein...
  /* symmetrische Differenzen ermitteln */
{
  //cout << "Multi-Kombiniere Relationen..." << endl;
  ++multi_kombinieren_Zaehler; // Anzahl Aufrufe zhlen

  /*
    Symmetrische Differenz der Factors in Relation bilden.
    Gemeinsamen Anteil in einfacher(!!) Potenz in Exponentenfeld vermerken
  */

  if ( GL2.Relation_dense ) /* GL2.Relation is dense */
    {
      if (!Relation_dense) // Relation ist nicht dense
	convert_Relation_to_dense(); // Relation in dichte Darstellung bringen

      static myBitString BitString_Carry; // static, damit nicht immer neu allokiert werden mu...
      /* in BitString_Carry werden nun die Bits der beiden Relationen
         komponentenweise durch "and" verknpft.
         Dadurch werden die Primzahlen ermittelt, die verknpft Quadrate
         ergeben. Diese knnen dann im Exponentenfeld bercksichtigt werden.
         In gewisser Weise wird die Bitstrings komponentenweise wie
         mit einem "Halbaddierer" verknpft.
          Additionsergebnis (durch xor) berschreibt anschlieend den
          Bitstring, whrend der bertrag nach "BitString_Carry" wandert.
       */
      int i=BitString_Carry.quick_and_with_lastpos(*Relation_dense,*GL2.Relation_dense);
      while (i>=0) { ++Exponentenfeld[i]; i=BitString_Carry.prev(i); }
      (*Relation_dense)._xor(*GL2.Relation_dense); // symmetrische Differenz bilden
    }
  else /* GL2.Relation is sparse */
    if (Relation_dense) /* and Relation is dense */
      { /* symmetrische Differenz in dense-Darstellung bilden,
	   wobei der andere Vektor sparse ist */
	const CTinyFBsizetypeVector& R2 = *GL2.Relation_sparse;
	for (int pos=0; pos<R2.size(); ++pos)
	  {
	    if (Relation_dense->test(R2[pos]))
              ++Exponentenfeld[R2[pos]];
	    Relation_dense->invert(R2[pos]);
	  }
      }
    else
      { /* symmetrische Differenz in sparse-Darstellung bilden */
	int p1 = 0;
	int p2 = 0;
	static CTinyFBsizetypeVector neueMenge(Primbasis_size);
	neueMenge.reset();
	const CTinyFBsizetypeVector& R  = *Relation_sparse;
	const CTinyFBsizetypeVector& R2 = *GL2.Relation_sparse;
	while (p2<R2.size() && p1<R.size())
	  {
	    while (R2[p2]<R[p1])
	      {
		neueMenge.fast_append(R2[p2]); ++p2;
		if (p2==R2.size()) goto fertig;
	      }
	    while (R[p1]<R2[p2])
	      {
		neueMenge.fast_append(R[p1]); ++p1;
		if (p1==R.size()) goto fertig;
	      }
	    while (R2[p2]==R[p1])
	      {
		++Exponentenfeld[R[p1]];
		++p1; ++p2;
		if (p2==R2.size()) goto fertig;
		if (p1==R.size()) goto fertig;      
	      }
	  }
      fertig:
	while (p2!=R2.size()) { neueMenge.fast_append(R2[p2]); ++p2; }
	while (p1!=R.size()) { neueMenge.fast_append(R[p1]); ++p1; }
	Relation_sparse->copy_from(neueMenge); // neue Menge bernehmen
        //optisize(); // Speicherbedarf optimieren (sparse<->dense, etc.)
      }
  
  // nicht vergessen: Das neue Delta beinhaltet natrlich auch das
  // alte Delta der anderen Relation; dieses also dazumultiplizieren!
  mpz_mul(Delta,Delta,GL2.Delta); mpz_mod(Delta,Delta,n);
  
  //-------------------------------------------------------------------
  // WICHTIG: Nach dem letzten Aufruf von multi_kombinieren_main
  // mu die Relation mit dem Exponentenfeld abgeglichen werden!!!
  // Hierzu ist ein abschlieendes multi_kombinieren_exit aufzurufen!
  //-------------------------------------------------------------------

  relevanter_Faktor=groesster_Faktor_in_Relation();
  //cout << "Kombinierung beendet." << endl;
}



CmpqsFactor CRelation::kombinieren(istream &in, const streampos pos)
{
  const streampos merkfpos=in.tellg(); // ursprngliche Fileposition merken
  in.seekg(pos);
  CmpqsFactor f=kombinieren(in);
  in.seekg(merkfpos); // ursprngliche Fileposition wiederherstellen
  return f;
}

CmpqsFactor CRelation::multi_kombinieren_main(istream &in, const streampos pos)
{
  const streampos merkfpos=in.tellg(); // ursprngliche Fileposition merken
  in.seekg(pos);
  CmpqsFactor f=multi_kombinieren_main(in);
  in.seekg(merkfpos); // ursprngliche Fileposition wiederherstellen
  return f;
}


int hole_Wert (istream &in)
{
  static int ahead_value = 0;
  if (ahead_value>0) { int w=ahead_value; ahead_value=0; return w; }

  char c;
  in >> c;
  if ( c>='A' && c<='Z' ) return c-('A'-1); // A=1,...,Z=26
  if ( c>='g' && c<='z' )
   {
     int h = (c-'g');
     ahead_value=(h%5)+1; // nchster "vorausgelesener" Wert
     return (h/5)+1; 
   }
  if (c=='0') return 0; // da keine fhrenden Nullen
  if (c=='$') return -1;
  if (c=='%') return -2;

  if (c==' ')
   {
     cerr << "hole_Wert: Leerzeichen gelesen? How to handle this?" << endl;
     exit(1);
   }

  // ansonsten normal gespeicherte Zahl:
  int w;
  in.unget();
  in >> w;
  return w;
}

void schreibe_Wert (ostream &out, const int w)
{
  static int prev_value = 0;
  if (w<=5 && w>=1 && prev_value>0) // dann kann kodiert werden
    {
      out << char( (prev_value-1)*5 + w-1 + 'g' );
      prev_value=0;
      return;
    }

  if (prev_value>0)
   {
     out << char(prev_value+('A'-1)); // Wert ausgeben
     prev_value=0;
   }

  if (w>26) 
   {
     out << w << " "; // Wert unkodiert ausgeben
     return;
   }
  if (w>0)
   {
     if (w<=4) { prev_value=w; return; } // nchstes Mal abwarten
     out << char(w+('A'-1)); // Wert als char kodiert ausgeben
     return;
   }
  switch (w)
   {
     case  0: out << "0"; break;
     case -1: out << "$"; break;
     case -2: out << "%"; break;
     default:
      cerr << "schreibe_Wert: don't know how to handle this..." << endl;
      cerr << "w=" << w << endl;
      exit(1);
   }
}

CmpqsFactor CRelation::kombinieren(istream &in)
{
  //cout << "kombiniere Gleichungszeile vom File" << endl;
  CmpqsFactor ret;
  string s;
  int index;
 
  // Dateiformatflags retten
  const ios::fmtflags oldFlags = in.flags(); // alte Formatflags retten
  in.unsetf(ios::hex); in.setf(ios::dec); // Dezimalausgabe sicherstellen (beachte: Rekursiver Aufruf in Lazy-Variante wegen Korrekturfaktoren mglich!)
  in.unsetf(ios::showbase); // ohne Basiskennzeichnung
 
  in >> s; // Startzeichen
  if (s != "G")
    {
      MARK;
      cerr << "error: wrong position while reading relation" << endl;
      exit(1);
    }
  in >> ret; // Faktor einlesen (dynamic-factor oder special-factor der Relation oder 1)
  in >> s; // delta
  mpz_set_str(x, s.c_str(), mpzbase_f); // x ist klassenweite Hilfsvariable
  mpz_mul(Delta, Delta, x);
  mpz_mod(Delta, Delta, n);

  // ab jetzt wird hexadezimal gearbeitet:
  in.unsetf(ios::dec); in.setf(ios::hex); // Hexadezimalausgabe
  in.unsetf(ios::showbase); // ohne Basiskennzeichnung
  
  /*
    Symmetrische Differenz der Factors in Relation bilden.
    Gemeinsamen Anteil in einfacher(!!) Potenz in Relationswurzel aufnehmen
  */
  mpz_set_ui(x,1); // zur Beschleunigung Faktoren in x aufsammeln, bis sie grer als n werden
  int distance; // Hinweis: statt der Werte werden deren Abstnde eingelesen...

  if (Relation_dense) /* Relation is dense */
    { /* symmetrische Differenz in dense-Darstellung bilden */
      distance=hole_Wert(in); index=distance;
      while (distance >= 0)
	{
	  if (Relation_dense->test(index))
	    if (index>0)
	      {
		mpz_mul_ui(x,x,Primzahlen[index]);
		if (mpz_cmp(x,n)>=0)
		  {
		    mpz_mod(x,x,n); mpz_mul(Delta,Delta,x);
		    mpz_mod(Delta,Delta,n); mpz_set_ui(x,1);
		  }
	      } //else mpz_neg(Delta,Delta);
	  Relation_dense->invert(index);
	  distance=hole_Wert(in); index+=distance;
	}
    }
  else
    { /* symmetrische Differenz in sparse-Darstellung bilden */
      int p1 = 0;
      static CTinyFBsizetypeVector neueMenge(Primbasis_size);
      neueMenge.reset(); // "leeren" ohne Speicherfreigabe
      const CTinyFBsizetypeVector& R = *Relation_sparse;
      
      distance=hole_Wert(in); index=distance; 
      while (distance>=0 && p1<R.size())
	{
	  while (index<R[p1])
	    {
	      neueMenge.append(index); distance=hole_Wert(in); index+=distance;
	      if (distance<0) goto fertig;
	    }
	  while (R[p1]<index)
	    {
	      neueMenge.append(R[p1]); ++p1;
	      if (p1==R.size()) goto fertig;
	    }
	  while (index==R[p1])
	    {
	      if (index>0)
		{
		  mpz_mul_ui(x,x,Primzahlen[index]);
		  if (mpz_cmp(x,n)>=0)
		    {
		      mpz_mod(x,x,n); mpz_mul(Delta,Delta,x);
		      mpz_mod(Delta,Delta,n); mpz_set_ui(x,1);
		    }
		} //else mpz_neg(Delta,Delta);
	      distance=hole_Wert(in); index+=distance; ++p1;
	      if (distance<0) goto fertig;
	      if (p1==R.size()) goto fertig;      
	    }
	}
    fertig:
      while (distance>=0) { neueMenge.append(index); distance=hole_Wert(in); index+=distance; }
      while (p1!=R.size()) { neueMenge.append(R[p1]); ++p1; }
      Relation_sparse->optisized_copy_from(neueMenge); // neue Menge bernehmen
      //optisize(); // Speicherbedarf optimieren (sparse<->dense, etc.)
    }
  
  // x beinhaltet (mglicherweise) noch einen Restwert, der zu Delta multipliziert werden mu:
  mpz_mul(Delta,Delta,x); mpz_mod(Delta,Delta,n);
 
  if (distance == -2) // Korrekturfaktoren
    {
      /* Korrekturfaktoren sind dynamische Faktoren, die zum Zeitpunkt der Generierung
	 der gelesenen Relation nicht aufgelst wurden. Sie mssen daher jetzt verknpft werden. */
      TDynamicFactorRelation FaRelSearch;
      stack<int> Korrekturfaktoren; // Breitensuche, um dauerndes seeken zu vermeiden
      while ( (FaRelSearch.factor=hole_Wert(in))>0 ) Korrekturfaktoren.push(FaRelSearch.factor);
       
      while (!Korrekturfaktoren.empty())
	{
	  FaRelSearch.factor=Korrekturfaktoren.top(); Korrekturfaktoren.pop();
	  //cout << "Korrekturfaktor: " << FaRelSearch.factor << " fr " << setprecision(20) << ret << endl;
	  TDynamic_Factor_Relations::iterator p = Dynamic_Factor_Relations.find(FaRelSearch);
	  if (p==Dynamic_Factor_Relations.end())
	    {
	      cerr << "Unbekannter Korrekturfaktor " << FaRelSearch.factor << endl;
	      exit(1); // sicherheitshalber fr die partiell korrekte Variante...
	      ret=0; break; // Weiterarbeiten trotzdem ermglichen... 
	    }
	  mpz_mul_ui(Delta,Delta,(*p).factor); mpz_mod(Delta,Delta,n);
	  if (kombinieren(DynamicRelations_from_file,(*p).fpos).Typ()==CmpqsFactor::empty) { ret=0; break; } // auf Hintergrundspeicher kombinieren und bei Fehler ret=0 setzen
	  //cout << "fertig kombiniert von " << FaRelSearch.factor << endl;
	}
    }
  
  in.flags(oldFlags); // alte Formatflags des Streams wiederherstellen
 
  relevanter_Faktor=groesster_Faktor_in_Relation();
  //cout << "Kombinierung beendet." << endl;
  return ret;
}

CmpqsFactor CRelation::multi_kombinieren_main(istream &in)
{
  //cout << "Multi-Kombiniere Gleichungszeilen vom File..." << endl;
  ++multi_kombinieren_Zaehler; // Anzahl Aufrufe zhlen

  CmpqsFactor ret;
  string s;
  int index;
 
  // Dateiformatflags retten
  const ios::fmtflags oldFlags = in.flags(); // alte Formatflags retten
  in.unsetf(ios::hex); in.setf(ios::dec); // Dezimalausgabe sicherstellen (beachte: Rekursiver Aufruf in Lazy-Variante wegen Korrekturfaktoren mglich!)
  in.unsetf(ios::showbase); // ohne Basiskennzeichnung
 
  in >> s; // Startzeichen
  if (s != "G")
    {
      MARK;
      cerr << "error: wrong position while reading relation" << endl;
      exit(1);
    }
  in >> ret; // Faktor einlesen (dynamic-factor oder special-factor der Relation oder 1)
  in >> s; // delta
  mpz_set_str(x, s.c_str(), mpzbase_f); // x ist klassenweite Hilfsvariable
  mpz_mul(Delta, Delta, x);
  mpz_mod(Delta, Delta, n);

  // ab jetzt wird hexadezimal gearbeitet:
  in.unsetf(ios::dec); in.setf(ios::hex); // Hexadezimalausgabe
  in.unsetf(ios::showbase); // ohne Basiskennzeichnung
  
  /*
    Symmetrische Differenz der Factors in Relation bilden.
    Gemeinsamen Anteil in einfacher(!!) Potenz in Relationswurzel aufnehmen
  */
  int distance; // Hinweis: statt der Werte werden deren Abstnde eingelesen...

  if (Relation_dense) /* Relation is dense */
    { /* symmetrische Differenz in dense-Darstellung bilden */
      distance=hole_Wert(in); index=distance;
      while (distance >= 0)
	{
	  if (Relation_dense->test(index)) ++Exponentenfeld[index];
	  Relation_dense->invert(index);
	  distance=hole_Wert(in); index+=distance;
	}
    }
  else
    { /* symmetrische Differenz in sparse-Darstellung bilden */
      int p1 = 0;
      static CTinyFBsizetypeVector neueMenge(Primbasis_size);
      neueMenge.reset(); // "leeren" ohne Speicherfreigabe
      const CTinyFBsizetypeVector& R = *Relation_sparse;
      
      distance=hole_Wert(in); index=distance; 
      while (distance>=0 && p1<R.size())
	{
	  while (index<R[p1])
	    {
	      neueMenge.append(index); distance=hole_Wert(in); index+=distance;
	      if (distance<0) goto fertig;
	    }
	  while (R[p1]<index)
	    {
	      neueMenge.append(R[p1]); ++p1;
	      if (p1==R.size()) goto fertig;
	    }
	  while (index==R[p1])
	    {
	      ++Exponentenfeld[index];
	      distance=hole_Wert(in); index+=distance; ++p1;
	      if (distance<0) goto fertig;
	      if (p1==R.size()) goto fertig;      
	    }
	}
    fertig:
      while (distance>=0) { neueMenge.append(index); distance=hole_Wert(in); index+=distance; }
      while (p1!=R.size()) { neueMenge.append(R[p1]); ++p1; }
      Relation_sparse->optisized_copy_from(neueMenge); // neue Menge bernehmen
      //optisize(); // Speicherbedarf optimieren (sparse<->dense, etc.)
    }

  if (distance == -2) // Korrekturfaktoren
    {
      /* Korrekturfaktoren sind dynamische Faktoren, die zum Zeitpunkt der Generierung
	 der gelesenen Relation nicht aufgelst wurden. Sie mssen daher jetzt verknpft werden. */

      TDynamicFactorRelation FaRelSearch;
      stack<int> Korrekturfaktoren; // Breitensuche, um dauerndes seeken zu vermeiden
      while ( (FaRelSearch.factor=hole_Wert(in))>0 ) Korrekturfaktoren.push(FaRelSearch.factor);
       
      while (!Korrekturfaktoren.empty())
	{
	  FaRelSearch.factor=Korrekturfaktoren.top(); Korrekturfaktoren.pop();
	  //cout << "Korrekturfaktor: " << FaRelSearch.factor << " fr " << setprecision(20) << ret << endl;
	  TDynamic_Factor_Relations::iterator p = Dynamic_Factor_Relations.find(FaRelSearch);
	  if (p==Dynamic_Factor_Relations.end())
	    {
	      cerr << "Unbekannter Korrekturfaktor " << FaRelSearch.factor << endl;
              exit(1); // fr die partiell korrekte Variante...
	      ret=0; break; // Weiterarbeiten trotzdem ermglichen... 
	    }
	  mpz_mul_ui(Delta,Delta,(*p).factor); mpz_mod(Delta,Delta,n);
	  if (multi_kombinieren_main(DynamicRelations_from_file,(*p).fpos).Typ()==CmpqsFactor::empty) { ret=0; break; } // auf Hintergrundspeicher kombinieren; im Fehlerfall ret=0 setzen
	  //cout << "fertig kombiniert von " << FaRelSearch.factor << endl;
	}
    }
  
  in.flags(oldFlags); // alte Formatflags des Streams wiederherstellen
 
  relevanter_Faktor=groesster_Faktor_in_Relation();
  //cout << "Kombinierung beendet." << endl;
  return ret;
}

streampos CRelation::speichern(ostream &out, const CmpqsFactor factor, const short int HitCount /* = 0 */) const
{
  //cout << "Speichern einer Gleichungszeile" << endl;
#ifndef IS_CLIENT
  streampos pos;
  // out.seekp(0,ios::end);  // sollte jetzt schon am Ende sein
  pos = out.tellp(); // fileposition ans Ende setzen und auslesen
#endif
  char *str;

  str = new char [mpz_sizeinbase(Delta,mpzbase_f)+2]; mpz_get_str(str,mpzbase_f, Delta);
  out << "G " << setprecision(20) << factor << " " << str << " ";
  delete [] str;

  // ab hier wird hexadezimal gearbeitet:
  const ios::fmtflags oldFlags = out.flags(); // alte Formatflags retten
  out << hex; // Hexadezimalausgabe
  out.unsetf(ios::showbase); // ohne Basiskennzeichnung
  
  /* Bemerkung:
     Statt der Werte werden die Abstnde gespeichert, das spart Platz */
  if (Relation_dense)
    { 
      int prev = 0;
      for (int i=Relation_dense->first(); i>=0; i=Relation_dense->next(i))
	{
          schreibe_Wert(out,i-prev);
	  prev=i;
	} 
    }
  else
    {
      int prev = 0;
      for (int i=0; i<Relation_sparse->size(); ++i)
	{
 	  int h = (*Relation_sparse)[i];
          schreibe_Wert(out,h-prev);
	  prev=h;
	}
    }

  int Anzahl_echte_Faktoren = 0;
  for (int i=0; i<HitCount; i++) if (Hits[i].Exponent>0) Anzahl_echte_Faktoren++;
  if (Anzahl_echte_Faktoren>0)
    {
      schreibe_Wert(out,-2);
      //cout << "Korrekturfaktoren fr Relation mit Faktor " << setprecision(20) << factor << endl;
      for (int i=0; i<HitCount; i++)
	{ 
	  //cout << Hits[i].Faktor << "^" << Hits[i].Exponent << endl; 
	  if (Hits[i].Exponent>0) schreibe_Wert(out,Hits[i].Faktor);
	}
    }
  schreibe_Wert(out,-1); out << endl;
  out.flags(oldFlags); // alte Formatflags wiederherstellen
#ifdef IS_CLIENT
  return 0;
#else
  return pos;
#endif
}

void CRelation::Kongruenzcheck() const
{
  mpz_set_ui(y,1); 
  if (Relation_sparse)
    {
      for (int i=0; i<Relation_sparse->size(); ++i)
	{
	  if ((*Relation_sparse)[i]>0) 
	    mpz_mul_ui(y,y,Primzahlen[(*Relation_sparse)[i]]);
	  else mpz_neg(y,y);
	  mpz_mod(y,y,n);
	}
    }
  else
    {
      for (int i=Relation_dense->first(); i>=0; i=Relation_dense->next(i))
	{
	  if (i>0) mpz_mul_ui(y,y,Primzahlen[i]); else mpz_neg(y,y);
	  mpz_mod(y,y,n);
	}
    }
  
  mpz_invert(x,Delta,n); mpz_powm_ui(x,x,2,n);
  
  cout << "Kongruenzcheck: ";
  cout << x << " = " << y << " (mod N)" << endl;
  if (mpz_cmp(x,y)!=0) 
    {
      cout << "Error! Values are NOT congruent!" << endl;
      exit(1);
    }
  
}

#ifndef IS_CLIENT
void CRelation::Quadratische_Kongruenz_ausrechnen() const
{
  /* Die ermittelte Relation erfllt die Kongruenz (A=1) mod n.
     Wegen A-1=(A+1)*(A-1)=0 mod n knnen wir daher nun durch ggT(A+1,n)
     einen Teiler von n ermitteln und haben gute Chancen, da es sich dabei
     auch um einen echten (also nichttrivialen) Teiler handelt... */

  cout << endl << "Found a suitable relation..." << endl;

  if (!Relation_ist_leer()) cout << "Error! Relation contains unresolved members!" << endl;

  cout << "[" << Delta << "] = 1 (mod N)" << endl;
   
  mpz_powm_ui(x,Delta,2,n);
  cout << "-> " << x << " = 1 (mod N)" << endl;
  
  if (mpz_cmp_ui(x,1)!=0) 
    {
      cout << "Error! Squares are NOT congruent!" << endl;
      exit(1);
    }
  
  mpz_add_ui(x,Delta,1);
  mpz_gcd(x,x,n); mpz_divexact(y,n,x); /* beachte: n und nicht kN
				          ist die eigentlich zu
				          faktorisierende Zahl! */
  
  if ((mpz_cmp_ui(x,1)==0) || (mpz_cmp_ui(y,1)==0))
    {
      cout << "factorization was trivial..." << endl;
    }
  else
    {
      cout << endl << "factor found!" << endl;
      cout << x << " * ";
      cout << y << endl;
      cout << "= ";
      cout << n << endl;
      
      if (mpz_probab_prime_p(x,probab_prime_checks))
	{
	  mpz_divexact(n,n,x);
	  cout << x << " is a factor." << endl;
	  Factorization_to_file << MAL() << x << flush;
	}
      else
      	if (Potenztest(x)) mpz_divexact(n,n,x); // ggf. Potenzen streichen...

      if (mpz_probab_prime_p(y,probab_prime_checks))
	{
	  mpz_divexact(n,n,y);
	  cout << y << " is a factor." << endl;
	  Factorization_to_file << MAL() << y << flush;
	}
      else
	if (Potenztest(y)) mpz_divexact(n,n,y); // ggf. Potenzen streichen...
      

      if (mpz_cmp_ui(n,1)==0) // Faktorisierung vollstndig?
	{ 
	  cout << "factorizition successfully completed." << endl;
          Factorization_to_file << endl;
	  
	  // Streams schlieen
          SpecialRelations_buffer.close();
	  DynamicRelations_buffer.close();
	  StaticRelations_buffer.close();
	  Recovery_buffer.close();

	  // Files lschen
	  remove(SpecialRelationsFile.c_str());
	  remove(DynamicRelationsFile.c_str());
	  remove(StaticRelationsFile.c_str());
	  remove(RecoveryFile.c_str());
	  
	  exit(0); // Programm erfolgreich beenden...
	}
    }
  cout << "continue factorization..." << endl;
}
#endif
  

void CRelation::Relation_optimieren(int bitpos /* =1 */)
{ 
  /* optimiere ab bitpos:
     bitpos = 0 -> letzter Faktor der Relation (Vorsicht: wenn dieselbe
                   Relation gerade eingefgt wurde, ergibt eine erneute
                   Kombination eine leere Relation!
     bitpos = 1 -> vorletzter Faktor
     bitpos = 2 -> vorvorletzter Faktor, usw.
  */
  cout << "Optimiere " << groesster_Faktor_in_Relation() << endl;

  multi_kombinieren_init();
  while (true)
    {
      signed int vorletzter_Faktor;
      if (Relation_sparse) 
	if (Relation_sparse->size()>bitpos)
	  vorletzter_Faktor=(*Relation_sparse)[Relation_sparse->size()-1-bitpos];
	else break;
      else
	{
	  vorletzter_Faktor=Relation_dense->last(1);
	  for (int j=0; j<bitpos && vorletzter_Faktor>=0; j++)
	    vorletzter_Faktor=Relation_dense->prev(vorletzter_Faktor,1);
	  if (vorletzter_Faktor<0) break;
	}
      if (GLS[vorletzter_Faktor]==NULL) { bitpos++; continue; }
      cout << "Optimiere " << groesster_Faktor_in_Relation()
           << " mit " << vorletzter_Faktor << endl;
      multi_kombinieren_main(*GLS[vorletzter_Faktor]); 
    }
  multi_kombinieren_exit();
}


void Gleichungssystem_optimieren()
{
  /* Mglichst viele redundante Faktoren aus den Relationen entfernen.
     Dadurch dafr sorgen, da die Relationen mglichst dnn mit Faktoren
     besetzt sind.
     -> Nachteil: bei dicht besetztem Gleichungssystem hohe Laufzeit! */
  cout << "Optimiere Gleichungssystem" << endl;
  for (int i=1; i<Primbasis_size; i++) if (GLS[i]) 
    {
      cout << " GL " << i << ", " << flush;
      GLS[i]->Relation_optimieren();
    }
  cout << endl;
}


#include <ctime>

time_t StartZeitpunkt; // Zeitpunkt, zu dem das Sieben gestartet wurde
time_t PreviousZeitpunkt; // Zeitpunkt der vorigen Meldung
double StartFuellung; // Fllung des GLS bei Start des Aussiebens

void Restzeit_kalkulieren(void)
{
  // eigentlich wird hier weniger die Restzeit abgeschtzt als
  // vielmehr die durchschnittlich bentigte Zeit zum Sieben einer
  // Relation ermittelt...
  // Fr eine Restzeitabschtzung mte man das Zeitverhalten eigentlich
  // extrapolieren (und zustzlich die Schwankungen zwischen den
  // "Zwischenankunftszeiten" der Relationen ausmitteln).
  // Dies wre aber ein hoffnungsloses Unterfangen, da die Systemlast
  // schwankt und (im Serverbetrieb) die Anzahl der siebenden Clients
  // unbekannt ist (und zudem auch die Fileclients die Ergebnisse
  // verflschen wrden).
  if (Fuellung_GLS==StartFuellung) return;
  double t=difftime(time(NULL),StartZeitpunkt);
  t/=(Fuellung_GLS-StartFuellung);
  double Restzeit=(Primbasis_size-Fuellung_GLS)*t;
  double days = floor(Restzeit/(60*60*24));
  int hours,mins,secs;
  Restzeit=fmod(Restzeit,60*60*24);
  secs=static_cast<int>(Restzeit); mins=secs/60; hours=mins/60;
  mins%=60; secs%=60;
  if (days)
   cout << "time left (estimated): " << days << "d " << hours << "h" << endl;
  else
   cout << "time left (estimated): " << hours << "h " << mins << "m "
        << secs << "s" << endl;
}

void Statuslegende_ausgeben(void)
{
  cout << "Bedeutung der nachfolgenden Statusinformationen:" << endl;
#ifndef USE_NETWORK 
  cout << "#Relationen{Verteilung}/Faktorbasis,#dynamic-FB,#Special-Factors(Verteilung)"
       << " [Siebpos] -> Fortschritt" << endl;
#else
  cout << "#Relationen{Verteilung}/Faktorbasis,#dynamic-FB,#Special-Factors(Verteilung)"
       << " -> Fortschritt" << endl;
#ifndef IS_CLIENT
 cout << "Hinweis: Verteilungen = vom Server vorgenommene Verteilungen." << endl;
 cout << "         Extern vorgenomme Verteilungen bleiben unbercksichtigt." << endl;
#endif
#endif
 StartZeitpunkt=time(NULL); StartFuellung=Fuellung_GLS; // Initialisierung fr Restzeitberechnung
 PreviousZeitpunkt=StartZeitpunkt; // Zeitpunkt dieser Ausgabe merken
}

bool Statusmeldung(const bool force_now = false)
{
  /* 
     Statusmeldung ausgeben, wenn seit voriger Statusmeldung/Statuslegende
     mindestens 10 Sekunden vergangen sind.
     Der Funktionswert ist true, falls eine Ausgabe erfolgte, ansonsten false
   */ 
  if (!force_now && difftime(time(NULL),PreviousZeitpunkt)<10.0) return false;
  // aktuellen Stand ausgeben  
  cout << Fuellung_GLS
       << "{" << Erfolgsaufteilung[0]
       << "," << Erfolgsaufteilung[1]
       << "," << Erfolgsaufteilung[2]
       << "," << Erfolgsaufteilung[3]
       << "," << Erfolgsaufteilung[4]
       << "," << Erfolgsaufteilung[5]
       << "," << Erfolgsaufteilung[6]
       << "}/" << Primbasis_size << ","
       << Dynamic_Factor_Relations.size() << ","
       << Special_Factor_Relations.size()
       << "(" << Special_to_dynamic_Factor_hit << "," << Special_hit
       << "," << gefundene_DLP_Zyklen << ")";
#ifndef USE_NETWORK
  // Sieboffset macht nur bei standalone-Version Sinn
  cout << " [" << Sieboffset << "]";
#endif
  cout.setf(ios::left, ios::adjustfield);
  cout << " -> ca. " << setw(6) << setprecision(5)
       << 100.0*Fuellung_GLS/Primbasis_size
       << "%  " << endl;
  PreviousZeitpunkt=time(NULL); // Zeitpunkt dieser Ausgabe merken
#ifndef IS_CLIENT
  Restzeit_kalkulieren();
#endif
  return true;
}


void Relation_einfuegen(CRelation *GL, const bool do_multi_kombinieren_init /* =true */)
{ /* hier wird eine neu gefundene Relation in das bisherige Gleichungssystem
     eingefgt. Falls sie redundant ist bzw. zu einer Faktorisierung fhrt,
     so wird dies automatisch erkannt. */
#ifdef IS_CLIENT
  if (GL->relevanter_Faktor >= 0) Fuellung_GLS++; // beim Client mangels Gleichungssystem: jede gefundene Relation
  delete GL;
  return;
#else  
  if (GL->relevanter_Faktor==dynamic_factor || GL->relevanter_Faktor==special_factor)
    { delete GL; return; } /* diese Relationen sind auf Hintergrundspeicher ausgelagert
			      und brauchen deshalb nicht im Hauptspeicher zu verbleiben */
  
  if (do_multi_kombinieren_init) GL->multi_kombinieren_init();
  while (!GL->Relation_ist_leer())
    {
      if (GL->relevanter_Faktor>=Primbasis_size) cerr << "zu grosser Wert!!" << endl;
      CRelation* pos = GLS[GL->relevanter_Faktor];
      if ( pos!=NULL )
	{ /* Relation mit diesem Eintrag existiert schon -> neue Kombination mglich! */
#if 1 /* Minimierung der Relationen ein-/ausschalten */
          if (
	       (GL->Relation_sparse) && (pos->Relation_sparse) // beide in Sparse-Darstellung
               && GL->SizeOfRelation()<=pos->SizeOfRelation()
	       &&GL->zweitgroesster_Faktor_in_Relation()<=pos->zweitgroesster_Faktor_in_Relation()
             )
	    { /* weniger Primzahlen in der Relation bedeuten schnelleres Abarbeiten.
	  	 Da beide Relationen quivalent sind, knnen sie problemlos so
		 getauscht werden, da die platzsparende im Gleichungssystem verbleibt.
		 Andererseits bestimmt das Maximum der zweitgrten Faktoren zweier
		 Relationen die nchste Verknpfung. Also ist auch dies zu minimieren.
                 Beim Zielkonflikt wren beide Kriterien also abzuwgen. Dies geschieht
		 hier nicht. Nur wenn beide Kriterien zugleich die Lage nicht
                 verschlechtern, werden die Relationen getauscht.

                 Man beachte, da der Tausch das Ergebnis der aktuellen Verknpfung
                 nicht verndert, sondern nur zuknftige Verknpfungen
	         beschleunigen kann bzw. den Speicherbedarf vermindert.
              */
	      GL->multi_kombinieren_exit(); 
	      GLS[GL->relevanter_Faktor]=GL;
	      register CRelation* h = pos; pos=GL; GL=h; // Relationen tauschen
              // GL->multi_kombinieren_init(); <- darf entfallen
	    }
#endif
          // grter Faktor mu eliminiert werden:
 	  GL->multi_kombinieren_main(*pos); 
	}
      else
	{
	  GL->multi_kombinieren_exit();
	  GLS[GL->relevanter_Faktor]=GL; // Relation ins Gleichungssystem eintragen
	  Fuellung_GLS++; // eine Relation mehr im Gleichungssystem
          GL->speichern(StaticRelations_to_file);
          Statusmeldung();
	  return;
	}
    }

  GL->multi_kombinieren_exit();

  // Relation redundant -> Faktorisierung probieren!
  Statusmeldung();
  GL->Quadratische_Kongruenz_ausrechnen();
  delete GL;
#endif
}

const bool SiebRegler::Hit_after_DirtyKorrektur(const signed int Siebpos, TSiebElement Value)
{
  // An dieser Stelle im Sieb wurde ein Treffer gefunden;
  // aber wegen der Dirty-Siebfaktoren, die fr alle Siebelemente als Treffer gezhlt werden,
  // knnte es tatschlich gar kein Treffer sein.
  // Wir mssen daher den Dirty-Siebwert um die nicht-teilenden Dirty-Faktorbasiselemente korrigeren.
  // Dieser Korrekturwert wird hier bestimmt (allerdings mit shortcut).

  // Wenn diese Rotine aufgerufen wird, ist mglicherweise ein Siebtreffer gelandet worden;
  // Da es sich um ein Regelungssystem handelt, ist dies (zwar kein perfekter, aber)
  // ein sinnvoller Indikator, um nun wieder etwas weniger Dirtyness zu verlangen,
  // wenn doch kein Siebtreffer vorlag...

  for (int i=_sieben_ab_Primzahlindex-1; i>=PrimzahlStartIndex; --i)
   {
     register int d = normalized_signed_mod(Siebpos,Primzahlen[i]);
     // same as: Siebpos%Primzahlen[i]; if (d<0) d+=Primzahlen[i];
     // Bemerkung: %-operation does *not* eliminate the sign!
     //            ugly: -1=1 (mod 2), but (-1)%2=-1, 1%2=1
     // -> so we have to eliminate the sign!!!
     if (d!=Delta_der_Primzahlen[i][0] && d!=Delta_der_Primzahlen[i][1])
      {
        Value+=log_Primzahlen[i];
        if (Value>=0)
         {
           SiebRegler::RequestLessDirtyness(); // "gegenregeln"
	   return false; // Threshold berschritten, kein Siebtreffer
         }
      }
   }
  return true; // Threshold nicht berschritten -> Siebtreffer
}


TSiebElement SiebRegler::log_Threshold = 0; // wird aber erst durch "Grundinitialisierung_des_Siebs" initialisiert

void SiebRegler::SiebSchwellwert_kalkulieren(void)
{
  // Hier wird der Threshold (berschlagsmig) ermittelt
  
  // Voraussetungen zum Aufruf der Initialisierung:
  // 1. Logische Siebintervallgre M (fr Intervall [-M,M])
  // 2. kN (die mit Vorfaktor versehene zu faktorisierende Zahl)
  // 3. die statische Faktorbasis
  // mssen bereits bekannt und berechnet sein.
 
  // logarithmischen Schwellwert kalkulieren:
  mpz_t x; mpz_init(x);
  mpz_div_ui(x,kN,2); mpz_sqrt(x,kN); mpz_mul_ui(x,x,M);
  double d = log(fabs(mpz_get_d(x))) - (Factor_Threshold*log(biggest_Prime_in_Factorbase));
  mpz_clear(x);
  if (Primzahlen[1]==2) d-=log(2); // Falls 2 in der Faktorbasis, dann teilt 2 alle Siebeintrge
  d*=log_Multiplikator;
  cout << "Der Threshold zum Erkennen einer Relation betrgt "<< d << endl;
  log_Threshold=static_cast<TSiebElement>(floor(d));
  if (log_Threshold!=floor(d)) 
   {
     cerr << "Mit den momentan einkompilierten Typen/Konstanten kann diese" << endl;
     cerr << "Faktorisierung leider nicht durchgefhrt werden." << endl; 
     cerr << "TSiebElement ist zu klein fr Threshold!" << endl;
     cerr << "Bitte modifizieren Sie log_Multiplikator und/oder TSiebElement" << endl;
     cerr << "und kompilieren Sie dann das Programm erneut." << endl;
     exit(1);
   }

#if TSIEBELEMENTSIZE > 0
  // sanity check...
  if (sizeof(TSiebElement)!=TSIEBELEMENTSIZE)
   {
     cerr << "Die einkompilierten Konstanten/Typen fr " << endl
          << "TSiebElement (Size=" << sizeof(TSiebElement) << ")"
          << " und TSIEBELEMENTSIZE (=" << TSIEBELEMENTSIZE << ")" << endl
          << "stimmen nicht berein. Bitte korrigieren!" << endl;
     exit(1);
   }
#endif
}


#ifndef IS_SERVER

#ifdef ASM386
#define SIEBASM386

unsigned int clobbered_int; // dummy

#if TSIEBELEMENTSIZE == 1
#define asm_sieb_init() \
  asm volatile ( \
       "movb %%al,%%ah\n\t" \
       "pushw %%ax\n\t" \
       "shll $16,%%eax\n\t" \
       "popw %%ax\n\t" \
       "shrl $2,%%ecx\n\t" \
       "cld\n\t" \
       "rep\n\t" \
       "stosl" \
       : "=c" (clobbered_int), "=D" (clobbered_int) : "a" (SiebRegler::log_Threshold), "D" (Sieb), "c" (Siebgroesse) : "memory");

#ifdef ASMATHLON
// ATHLON-optimized code
#warning "using ATHLON version for macro: asm_search_sieb"
#define asm_search_sieb(offset) \
  asm volatile ( \
       "0: # Alignment suchen \n\t" \
       "testl $3,%[pos] \n\t" \
       "jz 1f # is aligned \n\t" \
       "decl %[pos] \n\t" \
       "testb $0x80,(%[sieb],%[pos]) \n\t" \
       "jns 0b \n\t" \
       "js 4f # Treffer gefunden \n\t" \
       "1: # Treffer suchen (big steps) loop \n\t" \
       "prefetch -256(%[sieb],%[pos]) # tuning cache!! \n\t" \
       "subl $64,%[pos] \n\t" \
       "movl 60(%[sieb],%[pos]),%%eax \n\t" \
       "orl 56(%[sieb],%[pos]),%%eax \n\t" \
       "orl 52(%[sieb],%[pos]),%%eax \n\t" \
       "orl 48(%[sieb],%[pos]),%%eax \n\t" \
       "orl 44(%[sieb],%[pos]),%%eax \n\t" \
       "orl 40(%[sieb],%[pos]),%%eax \n\t" \
       "orl 36(%[sieb],%[pos]),%%eax \n\t" \
       "orl 32(%[sieb],%[pos]),%%eax \n\t" \
       "orl 28(%[sieb],%[pos]),%%eax \n\t" \
       "orl 24(%[sieb],%[pos]),%%eax \n\t" \
       "orl 20(%[sieb],%[pos]),%%eax \n\t" \
       "orl 16(%[sieb],%[pos]),%%eax \n\t" \
       "orl 12(%[sieb],%[pos]),%%eax \n\t" \
       "orl  8(%[sieb],%[pos]),%%eax \n\t" \
       "orl  4(%[sieb],%[pos]),%%eax \n\t" \
       "orl   (%[sieb],%[pos]),%%eax \n\t" \
       "andl $80808080,%%eax \n\t" \
       "jz 1b \n\t" \
       "2: # Treffer suchen (little steps) init \n\t" \
       "addl $64,%[pos] \n\t" \
       "3: # Treffer suchen (little steps) loop \n\t" \
       "subl $4,%[pos] \n\t" \
       "testl $0x80808080,(%[sieb],%[pos]) \n\t" \
       "jz 3b \n\t" \
       "movl (%[sieb],%[pos]),%%eax \n\t" \
       "addl $3,%[pos] \n\t" \
       "testl $0x80000000,%%eax \n\t" \
       "jnz 4f \n\t" \
       "decl %[pos] \n\t" \
       "testl $0x00800000,%%eax \n\t" \
       "jnz 4f \n\t" \
       "decl %[pos] \n\t" \
       "testl $0x00008000,%%eax \n\t" \
       "jnz 4f \n\t" \
       "decl %[pos] \n\t" \
       "4: # Treffer gefunden" \
       : [pos] "=q" (offset) : "[pos]" (offset), [sieb] "r" (Sieb) : "eax");
#else
// optimized i386 code
#define asm_search_sieb(offset) \
  asm volatile ( \
       "0: # Alignment suchen \n\t" \
       "testl $3,%[pos] \n\t" \
       "jz 1f # is aligned \n\t" \
       "decl %[pos] \n\t" \
       "testb $0x80,(%[sieb],%[pos]) \n\t" \
       "jns 0b \n\t" \
       "js 2f # Treffer gefunden \n\t" \
       "1: # Treffer suchen \n\t" \
       "subl $4,%[pos] \n\t" \
       "testl $0x80808080,(%[sieb],%[pos]) \n\t" \
       "jz 1b \n\t" \
       "addl $3,%[pos] \n\t" \
       "testb $0x80,(%[sieb],%[pos]) \n\t" \
       "js 2f \n\t" \
       "decl %[pos] \n\t" \
       "testb $0x80,(%[sieb],%[pos]) \n\t" \
       "js 2f \n\t" \
       "decl %[pos] \n\t" \
       "testb $0x80,(%[sieb],%[pos]) \n\t" \
       "js 2f \n\t" \
       "decl %[pos] \n\t" \
       "2: # Treffer gefunden" \
       : [pos] "=r" (offset) : "[pos]" (offset), [sieb] "q" (Sieb));
#endif
#define asm_sieb(lp, d0, d1, P) \
  asm volatile ( \
       "cmpl %[disp1],%[disp0]\n\t" \
       "jle 3f\n\t" \
       "xchgl %[disp1],%[disp0]\n" \
       "3:\n\t" \
       "testl %[disp0],%[disp0]\n\t" \
       "jl 1f\n" \
       "0:\n\t" \
       "subb %[val],(%[feld],%[disp1])\n\t" \
       "subb %[val],(%[feld],%[disp0])\n\t" \
       "subl %[step],%[disp1]\n\t" \
       "subl %[step],%[disp0]\n\t" \
       "jns 0b\n" \
       "1:\n\t" \
       "testl %[disp1],%[disp1]\n\t" \
       "jl 2f\n\t" \
       "subb %[val],(%[feld],%[disp1])\n" \
       "2:\n" \
       : [disp0] "=q" (clobbered_int), [disp1] "=q" (clobbered_int) : [feld] "r" (Sieb), [step] "g" (P), [val] "r" (lp), "[disp0]" (d0), "[disp1]" (d1));

#elif TSIEBELEMENTSIZE == 2
#define asm_sieb_init() \
  asm volatile ( \
       "pushw %%ax\n\t" \
       "shll $16,%%eax\n\t" \
       "popw %%ax\n\t" \
       "shrl $1,%%ecx\n\t" \
       "cld\n\t" \
       "rep\n\t" \
       "stosl" \
       : "=c" (clobbered_int), "=D" (clobbered_int) : "a" (SiebRegler::log_Threshold), "D" (Sieb), "c" (Siebgroesse) : "memory");
#define asm_search_sieb(offset) \
  asm volatile ( \
       "0: # Alignment suchen \n\t" \
       "testl $1,%[pos] \n\t" \
       "jz 1f # is aligned \n\t" \
       "decl %[pos] \n\t" \
       "testw $0x8000,(%[sieb],%[pos],2) \n\t" \
       "js 2f # Treffer gefunden \n\t" \
       "1: # Treffer suchen \n\t" \
       "subl $2,%[pos] \n\t" \
       "testl $0x80008000,(%[sieb],%[pos],2) \n\t" \
       "jz 1b \n\t" \
       "incl %[pos] \n\t" \
       "testw $0x8000,(%[sieb],%[pos],2) \n\t" \
       "js 2f \n\t" \
       "decl %[pos] \n\t" \
       "2: # Treffer gefunden" \
       : [pos] "=r" (offset) : "[pos]" (offset), [sieb] "q" (Sieb));
#define asm_sieb(lp, d0, d1, P) \
  asm volatile ( \
       "cmp %1,%0\n\t" \
       "jle 3f\n\t" \
       "xchgl %1,%0\n" \
       "3:\n\t" \
       "test %0,%0\n\t" \
       "jl 1f\n" \
       "0:\n\t" \
       "subw %4,(%2,%1,2)\n\t" \
       "subw %4,(%2,%0,2)\n\t" \
       "subl %3,%1\n\t" \
       "subl %3,%0\n\t" \
       "jns 0b\n" \
       "1:\n\t" \
       "test %1,%1\n\t" \
       "jl 2f\n\t" \
       "subw %4,(%2,%1,2)\n" \
       "2:\n" \
       : "=r" (d0), "=r" (d1) : "q" (Sieb), "g" (P), "q" (lp), "0" (d0), "1" (d1));
// fr neueren gcc funktioniert auch  : "+r" (d0), "+r" (d1) : "q" (Sieb), "g" (P), "q" (lp));
#elif TSIEBELEMENTSIZE == 4
#define asm_sieb_init() \
  asm volatile ( \
       "cld\n\t" \
       "rep\n\t" \
       "stosl" \
       : "=c" (clobbered_int), "=D" (clobbered_int) : "a" (SiebRegler::log_Threshold), "D" (Sieb), "c" (Siebgroesse) : "memory");
#define asm_search_sieb(offset) \
  asm volatile ( \
       "1: # Treffer suchen \n\t" \
       "decl %[pos] \n\t" \
       "testl $0x80008000,(%[sieb],%[pos],4) \n\t" \
       "jz 1b \n\t" \
       "2: # Treffer gefunden" \
       : [pos] "=r" (offset) : "[pos]" (offset), [sieb] "q" (Sieb));
#define asm_sieb(lp, d0, d1, P) \
  asm volatile ( \
       "cmp %1,%0\n\t" \
       "jle 3f\n\t" \
       "xchgl %1,%0\n" \
       "3:\n\t" \
       "test %0,%0\n\t" \
       "jl 1f\n" \
       "0:\n\t" \
       "subl %4,(%2,%1,4)\n\t" \
       "subl %4,(%2,%0,4)\n\t" \
       "subl %3,%1\n\t" \
       "subl %3,%0\n\t" \
       "jns 0b\n" \
       "1:\n\t" \
       "test %1,%1\n\t" \
       "jl 2f\n\t" \
       "subl %4,(%2,%1,4)\n" \
       "2:\n" \
       : "=r" (d0), "=r" (d1) : "q" (Sieb), "g" (P), "q" (lp), "0" (d0), "1" (d1));
// fr neueren gcc funktioniert auch  : "+r" (d0), "+r" (d1) : "q" (Sieb), "g" (P), "q" (lp));
#else
#undef SIEBASM386
#warning "Kein handoptimierter Asssemblerkode fr Aussieben mit TSiebElement!"
#endif
#endif


void Sieb_initialisieren(void)
{
  //cout << "Starte Initialisierung des Siebs von " << Sieboffset
  //   << " bis " << Sieboffset+Siebgroesse << endl;

 SiebRegler::RequestMoreDirtyness();

#ifdef SIEBASM386
  asm_sieb_init(); // siehe das entsprechende Assembler-Makro
#else
  register TSiebElement w = SiebRegler::log_Threshold;
  /* log_Threshold bezeichnet den Schwellwert, der zum Erkennen der Relation
     unterschritten werden mu. Es ist besser, den Threshold bereits bei der
     Initialisierung zu bercksichtigen (d.h. abzuziehen), da beim Auslesen
     des Siebs dann nur auf negative Werte getestet werden braucht. */
  
  // Initialisierung des Siebs mit dem berschlagsmigen Schwellwert
  register int i=Siebgroesse; do Sieb[--i]=w; while (i);
#endif

  // nachfolgender Block dient (falls aktiviert) nur noch Validierungszwecken:
#if 0
  Polynom.Werte_holen(Sieboffset,x,y);
  TSiebElement w_test = static_cast<TSiebElement>(log_Multiplikator*log(fabs(mpz_get_d(y))));
  TSiebElement FB_Threshold
    = static_cast<TSiebElement>(log_Multiplikator*Factor_Threshold*log(biggest_Prime_in_Factorbase));
  w_test-=FB_Threshold;
  if (Primzahlen[1]==2) w_test-=static_cast<TSiebElement>(log_Multiplikator*log(2));
  cout << "value " << y << " -> logwert: " << w_test << "<->" << log_Threshold << endl; 
#endif

  //cout << "Initialisierung abgeschlossen." << endl;
}


void schnelles_aussieben()
{
  //cout << "Aussieben mit Primzahlen der statischen FB: " << Sieboffset<< endl;
  
  /* wir sieben erst ab einer greren Primzahl aus der Faktorbasis,
     da wir sowieso unsauber und schnell sieben...
     ( je kleiner die Primzahl, um langsamer ist das Sieben mit dieser
     Zahl; und bei der gewhlten Implementierung ohne Bercksichtigung
     von Primzahlpotenzen ist auch der zustzlich gemachte Fehler
     vernachlssigbar und wird zudem ber Factor_Threshold kompensiert.
     Aber Vorsicht: Shortcut-Konstante nicht zu hoch ansetzen, sonst fallen
     zu viele Relationen raus!
     Achtung: als Startwert mindestens 2 whlen, damit -1 und 2 ignoriert
     werden knnen. Dann gibt es auch 2 Deltawerte. )
     -> relevant fr Schleife mit den kleinen Primzahlen
  */

  int nr=Primbasis_size-1;

  // groe Primzahlen (maximal 1 Siebtreffer pro Delta) 
  for (;; --nr) 
    {
      register int Primzahl=Primzahlen[nr];

      if (Primzahl<Siebgroesse) break;

      // neue d0/d1-Ermittlung (sollte schneller sein)
      register int d0,d1;
      {
        register int h=normalized_signed_mod(Sieboffset,Primzahl); // damned signed %-Operation...
	d0=Delta_der_Primzahlen[nr][0]-h;
	d1=Delta_der_Primzahlen[nr][1]-h;
	if (d1<0)
          d1+=Primzahl;
	if (d0<0)
          d0+=Primzahl;
      }
      
      register TSiebElement log_Primzahl = log_Primzahlen[nr];
      /* d.h. log_Primzahl erhlt den Wert (TSiebElement) (log_Multiplikator*log(Primzahl)) */
      
      if (d0<Siebgroesse)
        Sieb[d0]-=log_Primzahl;
      if (d1<Siebgroesse)
        Sieb[d1]-=log_Primzahl;
    }

  // kleine Primzahlen (mehrere Siebtreffer pro Delta)
  const int NonDirtyIndex = SiebRegler::sieben_ab_Primzahlindex();
  for (; nr>=NonDirtyIndex; --nr) 
    {
      register int Primzahl = Primzahlen[nr];
      //cout << "Quadratisches Sieb mit " << Primzahl << endl;
      
      // d0/d1-Ermittlung
      register int d0,d1;
      {
	register int h=normalized_signed_mod(Sieboffset+Siebgroesse,Primzahl); // damned signed %-Operation...
	d0=Delta_der_Primzahlen[nr][0]-h;
	d1=Delta_der_Primzahlen[nr][1]-h;
	if (d1>=0) d1-=Primzahl;
	if (d0>=0) d0-=Primzahl;
      }
      
      register TSiebElement log_Primzahl = log_Primzahlen[nr];
      /* d.h. log_Primzahl erhlt den Wert (TSiebElement) (log_Multiplikator*log(Primzahl)) */
      
      d0+=Siebgroesse;
      d1+=Siebgroesse;

      //if (d0==d1) cout << Primzahl << " teilt Vorfaktor!" << endl;
      
#ifdef SIEBASM386
      asm_sieb(log_Primzahl, d0, d1, Primzahl);
#else 
      if (d0>d1) { int h=d0; d0=d1; d1=h; } // ggf. tauschen, so da d0<d1
      while (d0>=0)
	{
	  Sieb[d0]-=log_Primzahl; d0-=Primzahl;
	  Sieb[d1]-=log_Primzahl; d1-=Primzahl;
	}
      if (d1>=0) Sieb[d1]-=log_Primzahl;
#endif
    }

  // hhere Primzahlpotenzen
  for (nr=0; nr<anz_weitere_Primzahlpotenzen; nr++)
    {
      register int PPotenz = Primzahlpotenzen[nr];
      
#ifdef ALTE_D_ERMITTLUNG
      register int d0=(Delta_der_Primzahlpotenzen[nr][0]-(Sieboffset+Siebgroesse))%PPotenz;
      register int d1=(Delta_der_Primzahlpotenzen[nr][1]-(Sieboffset+Siebgroesse))%PPotenz;
      if (d1>=0) d1-=PPotenz;
      if (d0>=0) d0-=PPotenz;
#else
      // neue d0/d1-Ermittlung (sollte schneller sein)
      register int d0,d1;
      {
	register int h=normalized_signed_mod(Sieboffset+Siebgroesse,PPotenz); // damned signed %-Operation...
	d0=Delta_der_Primzahlpotenzen[nr][0]-h;
	d1=Delta_der_Primzahlpotenzen[nr][1]-h;
	if (d1>=0) d1-=PPotenz;
	if (d0>=0) d0-=PPotenz;
      }
#endif
      
      register TSiebElement log_Primzahl = log_Primzahl_der_Primzahlpotenzen[nr];
      /* d.h. log_Primzahl erhlt den Wert (TSiebElement) (log_Multiplikator*log(Primzahl)) */
      
      // beachte: da mit den Primzahlen selbst auch gesiebt wird, darf nicht log(Primzahlpotenz), sondern
      // nur log(Primzahl) abgezogen werden!
      // Analogie: Wenn man schon durch p geteilt hat, mu man nocheinmal durch p teilen,
      //           um insgesamt durch p^2 zu teilen!

      d0+=Siebgroesse;
      d1+=Siebgroesse;

#ifdef SIEBASM386
      asm_sieb(log_Primzahl, d0, d1, PPotenz);
#else
      if (d0>d1) { int h=d0; d0=d1; d1=h; } // ggf. tauschen, so da d0<d1
      while (d0>=0)
	{
	  Sieb[d0]-=log_Primzahl; d0-=PPotenz;
	  Sieb[d1]-=log_Primzahl; d1-=PPotenz;
	}
      if (d1>=0) Sieb[d1]-=log_Primzahl;      
#endif
    }

}


void aussieben_mit_Quadraten()
{
  // Quadrate von statischen (und ggf. dynamischen Faktoren) sieben.
  // Wichtig: Daher nur den einfachen Logarithmus des Faktors subtrahieren!
  // Bemerkung: Zur Effizienzsteigerung wird davon ausgegangen, da jedes Quadrat nur einmal
  //            innerhalb des gesamten Siebintervalls trifft!

  //cout << "Quadrate sieben Start" << endl;
  if (Sieb_Delta_QuadrateHeap.empty()) return; // es gibt nichts zu sieben...
  TSieb_Delta sieb_delta = Sieb_Delta_QuadrateHeap.top();
  register int delta = sieb_delta.delta-Sieboffset;
  
  while (delta<Siebgroesse)
    {
      Sieb_Delta_QuadrateHeap.pop(); // Element aus der Priorittswarteschlange nehmen
      Sieb[delta]-=static_cast<TSiebElement>(ceil(log_Multiplikator*log(sieb_delta.factor))); /* Teiler im Sieb vermerken */
      if (Sieb_Delta_QuadrateHeap.empty()) break;
      sieb_delta = Sieb_Delta_QuadrateHeap.top();
      delta = sieb_delta.delta-Sieboffset;
    }
  //cout << "Quadrate sieben Ende" << endl;
}


void aussieben_mit_Dynamic_Factoren()
{
  if (Sieb_Delta_Heap.empty()) return; // es gibt nichts zu sieben...
  short int HitCount=0;
  TSieb_Delta sieb_delta = Sieb_Delta_Heap.top();
  register int delta = sieb_delta.delta-Sieboffset;
  goto Schleifentest;
  
 Schleifenanfang_weiterer_Faktor:
  ++HitCount;
  //if (HitCount>=MaxHits) { cerr << "MaxHits erhhen!"; exit(1); } 
 Schleifenanfang_normal:
  Sieb_Delta_Heap.pop(); // Element aus der Priorittswarteschlange nehmen
  Sieb[delta]-=static_cast<TSiebElement>(ceil(log_Multiplikator*log(sieb_delta.factor))); /* Teiler im Sieb vermerken */
  
  // Da dynamische Faktoren selten auftreten, diese Faktoren schon whrend
  // des Siebens vermerken...
  // ...das erhht die Effizienz bei der Refaktorisierung...    
  Hits[HitCount].Faktor=sieb_delta.factor;

#ifdef USE_FAKEHEAP
  if (Sieb_Delta_Heap.empty()) goto Schlange_leer; // keine weiteren Faktoren zum Sieben?
#else
  sieb_delta.delta += sieb_delta.factor; // nchste Hit-Position
  if (sieb_delta.delta<=M) // shortcut: nur dann eintragen, wenn die Relation noch im Siebintervall relevant wird
   {
     Sieb_Delta_Heap.push(sieb_delta); // wieder in Priorittswarteschlange eintragen
   }
  else
    if (Sieb_Delta_Heap.empty()) goto Schlange_leer; // keine weiteren Faktoren zum Sieben?
#endif
  
  sieb_delta = Sieb_Delta_Heap.top(); // nchsten Eintrag aus Priorittswarteschlange holen
  register int delta2; delta2 = sieb_delta.delta-Sieboffset;
  if (delta==delta2) goto Schleifenanfang_weiterer_Faktor;  
  if (Sieb[delta]<0) // Relation gefunden?
   if (SiebRegler::Hit_after_DirtyKorrektur(delta+Sieboffset,Sieb[delta]))
    {
      Sieb[delta]=1; // weiteren Treffer an dieser Stelle abwrgen...
      Relation_einfuegen(new CRelation(delta+Sieboffset, HitCount+1)); /* dann einfgen */
    }
  delta=delta2;
  
  HitCount=0; // und auf ein neues
 Schleifentest:
  if (delta<Siebgroesse) goto Schleifenanfang_normal; // noch im aktuellen (physikalischen) Intervall?
  return; // fertig mit diesem Intervall
  
 Schlange_leer:
  if (Sieb[delta]<0) // Relation gefunden?
   if (SiebRegler::Hit_after_DirtyKorrektur(delta+Sieboffset,Sieb[delta]))
    {
      Sieb[delta]=1; // weiteren Treffer an dieser Stelle abwrgen...
      Relation_einfuegen(new CRelation(delta+Sieboffset,HitCount+1)); // dann einfgen
    }
  return; 
}


void schnelles_auslesen_des_Siebs(void)
{
  //cout << "Auslesen des (gesiebten) Siebintervalls" << endl;

  // da nur ein Bruchteil der Siebelemente Treffer sind, sollte man das Suchen beschleunigen!
  if (Sieb[0]<0) // Relation an der Stelle 0 gefunden?
    Relation_einfuegen(new CRelation(Sieboffset)); // dann einfgen
  Sieb[0]=-1; // Dummytreffer generieren (fr Abbruch)
  register int d=Siebgroesse;
  do
   {
     // todo: hier bietet sich in assembler doch frmlich an...
#ifdef SIEBASM386
     asm_search_sieb(d); // nach Siebtreffer suchen...
     // if (d && Sieb[d]) cout << "Siebtreffer fr " << d << endl;
#else
     while (Sieb[--d]>=0);; // nach Siebtreffer suchen...
#endif
     if (d) // gefundener Treffer *nicht* Dummytreffer?
      {
        if (SiebRegler::Hit_after_DirtyKorrektur(Sieboffset+d,Sieb[d]))
          Relation_einfuegen(new CRelation(Sieboffset+d)); // dann einfgen
      }
   } while (d>1); // solange noch regulre Treffer mglich: weitersuchen
}

#endif /* #ifndef IS_SERVER */



/* ---------- Vorfaktorbestimmung ------------ */

// Um verschiedene Vorfaktorbestimmungsmethoden (mit jeweiliger Neu-Compilierung)
// testen zu knnen, ist die Prozedur " void bestes_kN_ermitteln() "
// auf das File "Vorfaktorbestimmung.cc" ausgelagert.

#include "Vorfaktorbestimmung.cc"

/* ------------------------------------------- */

#include "mpqsPolynom.cc" // we still need to include the sources here
                          // (the module has been "outsourced", but it still includes
                          // some dependencies!)

#ifndef IS_SERVER
void alle_Deltas_neu_berechnen()
{
  /* Beim Wechsel auf ein neues Polynom werden alle bisherigen Deltawerte der Primzahlen
     ungltig. Daher ist es erforderlich, diese neu zu berechnen */

  //cout << "Berechne die Deltas neu" << endl;

  // zunchst alle (mglicherweise) noch bestehenden dynamischen Faktoren aus der Warteschlange nehmen:
#ifdef USE_FAKEHEAP
//  cout << "clearing Fakeheap with " << Sieb_Delta_Heap.size() << " elements..." << endl;
  Sieb_Delta_Heap.clear();
#else
  while (!Sieb_Delta_Heap.empty()) Sieb_Delta_Heap.pop(); // fuer priority_queue, die kein clear hat
#endif

  // ebenso die Quadrate:
  //Sieb_Delta_QuadrateHeap.clear();
  while (!Sieb_Delta_QuadrateHeap.empty()) Sieb_Delta_QuadrateHeap.pop(); // fuer priority_queue, die kein clear hat
 
  
  // als erstes die Deltas aus der statische Faktorbasis ermitteln
  unsigned int inv_A2_modp, PolyB;
  mpz_t inv_A2, P, wuq, x, y;
  mpz_init(inv_A2); mpz_init(P); mpz_init(wuq); mpz_init(x); mpz_init(y);
  int i=1;
  if (Primzahlen[i]==2) i++; // Primzahl 2 berspringen, da Polynom damit Probleme hat...
  for (; i<Primbasis_size; i++)
    {
      //mpz_set_ui(P,Primzahlen[i]);
      //mpz_invert(inv_A2,Polynom.A2,P);
      inv_A2_modp=invmod(mpz_remainder_ui(Polynom.get_A2(),Primzahlen[i]),Primzahlen[i]);
      PolyB=mpz_remainder_ui(Polynom.get_B(),Primzahlen[i]);
      
      //mpz_set_ui(x,SQRT_kN_der_Primzahlen[i]);
      //mpz_sub(x,x,Polynom.B); mpz_mul_ui(x,x,inv_A2_modp); 
      //Delta_der_Primzahlen[i][0]=mpz_mod_ui(x,x,Primzahlen[i]);
      register int d;
      d=SQRT_kN_der_Primzahlen[i]-PolyB; if (d<0) d+=Primzahlen[i];
      Delta_der_Primzahlen[i][0]=mulmod(d,inv_A2_modp,Primzahlen[i]);
      
      //mpz_set_ui(x,SQRT_kN_der_Primzahlen[i]); mpz_neg(x,x);
      //mpz_sub(x,x,Polynom.B); mpz_mul_ui(x,x,inv_A2_modp); 
      //Delta_der_Primzahlen[i][1]=mpz_mod_ui(x,x,Primzahlen[i]);
      d=Primzahlen[i]-SQRT_kN_der_Primzahlen[i]-PolyB; if (d<0) d+=Primzahlen[i];
      Delta_der_Primzahlen[i][1]=mulmod(d,inv_A2_modp,Primzahlen[i]);

#if 0
      //ggf. testen, ob Werte in Ordnung sind...
      Polynom.Werte_holen(Delta_der_Primzahlen[i][0],x,y);
      if (mpz_mod_ui(y,y,Primzahlen[i])!=0) 
	{ 
	  cerr << "Deltawert0 " << Delta_der_Primzahlen[i][0] << " liefert Teiler fr "
	       << Primzahlen[i] << " nicht! (Rest ";
	  mpz_out_str(stdout,10,y); cout << ")" << endl;
	  Delta_der_Primzahlen[i][0]-=mpz_get_ui(y);
	  if (Delta_der_Primzahlen[i][0]<0) Delta_der_Primzahlen[i][0]+=Primzahlen[i];
	}
      Polynom.Werte_holen(Delta_der_Primzahlen[i][1],x,y);
      if (mpz_mod_ui(y,y,Primzahlen[i])!=0)
	{ 
	  cerr << "Deltawert1 " << Delta_der_Primzahlen[i][1] << " liefert Teiler fr "
	       << Primzahlen[i] << " nicht! (Rest ";
	  mpz_out_str(stdout,10,y); cout << ")" << endl;
	  Delta_der_Primzahlen[i][1]-=mpz_get_ui(y);
	  if (Delta_der_Primzahlen[i][1]<0) Delta_der_Primzahlen[i][1]+=Primzahlen[i];
	}
#endif

#ifdef SIEVING_LARGE_SQUARES
      if (i>FB_maxQuadrate && Vorfaktor_von_kN%Primzahlen[i]!=0)
	{
	  // Fr alle "groen" Quadrate aus der FB auch die Deltas ermitteln,
	  // aber diese (falls innerhalb des Siebintervalls liegend) zusammen mit den dynamischen Faktoren sieben.
	  mpz_set_ui(P,Primzahlen[i]); mpz_mul(P,P,P); // Quadrat der Primzahl holen
	  mpz_invert(inv_A2,Polynom.get_A2(),P);

	  // nun die Wurzel modulo P ermitteln (auf ein Caching wird verzichtet, da sie gro sein kann)
	  // Wurzeln von kN modulo Primzahl^2 ermitteln
	  const int Wu = SQRT_kN_der_Primzahlen[i];
	  mpz_set_ui(y,2*Wu);
	  if (mpz_invert(y,y,P)==0)
	    {
	      cerr << "PQ-Wurzel fr " << Primzahlen[i] << ": Inverses existiert nicht!" << endl;
	      exit(1);
	    }
	  
	  mpz_mod(x,kN,P); mpz_set_ui(wuq,Wu); mpz_mul_ui(wuq,wuq,Wu); mpz_sub(x,x,wuq);
	  mpz_mul(x,x,y); mpz_add_ui(x,x,Wu); mpz_mod(wuq,x,P);
	  
	  // ggf. Prfung, ob Wurzel korrekt ist:
	  /* 
	     mpz_mul(x,wuq,wuq); mpz_sub(x,kN,x); mpz_mod(x,x,P);
	     if (mpz_cmp_ui(x,0)!=0)
	      {
	        cerr << "PQ-Wurzel nicht korrekt!" << endl;
	        cerr << "Primzahl=" << Primzahlen[i] << endl;
	        cerr << "kN= " << kN << endl;
	        cerr << "SQRT(kN) mod p=" << SQRT_kN_der_Primzahlen[i] << endl;
	        cerr << "SQRT(kN) mod p^2=" << wuq << endl;
	        cerr << "aber falsch: " << x << endl;
	        exit(1); 
	      }
	  */
	  
	  // nun die Deltas:

	  // Delta1
	  mpz_sub(x,wuq,Polynom.get_B()); mpz_mul(x,x,inv_A2); mpz_mod(x,x,P);
	  mpz_sub_ui(x,x,M);
	  if (mpz_cmp_ui(x,0)<0) mpz_add_ui(x,x,M);
	  if (mpz_cmp_ui(x,M)<=0) // liegt der Wert im Intervall?
	    {
	      TSieb_Delta sieb_delta;
	      sieb_delta.factor=Primzahlen[i];
	      sieb_delta.delta=mpz_get_ui(x);
	      Sieb_Delta_QuadrateHeap.push(sieb_delta); // und ablegen
	      //cout << "Quadrat von " << sieb_delta.factor << " gepusht!" << endl;
	    }
	  
	  // Delta2
	  mpz_neg(x,wuq); mpz_sub(x,x,Polynom.get_B()); mpz_mul(x,x,inv_A2); mpz_mod(x,x,P);
	  mpz_sub_ui(x,x,M);
	  if (mpz_cmp_ui(x,0)<0) mpz_add_ui(x,x,M);
	  if (mpz_cmp_ui(x,M)<=0) // liegt der Wert im Intervall?
	    {
	      TSieb_Delta sieb_delta;
	      sieb_delta.factor=Primzahlen[i];
	      sieb_delta.delta=mpz_get_ui(x);
	      Sieb_Delta_QuadrateHeap.push(sieb_delta); // und ablegen
	      //cout << "Quadrat von " << sieb_delta.factor << " gepusht!" << endl;
	    }
	  
	}
#endif
      
    }

#ifdef SIEVING_MORE_LARGE_SQUARES
  // und weitere Quadrate pushen:
  {
    unsigned int Primzahl = biggest_Prime_in_Factorbase;
    unsigned int Obergrenze = 3*biggest_Prime_in_Factorbase;
    cout << "pushe weitere Prim-Quadrate bis " << Obergrenze << endl; 
    while (Primzahl<Obergrenze)
     {
       do // nchsthhere Primzahl der statischen Faktorbasis ermitteln 
	{
	  do Primzahl+=2; 
           while( Primzahl%3==0 || Primzahl%5==0 || Primzahl%7==0
                  || Primzahl%11==0 || Primzahl%13==0 || Primzahl%17==0
                  || !probab_prime(Primzahl)); /* nchste Primzahl */
	  mpz_set_ui(P,Primzahl);
	}
       while (mpz_legendre(kN,P)<0); /* Primzahl gltig fr Primzahlbasis? */

       mpz_mul(P,P,P); // Quadrat der Primzahl holen
       mpz_invert(inv_A2,Polynom.get_A2(),P);

       // nun die Wurzel modulo P ermitteln (auf ein Caching wird verzichtet, da sie gro sein kann)
       // Wurzeln von kN modulo Primzahl^2 ermitteln
       const int Wu = SQRT_kN_mod_Primzahl(Primzahl);
       mpz_set_ui(y,2*Wu);
       if (mpz_invert(y,y,P)==0)
        {
          cerr << "PQ-Wurzel fr " << Primzahl << ": Inverses existiert nicht!" << endl;
          exit(1);
	}
	  
       mpz_mod(x,kN,P); mpz_set_ui(wuq,Wu); mpz_mul_ui(wuq,wuq,Wu); mpz_sub(x,x,wuq);
       mpz_mul(x,x,y); mpz_add_ui(x,x,Wu); mpz_mod(wuq,x,P);

#if 0
       // ggf. Prfung, ob Wurzel korrekt ist:
       mpz_mul(x,wuq,wuq); mpz_sub(x,kN,x); mpz_mod(x,x,P);
       if (mpz_cmp_ui(x,0)!=0)
        {
          cerr << "PQ-Wurzel nicht korrekt!" << endl;
          cerr << "Primzahl=" << Primzahl << endl;
          cerr << "kN= " << kN << endl;
          cerr << "SQRT(kN) mod p=" << Wu << endl;
          cerr << "SQRT(kN) mod p^2=" << wuq << endl;
          cerr << "aber falsch: " << x << endl;
          exit(1); 
        }
#endif
       // nun die Deltas:

       // Delta1
       mpz_sub(x,wuq,Polynom.get_B()); mpz_mul(x,x,inv_A2);mpz_mod(x,x,P);
       mpz_sub_ui(x,x,M);
       if (mpz_cmp_ui(x,0)<0) mpz_add_ui(x,x,M);
       if (mpz_cmp_ui(x,M)<=0) // liegt der Wert im Intervall?
        {
          TSieb_Delta sieb_delta;
          sieb_delta.factor=Primzahl;
          sieb_delta.delta=mpz_get_ui(x);
          Sieb_Delta_Heap.push(sieb_delta); // und ablegen
          Sieb_Delta_QuadrateHeap.push(sieb_delta); // und ablegen
          // Bemerkung: Die Primzahl wird im Sieb_Delta_Heap *und* 
          // gleichzeitig im Sieb_Quadrate_Heap abgelegt.
          // Damit ist sichergestellt, da
          // - der Wert in den integer-Bereich pat
          // - der Wert doppelt (d.h. als Quadrat) im Sieb vermerkt wird.
          // Der Effekt eines zu hufigen Siebens im Falle, da
          // Primzahl bereits dynamischer Faktor ist, wird vernachlssigt.
          cout << "Quadrat von " << sieb_delta.factor << " gepusht!" << endl;
        }

       // Delta2
       mpz_neg(x,wuq); mpz_sub(x,x,Polynom.get_B()); mpz_mul(x,x,inv_A2); mpz_mod(x,x,P);
       mpz_sub_ui(x,x,M);
       if (mpz_cmp_ui(x,0)<0) mpz_add_ui(x,x,M);
       if (mpz_cmp_ui(x,M)<=0) // liegt der Wert im Intervall?
        {
          TSieb_Delta sieb_delta;
          sieb_delta.factor=Primzahl;
          sieb_delta.delta=mpz_get_ui(x);
          Sieb_Delta_Heap.push(sieb_delta); // und ablegen
          Sieb_Delta_QuadrateHeap.push(sieb_delta); // und ablegen
          // Bemerkung: Die Primzahl wird im Sieb_Delta_Heap *und* 
          // gleichzeitig im Sieb_Quadrate_Heap abgelegt.
          // Damit ist sichergestellt, da
          // - der Wert in den integer-Bereich pat
          // - der Wert doppelt (d.h. als Quadrat) im Sieb vermerkt wird.
          // Der Effekt eines zu hufigen Siebens im Falle, da
          // Primzahl bereits dynamischer Faktor ist, wird vernachlssigt.
          cout << "Quadrat von " << sieb_delta.factor << " gepusht!" << endl;
        }

     }
  }
#endif

if (!Sieb_Delta_QuadrateHeap.empty())
 cout << "Es wurden " << Sieb_Delta_QuadrateHeap.size() << " Quadrate gepusht." << endl;
  
  for (i=0; i<anz_weitere_Primzahlpotenzen; i++)
    {
      // Fr diese Zahlen auch die Deltas bzgl. der jeweiligen Potenz ermitteln
      // (fr das Sieben zusammen mit den statischen Faktoren)

      //mpz_set_ui(P,Primzahlpotenzen[i]);
      //mpz_invert(inv_A2,Polynom.A2,P);
      inv_A2_modp=invmod(mpz_remainder_ui(Polynom.get_A2(),Primzahlpotenzen[i]),Primzahlpotenzen[i]);
      PolyB=mpz_remainder_ui(Polynom.get_B(),Primzahlpotenzen[i]);

      //mpz_set_ui(x,SQRT_kN_der_Primzahlpotenzen[i]);
      //mpz_sub(x,x,Polynom.B); mpz_mul(x,x,inv_A2); 
      //Delta_der_Primzahlpotenzen[i][0]=mpz_mod_ui(x,x,Primzahlpotenzen[i]);
      register int d;
      d=SQRT_kN_der_Primzahlpotenzen[i]-PolyB; if (d<0) d+=Primzahlpotenzen[i];
      Delta_der_Primzahlpotenzen[i][0]=mulmod(d,inv_A2_modp,Primzahlpotenzen[i]);
      
      //mpz_set_ui(x,SQRT_kN_der_Primzahlpotenzen[i]); mpz_neg(x,x);
      //mpz_sub(x,x,Polynom.B); mpz_mul(x,x,inv_A2); 
      //Delta_der_Primzahlpotenzen[i][1]=mpz_mod_ui(x,x,Primzahlpotenzen[i]);
      d=Primzahlpotenzen[i]-SQRT_kN_der_Primzahlpotenzen[i]-PolyB; if (d<0) d+=Primzahlpotenzen[i];
      Delta_der_Primzahlpotenzen[i][1]=mulmod(d,inv_A2_modp,Primzahlpotenzen[i]);
    }
  
  mpz_clear(inv_A2); mpz_clear(P); mpz_clear(wuq); mpz_clear(x); mpz_clear(y);
  
  // dann die dynamische Faktorbasis behandeln:
  TDynamic_Factor_Relations::iterator pos;
  for (pos=Dynamic_Factor_Relations.begin();
       pos!=Dynamic_Factor_Relations.end(); ++pos) Deltas_berechnen_fuer(*pos, -M);
  
 // cout << "Deltaberechnung beendet." << endl;
}
#endif


#include "parse_term.cc"


const bool ohne_multi_kombinieren_init = false; // true ->multi-init, false ->kein multi_init bei Relation_einfuegen

bool SpecialFactorRelation_LAZY_vermerken(const CmpqsFactor &DLP, const string &GL_String)
{
  // Special-Factor in die Special-Factor-Set eintragen (doppelt, damit
  // beide Large-Primes schnell zugreifbar sind)
  // Falls der Special-Factor schon vorhanden sein sollte, kombinieren
  // und ins Gleichungssystem eintragen.

  if (DLP.LP1()==DLP.LP2()) // Sonderfall: DoubleLargePrime=LargePrime^2
   {
     cout << "Spezialfall: DLP ist Quadrat! (in LAZY)" << endl;
     istringstream is(GL_String);
     CRelation* GL = new CRelation();
     GL->multi_kombinieren_init();
     CmpqsFactor DLP_2 = GL->multi_kombinieren_main(is);
     if (DLP!=DLP_2) { cerr << "DLP-Fehler irgendwo in LAZY!" << endl; exit(1); }
     Special_hit++;
     // immer beachten: Delta ist nicht das Quadrat, sondern wird quadriert,
     // daher zu Delta immer nur Wurzeln multiplizieren!
     mpz_mul_ui(GL->Delta,GL->Delta,DLP.LP1()); mpz_mod(GL->Delta,GL->Delta,n);
     Relation_einfuegen(GL,ohne_multi_kombinieren_init); // LAZY -> einfgen geschieht hier...
     return false;
   }
  
  TSpecialFactorRelation relation;
  relation.factor=DLP;

  const TSpecial_Factor_Relations::iterator pos = Special_Factor_Relations.find(relation);
  if (pos==Special_Factor_Relations.end())
   {
     //cout << "Speichere DLP-Relation (LAZY)" << endl;
     //relation.fpos=GL->speichern(SpecialRelations_to_file,relation.factor);
     relation.fpos=SpecialRelations_to_file.tellp();
     SpecialRelations_to_file << GL_String;

     Special_Factor_Relations.insert(relation);
     relation.factor.swap();
     Special_Factor_Relations.insert(relation);

     // regelmig Zyklensuche auf den Special-Relations durchfhren:
     static int naechste_Zyklensuche = 50000;
     if (!--naechste_Zyklensuche)
      {
        naechste_Zyklensuche=50000;
        SpecialFactors_Zyklensuche();
      }
     return true;
   }
  else
   { 
     // Special-Factor ist bereits bekannt
     cout << "Special Hit durch bekannten Special Factor! (LAZY)" << endl;
     istringstream is(GL_String);
     CRelation* GL = new CRelation();
     GL->multi_kombinieren_init();
     CmpqsFactor DLP_2 = GL->multi_kombinieren_main(is);
     if (DLP!=DLP_2) { cerr << "DLP-Fehler irgendwo in LAZY!" << endl; exit(1); }
     Special_hit++;
     mpz_t x;
     mpz_init(x); DLP.in_mpz_ablegen(x);
     mpz_mul(GL->Delta,GL->Delta,x); mpz_mod(GL->Delta,GL->Delta,n);
     mpz_clear(x);
     GL->multi_kombinieren_main(SpecialRelations_from_file,(*pos).fpos);
     //GL->multi_kombinieren_exit() kann entfallen, wenn zugehriges init bei Relation_einfuegen entfllt
     Relation_einfuegen(GL,ohne_multi_kombinieren_init); // LAZY -> einfgen geschieht hier...
     return false;
   }
}


#ifdef IS_STANDALONE
void bearbeite_ecm()
{
  streampos fpos = Recovery_from_file.tellg();
  int Anzahl_Kurven = 0;
  string s;

  srand(time(NULL)); // Random-Seed setzen (Zeitfunktion drfte hierzu zufllig genug sein)
  
  unsigned int dezimalstellen = mpz_sizeinbase(n,10);
  tune_parameters(dezimalstellen); // Parameter abhngig von der Gre der Zahl!

  Recovery_from_file >> Anzahl_Kurven;
  Recovery_from_file.ignore(1,'\n'); // endl-berlesen
  while (Anzahl_Kurven<elcu_Kurven)
  {

    Anzahl_Kurven++;
    cout << "elliptic curve #" << Anzahl_Kurven << "/" << elcu_Kurven << endl;

    int ecm_sigma = rand(); // zufllige Kurve whlen
    elliptic_curves elliptic_curve; // Instanz der Kurve kreieren
    elliptic_curve.go(ecm_sigma,elcu_Phase1,static_cast<elliptic_curves::NATNUM>(elcu_Phase2)); /* Faktorisierung mit elliptischer Kurve versuchen */

    // Anzahl bearbeiteter Kurven ins Recovery-File schreiben...
    Recovery_to_file.seekp(fpos); 
    Recovery_to_file << Anzahl_Kurven << endl << flush;

    // Teiler gefunden?
    if (dezimalstellen==mpz_sizeinbase(n,10)) continue; // kein neuer Faktor
    
    // Stelligkeit von n hat sich gendert -> neuer Faktor

    if (mpz_probab_prime_p(n,probab_prime_checks))
      {
        Factorization_to_file << MAL() << n;
        cout << "remaining factor: " << n << endl;
        cout << "remaining factor is most probably prime!" << endl;
        mpz_set_ui(n,1); // n ist faktorisiert
      }

    if ( (mpz_cmp_ui(n,1)==0) || Potenztest(n) ) // Faktorisierung vollstndig?
      { 
        cout << "factorization successfully completed." << endl;
        Factorization_to_file << endl;
    
        // Streams schlieen
        SpecialRelations_buffer.close();
        DynamicRelations_buffer.close();
        StaticRelations_buffer.close();
        Recovery_buffer.close();

        // Files lschen
        remove(SpecialRelationsFile.c_str());
	remove(DynamicRelationsFile.c_str());
	remove(StaticRelationsFile.c_str());
	remove(RecoveryFile.c_str());
	exit(0); // Programm erfolgreich beenden...
      }

    // fr weitere ecm-Runden Parameter neu tunen...
    dezimalstellen=mpz_sizeinbase(n,10);
    tune_parameters(dezimalstellen);

    // nun mu noch das recovery-File angepat werden, da sich
    // die zu faktorisierende Zahl verkleinert hat!
    Recovery_buffer.close(); // File schlieen
    Recovery_buffer.open(RecoveryFile.c_str(),ios::in|ios::out|ios::trunc); // wieder ffnen (leeres File erzeugen!)
    Recovery_to_file << n << endl; // zu faktorisierende Zahl ablegen
    fpos = Recovery_from_file.tellg(); // Position merken
    Recovery_to_file << Anzahl_Kurven << endl << flush; // Anzahl bisher bearbeiteter Kurven vermerken

  }
  cout << "ECM-Phase (standalone) completed." << endl;
}
#endif



#ifdef IS_SERVER

void bearbeite_daten_stream_ecm(int server_socket)
{
  streampos fpos = Recovery_from_file.tellg();
  int Anzahl_Kurven = 0;
  string s;

  srand(time(NULL)); // Random-Seed setzen (Zeitfunktion drfte hierzu zufllig genug sein)

  Recovery_from_file >> Anzahl_Kurven;
  Recovery_from_file.ignore(1,'\n'); // endl-berlesen

  while (Anzahl_Kurven<elcu_Kurven)
  {
    int communication_socket = accept(server_socket, NULL, NULL);
    if (communication_socket == 0) continue; // auf Verbindung warten

    unix_buffer buffer(communication_socket);
    //cout << connection_info(buffer) << endl;
    iostream daten(&buffer);

    daten >> s;

    if (s == "ECM?")
     {
       int ecm_sigma = rand();
       daten << ecm_sigma << " "
             << elcu_Phase1 << " "
             << elcu_Phase2 << " "
             << n << endl; // sigma und n bertragen

       Anzahl_Kurven++;
       cout << "elliptic curve #" << Anzahl_Kurven << "/" << elcu_Kurven
            << " with sigma=" << ecm_sigma
            << " sent to " << peer_info(buffer) << endl;

       // Anzahl vergebener Kurven ins Recovery-File schreiben...
       Recovery_to_file.seekp(fpos); 
       Recovery_to_file << Anzahl_Kurven << endl << flush;

       daten << flush;
       close (communication_socket);
       continue;
     }

    if (s == "Faktor(ECM)!") // Faktoren annehmen...
      {
        cout << "Factor received from " << peer_info(buffer) << endl;
        int sigma;
        mpz_t x;
        mpz_init(x);
        daten >> sigma >> x; // Kurveninformation (sigma) und Faktor holen
        mpz_gcd(x,x,n); // ggt bilden (Faktor knnte bereits entdeckt sein!)
        if (mpz_cmp_ui(x,1)!=0)
         {
           // theoretisch knnte Faktor mehrfach teilen...
	   const unsigned int exponent = mpz_remove(n,n,x);
           if (mpz_probab_prime_p(x,probab_prime_checks))
             {
               cout << x << " is factor." << endl;
               if (mpz_sizeinbase(x,10)<25)
                 Factorization_to_file << MAL(x,exponent) << " [ecm]" << flush;
               else
                 Factorization_to_file << MAL(x,exponent) << " [ecm,sigma=" << sigma << "]" << flush;
             }
           else
             {
               cout << x << " is a composite factor." << endl;
	       if (mpz_probab_prime_p(n,probab_prime_checks))
		{
		  if (exponent>1) mpz_pow_ui(x,x,exponent);
                  mpz_swap(n,x);
		  cout << x << " is factor. (factorswap)" << endl;
		  Factorization_to_file << MAL() << x << " [ecm" << "/factorswap,sigma=" << sigma << "]" << flush;
                }
	       else
                {
                  if (mpz_sizeinbase(x,10)<25)
                   Factorization_to_file << MAL(x,exponent) << " [ecm] [composite]" << flush;
                  else
                   Factorization_to_file << MAL(x,exponent) << " [ecm,sigma=" << sigma << "] [composite]" << flush;
                }
             }
         }
        mpz_clear(x);
        daten << flush;
        close (communication_socket);

        if (mpz_probab_prime_p(n,probab_prime_checks))
          {
            Factorization_to_file << MAL() << n;
            cout << "remaining factor: " << n << endl;
            cout << "remaining factor is most probably prime!" << endl;
            mpz_set_ui(n,1); // n ist faktorisiert
          }

        if ( (mpz_cmp_ui(n,1)==0) || Potenztest(n) ) // Faktorisierung vollstndig?
          { 
            cout << "factorization successfully completed." << endl;
            Factorization_to_file << endl;
    
            // Streams schlieen
            SpecialRelations_buffer.close();
            DynamicRelations_buffer.close();
            StaticRelations_buffer.close();
            Recovery_buffer.close();

	    // Files lschen
	    remove(SpecialRelationsFile.c_str());
	    remove(DynamicRelationsFile.c_str());
	    remove(StaticRelationsFile.c_str());
	    remove(RecoveryFile.c_str());
	    exit(0); // Programm erfolgreich beenden...
          }
        tune_parameters(mpz_sizeinbase(n,10)); // Parametertuning

        // nun mu noch das recovery-File angepat werden, da sich
        // die zu faktorisierende Zahl verkleinert hat!
        Recovery_buffer.close(); // File schlieen
        Recovery_buffer.open(RecoveryFile.c_str(),ios::in|ios::out|ios::trunc); // wieder ffnen (leeres File erzeugen!)
        Recovery_to_file << n << endl; // zu faktorisierende Zahl ablegen
        fpos = Recovery_from_file.tellg(); // Position merken
        Recovery_to_file << Anzahl_Kurven << endl << flush; // Anzahl bisher bearbeiteter Kurven vermerken
	continue;
      }

    if (s == "Hunger!")
     {
       // requesting MPQS parameters is not adequate while processing ecm!
       cerr << "Inadequate request of MPQS parameters from " << peer_info(buffer) << endl;
       daten << -1 << " " << "please wait until ecm is processed!" << endl;
       close (communication_socket);
       continue;
     }

    cerr << "Invalid request \"" << s << "\""
         << " from " << peer_info(buffer) << endl;
    daten << flush;
    close (communication_socket);
  }
  cout << "distributed ECM-Phase (server) completed." << endl;
}


class Cprocess_clients
{
 private:
  static CMutex process;
  static void bearbeite_daten_stream(iostream &daten, const string client);
 public:
  static void* THREAD_bearbeite_daten_stream(void* arg);
};
CMutex Cprocess_clients::process;

void Cprocess_clients::bearbeite_daten_stream(iostream &daten, const string client)
{
  string s;
  daten >> s;
  
  if (s == "Hunger!")
    {
      cout << "Sende zu faktorisierende Zahl an " << client << endl;
      daten << n << " " << Primbasis_size << " " 
            << Factor_Threshold << " " << M << endl;
      return;
    }

  if (s == "ECM?")
   {
     daten << "-1" << endl; // in dieser Phase keine elliptischen Kurven (-> sigma<0)
   }

  if (s == "Faktor(ECM)!") // Faktoren werden noch akzeptiert...
    {
      cout << "Factor received from " << client << endl;
      int sigma;
      mpz_t x;
      mpz_init(x);
      daten >> sigma >> x; // Kurveninformation (sigma) und Faktor holen
      mpz_gcd(x,x,n); // ggt bilden (Faktor knnte bereits entdeckt sein!)
      if (mpz_cmp_ui(x,1)==0) // teilt "Faktor"?
       { mpz_clear(x); return; } // "Faktor" teilt nicht (mehr)

      if (mpz_cmp_ui(x,1)!=0)
       {
         // theoretisch knnte Faktor mehrfach teilen...
         const unsigned int exponent = mpz_remove(n,n,x);
         if (mpz_probab_prime_p(x,probab_prime_checks))
           {
             cout << x << " is factor." << endl;
             if (mpz_sizeinbase(x,10)<25)
               Factorization_to_file << MAL(x,exponent) << " [ecm]" << flush;
             else
               Factorization_to_file << MAL(x,exponent) << " [ecm,sigma=" << sigma << "]" << flush;
           }
         else
           {
             cout << x << " is a composite factor." << endl;
             if (mpz_sizeinbase(x,10)<25)
               Factorization_to_file << MAL(x,exponent) << " [ecm] [composite]" << flush;
             else
               Factorization_to_file << MAL(x,exponent) << " [ecm,sigma=" << sigma << "] [composite]" << flush;
           }
       }
      mpz_clear(x);

      if (mpz_probab_prime_p(n,probab_prime_checks))
        {
          Factorization_to_file << MAL() << n;
          cout << "remaining factor: " << n << endl;
          cout << "remaining factor is most probably prime!" << endl;
          mpz_set_ui(n,1); // n ist faktorisiert
        }

      if ( (mpz_cmp_ui(n,1)==0) || Potenztest(n) ) // Faktorisierung vollstndig?
        { 
          cout << "factorization successfully completed." << endl;
          Factorization_to_file << endl;
	  
          // Streams schlieen
          SpecialRelations_buffer.close();
          DynamicRelations_buffer.close();
	  StaticRelations_buffer.close();
	  Recovery_buffer.close();

	  // Files lschen
	  remove(SpecialRelationsFile.c_str());
	  remove(DynamicRelationsFile.c_str());
	  remove(StaticRelationsFile.c_str());
	  remove(RecoveryFile.c_str());
	  exit(0); // Programm erfolgreich beenden...
        }

      /* Falls der Programmflu diese Stelle erreicht, so kann das jetzt
         nur noch durch Fall 3 geschehen:
         Fall 1: Ein defektierender Client hat einen falschen Faktor
                 als ECM-Faktor gemeldet -> schadet sonst nicht weiter,
                 da er nicht abgespalten konnte.
                 (dieser Fall wird nunmehr schon oben abgefangen)
         Fall 2: Ein langsamer Client hat im ECM-Mode als "Nachzgler" einen
                 bereits entdeckten Faktor aufgrund seiner veralteten Daten
                 "wiederentdeckt" -> schadet sonst nicht weiter, da der
                 Faktor nicht abgespalten werden konnte.
                 (dieser Fall wird nunmehr schon oben abgefangen)
         !!!!!!
         Fall 3: Ein Client hat in seiner ECM-Phase einen neuen Faktor
         !!!!!!  entdeckt, der nicht zu einer kompletten Faktorisierung
                 gefhrt hat. -> Da sich der Server aber bereits in der
                 Siebphase befindet (sehr wahrscheinlich sogar noch in der
                 Anfangsphase), sind extreme Laufzeitverluste zu befrchten,
                 da das Sieb nicht auf die nun verkleinerte Zahl umgestellt
                 werden kann.
                 -> Richtig schlimm wird es aber erst, wenn das Programm
                 abgebrochen wird, da die Recovery-Daten nun inkonsistent
                 sind. (Abgespaltener Faktor kann nur in den Recovery-Daten
                 gepeichert werden, wenn die Siebdaten gelscht werden!
                 Umgekehrt wre es unklug, den gefundenen Faktor einfach
                 zu ignorieren.)
                 Bemerkung: Fall 3 lt sich beim verteilten Faktorisieren
                 nicht ausschlieen, da der Server keine Annahmen ber den
                 Status der Clients treffen kann. Es bringt also nichts,
                 die Siebphase solange verzgern zu wollen, bis alle Clients
                 den ECM-Mode verlassen haben. (Abgesehen davon wre das
                 dazu quivalent, alle ECM-Faktoren in der Siebphase einfach
                 zu ignorieren.)

           <- diese Signatur dient nur dem schnellen Auffinden von "Programm-Baustellen"
          Bercksichtigung insbesondere des dritten Falls ist noch nicht
          implementiert. Am besten wre es wohl, die Recovery-Daten anzupassen
          und das Programm mit Neustart-Aufforderung zu beenden.
          Andererseits wird durch den "RECOVERY"-Marker in "factorizations.txt"
          sowieso auf mgliche Inkonsistenzen aufmerksam gemacht...

          BUG entdeckt (2000-02-09):
          Programmflu hat diese Stelle erreicht, Faktor wurde von n
          abdividiert, aber: dieser Client siebt ja spter mit! Und - das
          ist das fatale - er berechnet aufgrund des kleineren n einen
          anderen Vorfaktor! -> klar, da das zu inkonsistenten Daten fhrt...
          Da mu ich noch mal genau berlegen, wie man das am sinnvollsten
          beheben kann... (Man sieht mal wieder: Parallelitt fhrt zu
          sehr komplexen Fallunterscheidungen!)
          --> als Workaround Faktorisierung unvollstndig beenden.
      */

       {
          // Als Workaround wird jetzt erstmal die komplette Faktorisierung
          // mit dem Hinweis beendet, da der Restfaktor noch zusammengesetzt
          // ist...

          Factorization_to_file << MAL() << n << " [composite]" << endl;
          cout << "remaining factor: " << n << endl
               << "remaining factor is composite!" << endl;
          cout << "factorization needs to be stopped to avoid inconsistent files." << endl
               << "please restart factorization with remaining number." << endl;
          cout << "Faktorisierung mu abgebrochen werden, da ansonsten" << endl
               << "Inkonsistenzen bei den Files auftreten werden." << endl
               << "Bitte starten Sie das Programm mit der verbliebenen Zahl erneut." << endl;

          Factorization_to_file << endl;
	  
          // Streams schlieen
          SpecialRelations_buffer.close();
          DynamicRelations_buffer.close();
	  StaticRelations_buffer.close();
	  Recovery_buffer.close();

	  // Files lschen
	  remove(SpecialRelationsFile.c_str());
	  remove(DynamicRelationsFile.c_str());
	  remove(StaticRelationsFile.c_str());
	  remove(RecoveryFile.c_str());
	  exit(0); // Programm erfolgreich beenden...
        }
                 
      return;
    }

  if (s=="DynamicFactors?_ab_index")
    {
      // this feature is new!
      // Online-clients should use this one to fetch dynamic factors.
      cout << "new-style dynamic factor request from " << client << endl;

      process.unlock();
      int i;
      daten >> i;
      const int start = i;

      cout << "Sende dynamische Faktoren ab Index " << i
           << " an " << client << endl;
      for (int j=0; j<20; ++j)
       { 
         const int i_start = i;
         while (i<Dynamic_Factor_Relations.monitoredSize())
          {
            daten << Dynamic_Factor_Relations[i] << " ";
            ++i;
          }
         if (i>i_start)
          { 
            daten << flush;
            j=0;
            if (daten.fail()) break;
          }
         sleep(2);
       }
      daten << "-1" << endl;
      cout << "Dynamische Faktoren von " << start << " bis " << i-1
           << " an " << client << " bertragen." << endl;

      process.lock(); // it will be unlocked again by the calling function
      return;
    }
  
  if (s=="DynamicFactors?_ab_fpos")
    {
       cout << "old-style dynamic factor request from " << client << endl;
      // this feature is deprecated!
      // Online-clients should use the above one to fetch dynamic factors.
      // However, for offline clients and downwards compatibility with
      // client-versions up to 2.92 this feature is still provided...

      //streampos startfpos; // this is meant
      long long int startfpos; // and this will compile...
      daten >> startfpos;

      ostringstream temp_stringstream; // temporary to avoid long process locks
      cout << "Generiere dynamische Faktoren ab Fileposition " << startfpos << endl;
      const ostream::pos_type saved_fpos = DynamicRelations_to_file.tellp();
      DynamicRelations_to_file.seekp(0,ios::end);
#ifdef gcc3_4_workaround
      daten << static_cast<streamoff>(DynamicRelations_to_file.tellp()) << " ";
#else
      daten << DynamicRelations_to_file.tellp() << " ";
#endif
      DynamicRelations_to_file.seekp(saved_fpos);
      
      int prev=0; // wir bertragen die Abstnde
      for (TDynamic_Factor_Relations::iterator pos=Dynamic_Factor_Relations.begin();
	   pos != Dynamic_Factor_Relations.end(); ++pos)
#ifdef gcc3_4_workaround
	if (static_cast<streamoff>((*pos).fpos)>=startfpos)
#else
        if ((*pos).fpos>=startfpos)
#endif
	  {
	    temp_stringstream << (*pos).factor-prev << " ";
            prev=(*pos).factor;
	  }

      cout << "Sende dynamische Faktoren ab Fileposition " << startfpos << endl;
      process.unlock(); // allow other threads to sneak in
      if (!temp_stringstream.str().empty()) daten << temp_stringstream.str();
      daten << "-1" << endl;
      process.lock(); // it will be unlocked again by the calling function
      return;
    }
  
  if (s=="Polynom?")
    {
      int intervall;
      daten >> intervall;
      cout << "Sende neues Polynomintervall (" << intervall << ")"
           << " an " << client << endl;
      Polynom.Daten_sichern(daten); // Intervallanfang
      Polynom.naechstes_Polynom_ermitteln(intervall);
      Polynom.Daten_sichern(daten); // Intervallende
      
      // fr Recovery:
      streampos fpos = Recovery_to_file.tellp();
      Polynom.Daten_sichern(Recovery_to_file); // aktuelles Polynom abspeichern
      Recovery_to_file.flush(); // sicherheitshalber...
      Recovery_to_file.seekp(fpos);
      return;
    }

  // LAZY-Variante (ersetzt die nunmehr obsolete BUSY-Variante,
  // die bis einschlielich Version 2.92 als Option noch implementiert war):
  // Wenn die empfangenen Relationen Korrekturfaktoren enthalten, werden
  // diese erst dann ausgewertet, wenn eine statische Relation entsteht.
  // Diese Variante ist insbesondere fr das Client-Server-Modell
  // implementiert, da das relativ einfach ist und der Server alle
  // Relationen schnell entgegennehmen soll, was in der bisherigen
  // BUSY-Variante bei greren Gleichungssystemen nicht mehr gewhrleistet
  // werden konnte. Auerdem drfte die LAZY-Variante Plattenplatz sparen.
  // Nachteil: Der Check auf defektierende Relationen wird nun
  // unverhltnismig aufwendig. (Per Induktion gilt zwar: Wenn jede Relation
  // getestet wird, dann reicht es, alle Faktoren (einschlielich der 
  // Korrekturfaktoren) und den quadrierten Restfaktor zusammenzumultiplizieren
  // und das Produkt modulo n auf 1 zu testen. Die Korrektheit der zu den
  // Korrekturfaktoren gehrigen Relationen ergibt sich dann durch einen
  // Existenzcheck der zugehrigen Relation (und der Induktionsannahme, da
  // diese korrekt ist). Dieser Test ist also einfach zu implementieren, aber
  // whrend in der Busy-Variante der Test nicht allzu sehr ins Gewicht fllt,
  // nimmt er hier einen groen Teil der Verarbeitungszeit ein, da die
  // Relation nun nicht mehr als Zeichenkette verarbeitet werden kann, sondern
  // tatschlich ausgewertet werden mu. Gerade aber die Auswertung sollte
  // durch die LAZY-Methode auf das notwendige Ma beschrnkt werden.
  //
  // Vorgehensweise der LAZY-Variante:
  //  1. Relation als String empfangen
  //  2. Relation analysieren (factor.Typ ermitteln)
  //  3. Falls statische Relation -> wie gehabt verfahren (BUSY)
  //  4. Analyse-Routinen wie gehabt ausfhren
  //  5. String in das zugehrige File ablegen.
  if (s == "Relation!")
    {
      bool Relationsblock_okay = true;
      int anz_empfangene_Relationen = 0;
      do // Block von Relationen empfangen
        {
          static int counter = 100;
          if (--counter==0) { counter=100; Statusmeldung(); }

          process.unlock();
          // allow other threads to sneak in, because this
          // loop can take really a long time if we would do it atomicly!

          //cout << "Empfange bertragene Relation." << endl;
          anz_empfangene_Relationen++;

          ostringstream os; // hierin wird die Relation eingelesen

          //cout << "scanne Relation vom File" << endl;
          daten >> s; // Startzeichen
          if (s != "G")
           {
	     MARK;
             cerr << "error: wrong position while reading relation" << endl;
             exit(1);
           }

          CmpqsFactor factor;
          daten >> factor; // Faktor einlesen (dynamic-factor oder special-factor der Relation oder 1)
          os << "G " << setprecision(20) << factor; // << " ";
          while (daten.peek()!='\n')
           { 
             char line[1024];
             daten.get(line,sizeof(line),'\n'); os << line;
           }
          { char c; daten.get(c); os << c; }
          //cout << os.str();
      
          //daten.ignore(1,'\n');
          daten >> s;

          process.lock(); // but still each step should be processed atomicly...

          if (factor.Typ()==CmpqsFactor::static_prime)
	   {
	     //cout << "Habe statische Relation empfangen!" << endl;
             istringstream is(os.str()); // eingelesenen String aus Eingabe zur Verfgung stellen
             CRelation* GL = new CRelation();
             GL->multi_kombinieren_init();
             factor = GL->multi_kombinieren_main(is);
	     ++Erfolgsaufteilung[0];
             //cout << "Fge Relation ein." << endl;
	     Relation_einfuegen(GL,ohne_multi_kombinieren_init);
             continue;
	   }
          if (factor.Typ()==CmpqsFactor::single_large_prime)
	   {
	    // Dynamic Factor
	    TDynamicFactorRelation relation;
	    relation.factor = factor.int_value();
	    const TDynamic_Factor_Relations::iterator pos = Dynamic_Factor_Relations.find(relation); // dynamischen Faktor suchen
	    if (pos == Dynamic_Factor_Relations.end())
	      { // noch nicht bekannt
	        //relation.fpos = GL->speichern(DynamicRelations_to_file,relation.factor);
                relation.fpos=DynamicRelations_to_file.tellp();
                DynamicRelations_to_file << os.str(); // schnelles, weil direktes speichern
	        relation.SQRT_kN=SQRT_kN_mod_Primzahl(relation.factor);
	        Dynamic_Factor_Relations.insert(relation);
	        Special_Factoren_durch_Primfaktor_splitten_LAZY(relation.factor);
	      } 
	    else
	      { // schon bekannt
	        //cout << "Kann dynamische Relation zu statischer kombinieren!" << endl;
                istringstream is(os.str()); // eingelesenen String aus Eingabe zur Verfgung stellen
                CRelation* GL = new CRelation();
                GL->multi_kombinieren_init();
                factor = GL->multi_kombinieren_main(is);
                ++Erfolgsaufteilung[1];
                {
                  mpz_t x; mpz_init(x);
                  factor.in_mpz_ablegen(x); // factor in mpz-Variable x ablegen
	          mpz_mul(GL->Delta,GL->Delta,x); mpz_mod(GL->Delta,GL->Delta,n);
                  mpz_clear(x);
                }
	        GL->multi_kombinieren_main(DynamicRelations_from_file, (*pos).fpos);
                //GL->multi_kombinieren_exit();
	        Relation_einfuegen(GL,ohne_multi_kombinieren_init);
	      }
	    continue;
	   }
          if (factor.Typ()!=CmpqsFactor::double_large_prime)
           {
	     cerr << "Fehler: es wurde ein unbekannter Relationstyp bertragen!" << endl;
             continue;
           }
          // Special-Factor
          // Specialfactor mit Dynamicfactoren testen:
          /* Anmerkung:
	     Beim verteilten Sieben kann es passieren, da ein (notwendigerweise zusammengesetzter)
	     Special-Factor von einem Client entdeckt wird, der nicht mit dem aktuellen Satz der
	     dynamischen Faktoren siebt. - In diesem Fall, der mit steigender Anzahl der Clients immer
	     wahrscheinlicher wird, knnte es sinnvoll sein, Specialfaktoren auf ihre Teilbarkeit
	     mit den vorliegenden dynamischen Faktoren des Servers zu testen und ggf. zu splitten.
	     Allerdings bringt ein Verzicht auf diesen Test keine weiteren Nachteile mit sich.
	     Wir verzichten daher auf diese - nur beim verteilten Sieben mgliche - Prozedur.
          */
         // if(!Special_Factor_durch_Dynamicfactoren_splitten(factor)) // nicht implementiert (s.Anm.)
          {
	    // Echter Specialfactor
	    SpecialFactorRelation_LAZY_vermerken(factor,os.str());
          }
        } while (s=="Relation!");
      // Relationsblock ist eingelesen
      cout << "Relationsblock mit " << anz_empfangene_Relationen 
           << " Relationen von " << client << " eingelesen." << endl;
      if (!Relationsblock_okay) daten << "ignoriert!" << flush;
      else
       if (s=="Relationsblock_Ende") daten << "empfangen." << flush;
       else
        {
          cerr << "Relationsblock-Endekennung nicht erkannt!" << endl;
          daten << "Relationsblock_korrekt_beendet???" << flush;
        }
      Statusmeldung(true);
      return;
    }

  cerr << "Invalid request \"" << s << "\""
       << " from " << client << endl;
}


void* Cprocess_clients::THREAD_bearbeite_daten_stream(void* arg)
{
  int* pdata = static_cast<int*>(arg);
  const int communication_socket=*pdata;
  unix_buffer buffer(communication_socket);
  //cout << connection_info(buffer) << endl;
  iostream communication_stream(&buffer);

  process.lock();
  //cout << "processing thread..." << endl;
  bearbeite_daten_stream(communication_stream,peer_info(buffer));
  //cout << "... thread processed!" << endl;
  process.unlock();
  communication_stream.flush();
  close (communication_socket);

  delete pdata;
  return 0;
}

#endif /* IS_SERVER */

#ifdef IS_CLIENT
#ifdef USE_NETWORK

class CClientPolynomFetcher
{
private:
  static string buffer;
  static void * THREAD_fetch_polynom(void *);
public:
  static void fetch(mpz_t Obergrenze_D);
};
string CClientPolynomFetcher::buffer;

void * CClientPolynomFetcher::THREAD_fetch_polynom(void *)
{
  static unsigned int PolyInterval = 1000;
  static time_t prev_time = 0;
  if (PolyInterval>500 && time(NULL)>prev_time+300) PolyInterval=100*(PolyInterval/200);
  else if (time(NULL)<prev_time+120) PolyInterval+=100*(PolyInterval/200);
  prev_time=time(NULL);

  unix_io_client communication_stream2(communication_name, server_port);
  cout << "Hole Polynomintervall (" << PolyInterval << ")..." << endl;
  communication_stream2 << "Polynom? " << PolyInterval << endl;
  buffer.clear();
  while (!communication_stream2.eof())
   {
     char c;
     communication_stream2.get(c);
     buffer+=c;
   }
  cout << "background thread: Polynom fetched." << endl;
  return NULL;
}

void CClientPolynomFetcher::fetch(mpz_t Obergrenze_D)
{
  static pthread_t thread_fetch_polynom;

  int retcode;
  void * retval;

  static bool first_time_first_love = true;

  if (first_time_first_love)
   {
     first_time_first_love=false;
     retcode = pthread_create(&thread_fetch_polynom, NULL, THREAD_fetch_polynom, NULL);
     if (retcode != 0)
      {
        cerr << "CClientPolynomFetcher: pthread_create fehlgeschlagen!" << endl;
        exit(1);
      }
   }

  // Thread joinen
  cout << "Polynomfetch join..." << endl;
  retcode = pthread_join(thread_fetch_polynom, &retval);
  if (retcode != 0)
   {
     cerr << "CClientPolynomFetcher: Join fehlgeschlagen!" << endl;
     exit(1);
   }

  cout << "Setze Polynomintervall..." << endl;

  // Polynomdaten vom internen Klassen-Buffer einlesen
  istringstream is(buffer);
  Polynom.Daten_laden(is);
  cout << "Old Obergrenze_D = " << Obergrenze_D << endl;
  is >> Obergrenze_D;
  cout << "New Obergrenze_D = " << Obergrenze_D << endl;

  // und schonmal das nchste Intervall vielfdelnd im Hintergrund holen...
  retcode = pthread_create(&thread_fetch_polynom, NULL, THREAD_fetch_polynom, NULL);
  if (retcode != 0)
   {
     cerr << "CClientPolynomFetcher::pthread_create fehlgeschlagen!" << endl;
     exit(1);
   }
}


class CClientDynamicFactorFetcher
{
  static CMutex Mutex;
  static queue<int> buffer;
private:
  static void * THREAD_fetch_DynamicFactors(void *);
public:
  static void fetch();
};
CMutex CClientDynamicFactorFetcher::Mutex;
queue<int> CClientDynamicFactorFetcher::buffer;

void * CClientDynamicFactorFetcher::THREAD_fetch_DynamicFactors(void *)
{
 int sleep_secs = 100;
 while(true)
  {
   {
     unix_io_client communication_stream2(communication_name, server_port);
     cout << "Fetching dynamic factors from server..." << endl;
     static unsigned int dynfac_pos = 0;
     communication_stream2 << "DynamicFactors?_ab_index " << dynfac_pos << endl;

     int factor = 0;
     int counter = 0;
     while (true) // dynamische Relationen holen
      {
        communication_stream2 >> factor;
        // cout << factor << "\r" << flush;
        if (factor<=0) break;
        Mutex.lock(); buffer.push(factor); Mutex.unlock();
        ++counter;
      }
     dynfac_pos+=counter;
     cout << "background thread: " << counter << " dynamic factors fetched." << endl;
     if (counter>200) sleep_secs-=sleep_secs/4;
     else if (counter<20 && sleep_secs<10000) sleep_secs+=(sleep_secs+3)/4;
   }
   cout << "next dynamic factor fetch request in " << sleep_secs << " seconds." << endl;
   sleep(sleep_secs); // ein wenig warten...
  }
 return NULL;
}


void CClientDynamicFactorFetcher::fetch()
{
  static pthread_t thread_fetch_DynamicFactors;

  int retcode;
  static bool first_time_first_love = true;

  if (first_time_first_love)
   {
     first_time_first_love=false;
     retcode = pthread_create(&thread_fetch_DynamicFactors, NULL, THREAD_fetch_DynamicFactors, NULL);
     if (retcode != 0)
      {
        cerr << "CClientDynamicFactorFetcher: pthread_create fehlgeschlagen!" << endl;
        exit(1);
      }
   }

  Mutex.lock();
  if (buffer.empty())
   {
     Mutex.unlock();
     return;
   }

  cout << "Inserting " << buffer.size() << " dynamic factors..." << endl;

  // Dynamische Faktoren vom internen Klassen-Buffer einlesen
  TDynamicFactorRelation relation;
  relation.fpos=0; relation.factor=0;
  while (!buffer.empty()) // dynamische Relationen holen
   {
     relation.factor=buffer.front(); buffer.pop();
     // cout << relation.factor << "\r" << flush;
     relation.SQRT_kN=SQRT_kN_mod_Primzahl(relation.factor);
     Dynamic_Factor_Relations.insert(relation);
   }
  Mutex.unlock();
}


class CClientRelation_Versand
{
private:
  static volatile bool Sende_Relationen; // Semaphore
  static int  Sende_Nr;
  static string file_name;

  static void init(void);
  static const string create_unique_tempfilename(const string servername);
  static void cleanup_files(void);
  static void * THREAD_uebertrage_Relationen(void * arg);
  static void go(void);

  friend int main(const int argc, const char* const argv[]);
};
volatile bool CClientRelation_Versand::Sende_Relationen = true; // Semaphore
int  CClientRelation_Versand::Sende_Nr = 0;
string CClientRelation_Versand::file_name;


const string CClientRelation_Versand::create_unique_tempfilename(const string servername)
{
  ostringstream ostr;
  pid_t pid = getpid();
  struct utsname buf;
  uname(&buf);
  ostr << "/tmp/qsieve-" << buf.nodename
       << "-" << servername << "_" << pid;
  return ostr.str();
}

void CClientRelation_Versand::init(void)
{
  // first check whether the host is reachable
  // and get the hostname

  // refer "man getaddrinfo" and "man socket" 
  struct addrinfo hints; // our wishes are placed here
  memset(&hints,0,sizeof(hints)); // must be zeroed for default options
  hints.ai_family=PF_INET; // we want IPv4 as protocol
  hints.ai_socktype=SOCK_STREAM; // and we need a stream, not datagram!
  addrinfo *addrinfo_res = NULL; // here the result will be delivered
  const int retval = getaddrinfo(communication_name.c_str(),NULL,&hints,&addrinfo_res);
  if ( retval || addrinfo_res==NULL ) // any error?
   {
     cerr << "can't reach " << "\"" <<  communication_name << "\"" << endl;
     cerr << "Error given by getaddrinfo: " << endl;
     cerr << gai_strerror(retval) << endl;
     exit(1);
   }
  if (addrinfo_res->ai_socktype!=SOCK_STREAM) // we got a "stream"-protocol?
   {
     cerr << "provided protocol doesn't support SOCK_STREAM" << endl;
     exit(1);
   }

  freeaddrinfo(addrinfo_res); // free ressources allocated by getaddrinfo

  // create an unique tempfilename based on the hostname
  file_name = create_unique_tempfilename(communication_name);
}

void CClientRelation_Versand::cleanup_files(void)
{
  string name;
  name = file_name+"A";
  unlink(name.c_str());
  name = file_name+"B";
  unlink(name.c_str());
}

void * CClientRelation_Versand::THREAD_uebertrage_Relationen(void * arg)
{
  string s;
  ifstream file_stream;
  char line[1024];

  //cout << "Sende Daten" << endl; // stdout should be handled with mutual exclusion
  const string name = file_name + static_cast<char*>(arg);
  file_stream.open(name.c_str(), ios::in);
  while (file_stream.peek()!=EOF)
   {
     unix_io_client communication_stream2(communication_name, server_port);
     int count=2500+1; // maximimal count-1 Relationen am Block bertragen
     const int max_Uebertragungsdauer = 8; // "Maximale Sendezeit" in Sekunden (damit auch andere drankommen knnen...)
     const time_t Uebertragungsstart = time(NULL);
     while (file_stream.peek()!=EOF && --count && time(NULL)-Uebertragungsstart<=max_Uebertragungsdauer)
      {
        communication_stream2 << "Relation! ";
	while (file_stream.peek()!='\n')
	 {
	   file_stream.get(line,sizeof(line),'\n');
	   communication_stream2 << line;
	 }
	char c; file_stream.get(c);
	communication_stream2 << endl;
      }
     communication_stream2 << "Relationsblock_Ende " << flush;
     communication_stream2 >> s;
     if (s!="empfangen.")
       cerr << "Empfang der Relation nicht besttigt!" << endl;
     if (s=="ignoriert!")
      {
        cerr << "Oops! - Relationen passen dem Server nicht!" << endl;
        exit(1);
      }
   }
  unlink(name.c_str());
  sleep(10); // damit bei schnellen Rechnern nicht permanent neue Verbindungen geffnet werden
  Sende_Relationen = true; // neuen Thread fr bertragung anstoen...
  return NULL;
}

void CClientRelation_Versand::go(void)
{
  static pthread_t thread_Relationen_versenden;
  static char* SendeSuffix[2] = { "A", "B" };

  if (Sende_Relationen)
   {
     // Relationen sollen bertragen werden:
     // a) -> Flag setzen, da bertragung angestoen wurde
     //       (d.h. Sende_Relationen auf false setzen)
     // b) Thread zum bertragen erzeugen
     // c) dieser Thread setzt das Flag bei Beendigung
     
     Sende_Relationen = false;
     
     // close file
     communication_stream.close();

     int retcode;
     void* retval;

     static bool first_time_first_love = true;

     if (first_time_first_love)
      { // vor dem ersten Thread nicht joinen, ...
        first_time_first_love=false;
        retcode = pthread_create(&thread_Relationen_versenden, NULL,
                    CClientRelation_Versand::THREAD_uebertrage_Relationen, reinterpret_cast<void*>(SendeSuffix[Sende_Nr]));
      }
     else
      { // ... danach schon...
        retcode = pthread_join(thread_Relationen_versenden, &retval);
        if (retcode != 0)
         {
           cerr << "Join fehlgeschlagen!" << endl;
           exit(1);
         }
        retcode = pthread_create(&thread_Relationen_versenden, NULL,
                    CClientRelation_Versand::THREAD_uebertrage_Relationen, reinterpret_cast<void*>(SendeSuffix[Sende_Nr]));
      }

     if (retcode != 0)
      {
        cerr << "Thread zum bertragen der Relationen fehlgeschlagen!" << endl;
        exit(1);
      }

     Sende_Nr = 1 - Sende_Nr;
     // open new file
     const string name = file_name + SendeSuffix[Sende_Nr];
     communication_stream.open(name.c_str() ,ios::out|ios::trunc);
   }
}

#endif
#endif

#ifdef IS_SERVER

void catch_sigint(int )
{
  exit(1);
}

void close_server(void)
{
  close(server_socket);

#ifdef CLIENT_AUTOSTART
  // aufgerufen durch atexit, d.h. wir beenden uns eh gleich
  signal(SIGTERM, SIG_IGN);
  // kille alle prozesse meiner gruppe (auch mich aber ich ignoriere, s.o.)
  kill(0,SIGTERM);
#endif
}
#endif

int main(const int argc, const char* const argv[])
{
  cout.setf(ios::fixed); // Dezimaldarstellung, nicht Scientific!

  cout << " --------------------------------------" << endl;
  cout << "-     Faktorisierungsprogramm V2.94    -" << endl;
  cout << "- written 2004-01 by Thorsten Reinecke -" << endl;
  cout << "-      (thre@thorstenreinecke.de)      -" << endl;
  cout << " --------------------------------------" << endl;
#ifdef __GNUG__
  {
    const string s=__VERSION__;
    cout << "(" << __DATE__ << ", " << __TIME__ << " using ";
    if (s[0]<='9') cout << "gcc ";
    cout << __VERSION__ << ")" << endl;
  }
#endif
  cout << endl;
  cout << "This program may be freely distributed under the" << endl;
  cout << "terms of the GNU-Public-Licence." << endl;
  cout << endl;
    
  if (argc>2)
    {
#ifdef IS_CLIENT
       cerr << "servername expected!" << endl;
#else
      cerr << "number for factorization expected!" << endl;
#endif
      exit(0);
    }
  
  const bool recover = (argc==1); // ohne Argument: Recovery-Modus

  {
   // resolve ConfigFile
   if (getenv("QSIEVE_CFG"))
    {
      ConfigFile=getenv("QSIEVE_CFG");
      cout << "Setting ConfigFile to \"" << ConfigFile << "\"" << endl;
    }
   // and convert it to real/absolute path
#ifdef PATH_MAX
   char resolved_path[PATH_MAX];
#else
   char resolved_path[4096];
#endif /* PATH_MAX */
   if (realpath(ConfigFile.c_str(),resolved_path))
    ConfigFile=resolved_path;
   else
    {
      cerr << "Cannot find ConfigFile " << "\"" << ConfigFile << "\"" << endl;
      // server & standalone versions need a configuration file,
      // clients do not need them
#ifndef IS_CLIENT
      exit(1);
#endif
    }
  }

  mpz_init(n); /* Variable fr zu faktorisierende Zahl bereitstellen */
  mpz_init(kN);
  mpz_init(r);

#ifndef IS_CLIENT
  Read_ConfigFile();

  if (!recover) // Nicht im Recovery-Modus -> Argumente auswerten
    { 
      // bergebene Zahl holen (und ggf. ausrechnen)
      char* const neuer_str = strdup(argv[1]);
      char* str = neuer_str; 
      if (!parse_term::get_Zahl(n, str)) 
	{
	  cout << "Wrong input at: '" << str << "'" << endl;
	  exit(1);
	}
      else
	if (str[0]!='\0')
	  {    
	    cout << "Syntax error in input term. (parenthesis?)" << endl;
	    exit(1);
	  }
	else
	  if (mpz_cmp_ui(n,0)<=0)
	    {
	      cout << "Input must be positive natural number!" << endl;
	      exit(1);
	    }
      free(neuer_str); // kein delete [] wegen stdup/malloc
    }

  if (recover)
    {
      // Recovery: continue a factorization, which was previously aborted
      // (try to) open the necessary file streams
      StaticRelations_buffer.open(StaticRelationsFile.c_str(),ios::in|ios::out);
      SpecialRelations_buffer.open(SpecialRelationsFile.c_str(),ios::in|ios::out);
      if (!DynamicRelations_buffer.open(DynamicRelationsFile.c_str(),ios::in|ios::out))
       {
         cerr << "Cannot access " << DynamicRelationsFile << endl;
         exit(1);
       }
      Recovery_buffer.open(RecoveryFile.c_str(),ios::in|ios::out|ios::ate);

      // offenbar wird trotz ios::ate nicht (immer) ans Ende positioniert
      // deshalb wird die Testabfrage modifiziert:
      if (Recovery_from_file) Recovery_from_file.seekg(0,ios::end);
#ifdef gcc3_4_workaround
      if ( (!Recovery_from_file) || (Recovery_from_file.tellg()==0) || (Recovery_from_file.tellg()==-1) )
// tellg()==0 indicates empty file -> we cannot recover
// tellg()==-1 indicates failure -> we cannot recover
//  remark:
//  in gcc 3.4 (cvs-experimental-2003-10-17 we cannot compare with "<" !!)
//  do we really need a workaround to check this condition? 
#else
      if ( (!Recovery_from_file) || (Recovery_from_file.tellg()<1) )
#endif /* gcc3_4_workaround */
	{
	  cerr << "Recovery not possible!" << endl;
	  exit(1);
	}
      
      Recovery_from_file.seekg(0,ios::beg);
      Recovery_to_file.seekp(0,ios::beg);
      Recovery_from_file >> n; // Zahl holen
      Recovery_from_file.ignore(1,'\n');

      cout << "Continue factorizaction for " << endl;
      cout << n << endl;
      // auerdem Recovery-Modus im Factorization-File vermerken, denn es
      // 1. knnte gelscht sein
      // 2. knnte der Recovery-Modus in seltenen Fllen Faktoren unterdrcken
      //    (wenn vorher das Schreiben unterbrochen wurde)
      // 3. knnte der Recovery-Modus zu doppelten Faktor-Ausgaben fhren
      //    (falls das quadratische Sieb einen zusammengesezten Faktor
      //    gefunden hat und dann noch unterbrochen wird)
      // 4. knnte das Multiplikationssymbol inkorrekt hufig auftreten
      //    (wenn vorher kein Faktor gefunden wurde)
      Factorization_to_file << " [RECOVERY] "; // ohne flush, falls vor Finden weiterer Faktoren erneut abgebrochen wird... 
    }
  else
    {
      // Neustart: Starten einer neuen Faktorisierung
      // Streams ffnen
      StaticRelations_buffer.open(StaticRelationsFile.c_str(),ios::out|ios::trunc);
      SpecialRelations_buffer.open(SpecialRelationsFile.c_str(),ios::in|ios::out|ios::trunc);
      if (!DynamicRelations_buffer.open(DynamicRelationsFile.c_str(),ios::in|ios::out|ios::trunc))
       {
         cerr << "Cannot access " << DynamicRelationsFile << endl;
         exit(1);
       }

      Recovery_buffer.open(RecoveryFile.c_str(),ios::in|ios::out|ios::trunc);
      
      cout << "Starting factorization for" << endl;
      cout << n << endl;
      
      Factorization_to_file << endl << "Factorization of " << argv[1] << ":" << endl;
      Factorization_to_file << n << " = " << endl;
      
      easy_factor(); // "einfache" Faktoren rausstreichen
      
      cout << "Starting factorization with ECM and MPQS for" << endl;
      cout << n << endl;
      Recovery_to_file << n << endl;           
    }

#ifdef IS_SERVER
  {
    int i=0;
    while ((server_socket = open_internet_port(server_port,10))<0
	   && ++i < server_port_tries)
      sleep(1);  
  }
  atexit(close_server);
  signal(SIGINT, catch_sigint);
  if (server_socket<0) {
    cerr << "Kann Port " << server_port << " nicht oeffnen" << endl;
    exit(1);
  }
  tune_parameters(mpz_sizeinbase(n,10)); // Parametertuning
  cout << "Waiting for net-clients (ECM)..." << endl;
#ifdef CLIENT_AUTOSTART
  {
    string s="autostart_clients.sh ";
    struct utsname buf;
    setpgid(0,0);
    uname(&buf);
    s += buf.nodename;
    cout << "Calling script to start clients..." << endl;
    cout << "(exitstatus=" << system(s.c_str()) << ")" << endl;
  }
#endif /* CLIENT_AUTOSTART */
  if (!SkipECM) bearbeite_daten_stream_ecm(server_socket); // erstmal ECM verteilt starten...
#endif /* IS_SERVER */

#ifdef IS_STANDALONE
  // Auch die Standalone-Version sollte elliptische Kurven benutzen.
  if (!SkipECM) bearbeite_ecm(); // erstmal ECM starten...
#endif /* IS_STANDALONE */

#ifdef USE_DFT
  // an dieser Stelle sind in allen Programmvarianten die
  // Faktorisierungsalgorithmen, die discrete fast fourier transform
  // verwenden knnten, abgeschlossen.
  // Das DFT-Objekt kann also wieder freigegeben werden...
  {
    mpz_t x;
    mpz_init_set_ui(x,19); polynomial::get_dft(0,x); // Lschen des DFT-Objektes triggern...
    mpz_clear(x);
  }
#endif /* USE_DFT */

  // nun wird alles fr das Quadratische Sieb vorbereitet...

  bestes_kN_ermitteln(); /* optimaler Wert, fr den gesiebt werden kann */

  /* Streitfall:
     Soll man nach Ermittlung des bestem Multipliers die Siebparameter
     noch tunen?
     Ja   -> denn die Zahl kN hat ja die entsprechende Gre...
     Nein -> denn fr die Wahl des Multipliers wurden diese Gren nicht
             bercksichtigt und entscheidend sollte die Gre der
             zu faktorisierenden Zahl sein. Nur so ist eine Vergleichbarkeit
             der Multiplier unter gleichen Voraussetzungen zu gewhrleisten.
    Wie auch immer:
     Eine "optimale" Ermittlung der Siebgre und der Faktorbasis kann nur
     mittelbar von der Gre der Zahl abhngen. Die Laufzeit der Siebphase
     hngt davon ab, wie viele Relationen zu ermitteln sind *und* wie
     schnell die Relationen ermittelt (=gesiebt) werden knnen!
     Letzterer Wert ist funktional von der Gre der Zahl und der Qualitt
     der Faktorbasis abhngig. Diese Qualitt ist zwar durch Vorgabe eines
     Multipliers (nach Annahme) maximiert worden; aber das heit noch nicht,
     da dadurch die funktionale Abhngigkeit beseitigt wre.
     Mit anderen Worten: Wenn wir die Bestimmung der Faktorbasisgre und
     der Siebgre *nur* von der Gre der Zahl abhngig machen, so gehen
     wir idealisierend von einer einheitlichen Qualitt der Faktorbasis aus.
     Da wir diese einheitliche Qualitt der Faktorbasis nicht definiert haben,
     besitzen wir auch keine Entscheidungsgrundlage darber, ob fr unsere
     Zahl (n bzw. kN) eine unterdurchschnittliche, durchschnittliche oder
     berdurchschnittliche Faktorbasisqualitt besteht. Aus dieser Sichtweise
     erscheint der obige Streitfall nicht entscheidbar zu sein.
  */

#if 0   
  tune_parameters(mpz_sizeinbase(kN,10)); // Parametertuning fr kN
#else
  tune_parameters(mpz_sizeinbase(n,10)); // Parametertuning fr n
#endif
  /* Hinweis:
      Beim Tuning der Parameter ist in der verteilten Version auf Konsistenz
      zu achten, d.h. ein Client sollte keine Parameter tunen!
      Unterschiedliche cfg-Files auf verschiedenen Rechnern wrden sonst
      zu Problemen fhren!
  */

#else /* IS_CLIENT */
  if (recover)
    {
      cerr << "Fehlender Parameter" << endl;
      exit(1);
    }
  communication_name = argv[1];
#ifdef USE_NETWORK
  CClientRelation_Versand::init();

  { // zunchst elliptische Kurven abarbeiten (net-client):
    int ecm_sigma = -1, runde=0;
    while (true)
     {
       unix_io_client communication_stream(communication_name, server_port);
       communication_stream << "ECM?" << endl;
       communication_stream >> ecm_sigma;
       if (ecm_sigma<0) break; // sigma<0 -> keine Kurven mehr!
       communication_stream >> elcu_Phase1 >> elcu_Phase2;
       communication_stream >> n; // Zahl holen
       cout << "elliptic curve " << ++runde << endl;
       elliptic_curves elliptic_curve; // Instanz der Kurve kreieren
       elliptic_curve.go(ecm_sigma,elcu_Phase1,static_cast<elliptic_curves::NATNUM>(elcu_Phase2)); /* Faktorisierung mit elliptischer Kurve versuchen */
     }
  }
  
  {
    unix_io_client communication_stream(communication_name, server_port);
    communication_stream << "Hunger!" << endl;
    communication_stream >> n;
    communication_stream >> Primbasis_size;
    communication_stream >> Factor_Threshold;
    communication_stream >> M;
    cout << "n=" << n << endl;
    cout << "Gre der statischen Faktorbasis: " << Primbasis_size << endl;
    cout << "Factor-Threshold (Exponent): " << Factor_Threshold << endl;
    cout << "Siebintervall pro Polynom: [" << -M << "," << M << "]" << endl;
  }
  {
    string name;
    name = CClientRelation_Versand::file_name+"A";
    communication_stream.open(name.c_str(), ios::out|ios::trunc);
  }
  // atexit(CClientRelation_Versand::cleanup_files); // Vorsicht: Bei Absturz sind dann noch nicht bertragene Relationen futsch!
#else /* no USE_NETWORK */
  // existiert das file?
  ifstream communication_stream2(communication_name.c_str());
  if (!communication_stream2)
    {
      cerr << "Kann File " << communication_name << " nicht oeffnen" << endl;
      exit (1);
    }
  communication_name+=".out";
  
  communication_stream2 >> n; 
  communication_stream2 >> Primbasis_size;
  communication_stream2 >> Factor_Threshold;
  communication_stream2 >> M;
  communication_stream2.ignore(1,'\n');
  cout << "n=" << n << endl;
  cout << "Gre der statischen Faktorbasis: " << Primbasis_size << endl;
  cout << "Factor-Threshold (Exponent): " << Factor_Threshold << endl;
  cout << "Siebintervall pro Polynom: [" << -M << "," << M << "]" << endl;
  communication_stream.open(communication_name.c_str(), ios::out|ios::trunc);
#endif /* USE_NETWORK */
  bestes_kN_ermitteln(); /* optimaler Wert, fr den gesiebt werden kann */
#endif /* not IS_CLIENT */
  
  if (mpz_cmp_ui(r,Siebgroesse)<0)
    {
      cerr << "Sieb ist berdimensioniert (ggf. verkleinern)!" << endl;
      exit(1);
    }

  // Fr Double-Large-Primes einen Schwellwert setzen,
  // dieser ist das Quadrat des maximalen Specialfactors...
  mpz_init(CmpqsFactor::DLP_Threshold);
  mpz_set_ui(CmpqsFactor::DLP_Threshold,Specialfactor_Sieb_Threshold);
  mpz_mul(CmpqsFactor::DLP_Threshold,CmpqsFactor::DLP_Threshold,CmpqsFactor::DLP_Threshold);
  
  Statische_Faktorbasis_ermitteln();
  cout << "The first 20 members of the static factorbase are" << endl;
  for (int i=0; i<20; i++) cout << Primzahlen[i] << " "; cout << endl;
  cout << "Biggest prime in static factorbase: " << biggest_Prime_in_Factorbase << endl;
  cout << "#squares (of primes) in FB: " << FB_maxQuadrate << endl;
  cout << "#powers (of primes) in FB: " << anz_weitere_Primzahlpotenzen << endl;

  SiebRegler::SiebSchwellwert_kalkulieren(); // Schwellwert zur Erkennung der Relationen fr Siebvorgang bestimmen
  Sieb=new TSiebElement [Siebgroesse]; /* Speicher fr das Sieb allokieren */

  Polynom.erstes_Polynom_ermitteln(kN,M); // erstes Polynom ermitteln
  
#ifndef IS_CLIENT
  if (recover)
    {
      // Static Relations einlesen
      cout << "Reading static relations from file..." << endl;
      StaticRelations_from_file.seekg(0,ios::beg);
      while (StaticRelations_from_file.peek()=='G') 
	{
	  CRelation* GL = new CRelation();
	  GL->kombinieren(StaticRelations_from_file);
          GL->optisize(); // Relation optimieren (dense<->sparse, etc.)
          GLS[GL->relevanter_Faktor]=GL; Fuellung_GLS++; // Relation ins Gleichungssystem einfgen
	  StaticRelations_from_file.ignore(1,'\n');
	}
      StaticRelations_from_file.clear();
      cout << "Recover: " << Fuellung_GLS << " static relations read." << endl;

      // Dynamic Relations einlesen
      cout << "Reading dynamic relations from file..." << endl;
      DynamicRelations_from_file.seekg(0,ios::beg);
      while (DynamicRelations_from_file.peek()=='G') 
	{
	  TDynamicFactorRelation relation;
	  relation.fpos = DynamicRelations_from_file.tellg();
	  string s; DynamicRelations_from_file >> s;
	  DynamicRelations_from_file >> relation.factor;
	  relation.SQRT_kN=SQRT_kN_mod_Primzahl(relation.factor);
	  DynamicRelations_from_file.ignore(1000000,'\n');
	  Dynamic_Factor_Relations.insert(relation);
	}
      DynamicRelations_from_file.clear();
      DynamicRelations_to_file.seekp(0, ios::end); // sicherheitshalber...
      cout << "Recover: " << Dynamic_Factor_Relations.size()
	   << " dynamic relations read." << endl;
      {
        // Special Relations einlesen
        // zustzlich Speicherkompaktifizierung der special relations
        // auf dem File durchfhren, d.h. ungltige (=als redundant
        // markierte) Relationen streichen.
        cout << "Reading special relations and reorganizing file..." << endl;
        unsigned int u_hits = 0, g_hits = 0;
        streampos rpos=0, wpos=0;
        // Special Relations einlesen
        SpecialRelations_from_file.seekg(0,ios::beg);
        SpecialRelations_to_file.seekp(0,ios::beg);

        while (SpecialRelations_from_file.peek()=='G')
         {
           TSpecialFactorRelation relation;
           relation.fpos = SpecialRelations_from_file.tellg();
	   string s; SpecialRelations_from_file >> s; // "G " einlesen
           SpecialRelations_from_file >> relation.factor; // special relation factor einlesen
           Special_Factor_Relations.insert(relation);
           relation.factor.swap();
           Special_Factor_Relations.insert(relation);
           relation.factor.swap();
	   SpecialRelations_from_file.ignore(1000000,'\n'); // ungltige Zeilen berlesen
           ++g_hits;
          }
        wpos=rpos=SpecialRelations_from_file.tellg(); //Schreibposition=Leseposition

        while (true) 
	  {
            SpecialRelations_from_file.seekg(rpos);
	    while (SpecialRelations_from_file.peek()=='U')
             {
	       SpecialRelations_from_file.ignore(1000000,'\n'); // ungltige Zeilen berlesen
               ++u_hits;
             }

            int cached_g = 0;
	    const int max_cached_g = 1000;
	    string sG[max_cached_g];
	    while (SpecialRelations_from_file.peek()=='G')
             {
	       while (SpecialRelations_from_file.peek()!='\n')
	        {
                  char buffer[2048];
                  SpecialRelations_from_file.get(buffer, sizeof(buffer), '\n');
                  sG[cached_g]+=buffer;
                }
               SpecialRelations_from_file.ignore(10,'\n');
	       if (++cached_g>=max_cached_g) break;
	       while (SpecialRelations_from_file.peek()=='U')
                {
	          SpecialRelations_from_file.ignore(1000000,'\n'); // ungltige Zeilen berlesen
                  ++u_hits;
                }
             }
            rpos=SpecialRelations_from_file.tellg();
            if (cached_g==0) break;
            SpecialRelations_to_file.seekp(wpos);

	    for (int i=0; i<cached_g; ++i)
             {
	       TSpecialFactorRelation relation;
	       relation.fpos = SpecialRelations_to_file.tellp();
               {
                 istringstream is(sG[i]);
	         string s; is >> s;
                 is >> relation.factor;
               }
	       SpecialRelations_to_file << sG[i] << "\n";
	       Special_Factor_Relations.insert(relation);
	       relation.factor.swap();
	       Special_Factor_Relations.insert(relation);
               relation.factor.swap();
             };
            SpecialRelations_to_file.flush();
            wpos=SpecialRelations_to_file.tellp();
            g_hits+=cached_g;
            cout << g_hits << " special relations written, "
                 << u_hits << " removed." << "\r" << flush;
	    //cout << (long long int)static_cast<streamoff>(rpos) << " , " << (long long int)static_cast<streamoff>(wpos) << endl;
          }
         cout << endl;
         
        // mangels truncate-Funktion den Rest mit U's auffllen
        {
          streampos pos, ende;
          pos = wpos;
          SpecialRelations_to_file.seekp(0, ios::end);
          ende = SpecialRelations_to_file.tellp();
	  SpecialRelations_to_file.seekp(wpos);
          //cout << (long long int)static_cast<streamoff>(pos) << " , " << (long long int)static_cast<streamoff>(ende) << endl;

          char buffer[4096];
          for (unsigned int i=0; i<sizeof(buffer); ++i) buffer[i]='U';
          while (static_cast<streamoff>(SpecialRelations_to_file.tellp()) < static_cast<streamoff>(ende))
            {
              SpecialRelations_to_file.write(buffer,sizeof(buffer));
              //cout << (long long int)static_cast<streamoff>(SpecialRelations_to_file.tellp()) << " , " << (long long int)static_cast<streamoff>(ende) << endl;
            }
          SpecialRelations_to_file << flush;
          SpecialRelations_to_file.seekp(pos);
        }
        SpecialRelations_from_file.clear();
        cout << "Recover: " << Special_Factor_Relations.size()
	     << " special relations read, " << endl
             << u_hits << " invalid DLP-relations eliminated." << endl;
      }
      SpecialFactors_Zyklensuche();
      
      
#ifdef IS_SERVER
      // beim verteilten Sieben knnten den Clients einige potentielle dynamische Faktoren
      // durch die Lappen gegangen sein. Dies geschieht, falls ein Client einen Special-Factor
      // findet, der nicht mit dem vollen Satz der aktuellen (und daher nur dem Server bekannten)
      // dynamischen Faktoren gesiebt wurde.
      // Anstelle eines Tests zur Laufzeit werden diese im Normalfall eher seltenen Flle
      // nun hier beim Neu-Hochfahren des Servers bercksichtigt:
      // WICHTIG: erst mssen alle dynamischen Faktoren eingelesen sein! Ansonsten knnen
      //          doppelt auftretende dynamische Faktoren nicht zu einer Relation reduziert werden!
      // folgende Anweisungen wrden das bewerkstelligen, aber es lohnt sich kaum und dauert (zu) lange:
#if 1 /* 0: wir lassen es, 1: wir machen es (trotzdem) */
      cout << "Try to discover previously hidden special factors." << endl;
      for (TDynamic_Factor_Relations::iterator pos = Dynamic_Factor_Relations.begin();
	   pos!=Dynamic_Factor_Relations.end(); ++pos)
	Special_Factoren_durch_Primfaktor_splitten_LAZY((*pos).factor);
      cout << "discovered dynamic factors: " << Special_to_dynamic_Factor_hit << endl;
#endif
#endif
      
    }
#else // is client:
#ifndef USE_NETWORK 
  TDynamicFactorRelation relation;
  relation.fpos = 0; relation.factor=0;
  int diff; // wir lesen die Abstnde zwischen den Faktoren ein
  communication_stream2 >> diff; // filepos ueberlesen
  while (true)
    {
      communication_stream2 >> diff;
      relation.factor+=diff;
      // cout << "*" << relation.factor << "*" << flush;
      if (diff<=0) break;
      relation.SQRT_kN=SQRT_kN_mod_Primzahl(relation.factor);
      Dynamic_Factor_Relations.insert(relation);
    }
  //cout << endl;
#endif
#endif  
  
  
  
#ifndef IS_CLIENT
  if (recover)
    {
      streampos fpos = Recovery_from_file.tellg();
      Polynom.Daten_laden_wenn_vorhanden(Recovery_from_file); // aktuelles Polynom (zum Fortsetzen) ermitteln
      Recovery_to_file.seekp(fpos);
    }
#endif

Statuslegende_ausgeben();

#ifdef IS_SERVER
  cout << "Waiting for clients..." << endl;

  while (true)
    {    
      
      int communication_socket = accept(server_socket, NULL, NULL);
      if (communication_socket > 0)
	{
          int* pdata = new int;
          *pdata = communication_socket;
	  pthread_attr_t detached_thread;
          pthread_t thread_process_client_data;

          pthread_attr_init(&detached_thread);
	  pthread_attr_setdetachstate(&detached_thread, PTHREAD_CREATE_DETACHED);

          //cout << "creating thread..." << endl;
          const int retcode = pthread_create(&thread_process_client_data,&detached_thread,
                                   Cprocess_clients::THREAD_bearbeite_daten_stream, pdata);
          if (retcode != 0)
           {
             cerr << "pthread_create failed!" << endl;
             exit(1);
           }
	}
      
      // collecting_phase_beendet= ( Fuellung_GLS > 0.6*Primbasis_size );
      /* Durch das Flag "collecting_phase_beendet" ist es mglich, das Aufsammeln weiterer dynamischer
	 oder special-Faktoren zu unterbinden. Dies knnte ratsam sein, wenn der Haupt- oder auch
	 der Hintergrundspeicher knapp wird oder das Programm schon solche Fortschritte im
	 Sammeln der eigentlichen Relationen macht, da dynamische oder special-Faktoren hinderlich
	 wirken.
	 Da aber eine Speicherknappheit bei UNIX nicht sinnvoll ermittelt werden kann (andere Prozesse
	 knnen jederzeit Speicher anfordern/freigeben), empfiehlt es sich eher, das Programm
	 bei Speicherknappheit abzubrechen, die dynamic_relations-Datei zu krzen/zu lschen und
	 das Programm anschlieend neu zu starten. Auf diese Weise kann man die Speichernutzung
	 des Programms wesentlich flexibler beeinflussen.
      */
      
    }

// Server
#else
// not Server


#ifdef IS_CLIENT
  mpz_t Obergrenze_D;
  mpz_init(Obergrenze_D);
    
#ifdef USE_NETWORK

 ClientMainLoop: // Hauptschleife: Polynomintervall holen und diese durchsieben

  CClientPolynomFetcher::fetch(Obergrenze_D);

#else // kein Netzwerk
  Polynom.Daten_laden(communication_stream2);
  communication_stream2 >> Obergrenze_D;
#endif // if Netzwerk else keinNetzwerk
#endif // Client
  
  /* Quadratischen Reste mit Primzahlen aus der Primzahlbasis aussieben */
  do /* solange Relationen aussieben, bis mehr Relationen gefunden sind
	als Primzahlen in der Primzahlbasis sind (oder eine Faktorisierung
	gefunden ist). */
    {
#ifdef USE_NETWORK
      CClientDynamicFactorFetcher::fetch();
#endif

#ifndef IS_CLIENT
      // fr Recovery:
      streampos fpos = Recovery_to_file.tellp();
      Polynom.Daten_sichern(Recovery_to_file); // aktuelles Polynom abspeichern
      Recovery_to_file.seekp(fpos);
#endif

      Sieboffset=-M;
      alle_Deltas_neu_berechnen();
#ifdef USE_FAKEHEAP
      Sieb_Delta_Heap.sort();
#endif
      while (Sieboffset<M)
	{
	  Sieb_initialisieren();
	  schnelles_aussieben(); // mit Primzahlen der statischen Faktorbasis
	  aussieben_mit_Quadraten(); // (grere) Quadrate von statischen Faktoren sieben
	  aussieben_mit_Dynamic_Factoren(); // ermglicht weitere Relationen...
	  schnelles_auslesen_des_Siebs();
	  Sieboffset+=Siebgroesse;
	}
      Statusmeldung(); // Ausgabe von Informationen

#if defined(IS_CLIENT) && defined(USE_NETWORK)
      CClientRelation_Versand::go();
#endif  // Client && Network

      Polynom.naechstes_Polynom_ermitteln();
      
      // collecting_phase_beendet= ( Fuellung_GLS > 0.6*Primbasis_size );
      /* Durch das Flag "collecting_phase_beendet" ist es mglich, das Aufsammeln weiterer dynamischer
	 oder special-Faktoren zu unterbinden. Dies knnte ratsam sein, wenn der Haupt- oder auch
	 der Hintergrundspeicher knapp wird oder das Programm schon solche Fortschritte im
	 Sammeln der eigentlichen Relationen macht, da dynamische oder special-Faktoren hinderlich
	 wirken.
	 Da aber eine Speicherknappheit bei UNIX nicht sinnvoll ermittelt werden kann (andere Prozesse
	 knnen jederzeit Speicher anfordern/freigeben), empfiehlt es sich eher, das Programm
	 bei Speicherknappheit abzubrechen, die dynamic_relations-Datei zu krzen/zu lschen und
	 das Programm anschlieend neu zu starten. Auf diese Weise kann man die Speichernutzung
	 des Programms wesentlich flexibler beeinflussen.
      */
    }
#ifndef IS_CLIENT
  while (true);
#else
  while (Polynom < Obergrenze_D);
#ifdef USE_NETWORK
  goto ClientMainLoop; // Hauptschleife: neues Polynomintervall holen und fortsetzen
#endif
  mpz_clear (Obergrenze_D);
#endif
  
  delete [] Sieb; /* Sieb lschen, d.h. Speicher wieder freigeben */

#endif /* not server */
  
  cout << endl << "Fertig..." << endl;

  return 0;
}
