#ifndef qsieve_header
#define qsieve_header

/*! @file
 * @brief
 * main header file of the "Faktorisierungsprogramm"
 */


#include "utils.H"
#include "TinyVector.H"
typedef CTinyVector<FBsizetype> CTinyFBsizetypeVector;

const int no_factor = -1;
const int dynamic_factor = -2;
const int special_factor = -3;

class CProvideHilfsvariablen : private ForbidAssignment
{
 // it's not nice, but better than providing temporary mpz_t x,y
 // in the global namespace: threadsafe & restricted to the class
 // And we have no need to change the source code at 1000 places...
 protected:
  mutable mpz_t x,y; // zwei Hilfsvariablen
 public:
  CProvideHilfsvariablen() { mpz_init(x); mpz_init(y); }
  ~CProvideHilfsvariablen() { mpz_clear(x); mpz_clear(y); }
};

class CRelation : protected CProvideHilfsvariablen
/* CRelation verwaltet eine Relation der Form
       Delta^2 = Produkt von Faktoren (mod N)
   Ziel ist es, die Menge der Faktoren durch Verknpfung von Relationen
   zu leeren. Das Produkt wird dann 1.
   Delta pat sich den Vernderungen dynamisch an.
*/ 
{
 public:
  int relevanter_Faktor;
 private:
  CTinyFBsizetypeVector *Relation_sparse;
  myBitString *Relation_dense;
  mpz_t Delta; // dynamischer Delta-Wert
 public:
  inline CRelation() /* Konstruktor */
   : CProvideHilfsvariablen(), relevanter_Faktor(no_factor)
    {
      Relation_sparse = new CTinyFBsizetypeVector;
      Relation_dense = NULL;
      mpz_init_set_ui(Delta,1);
    }
 public:
  inline ~CRelation() /* Destruktor */
    {
      if (Relation_sparse) { delete Relation_sparse; Relation_sparse=NULL; }
      if (Relation_dense) { delete Relation_dense; Relation_dense=NULL; }
      mpz_clear(Delta);
      relevanter_Faktor=no_factor;
    }
  
  CRelation(const signed int Siebpos, const short int HitCount=0); /* Konstruktur */
  /* "Deltas" mit dem Wert Delta initialisieren,
     "Relationen" mit den Primfaktoren des reduzierten Quadrates fllen,
     deren Exponenten ungerade sind. */

 public:
  int groesster_Faktor_in_Relation() const
    /* (Index) des grten Faktors in der Relation */
    {
      if (Relation_sparse)
	if (Relation_sparse->empty()) return no_factor;
        else 
	  {
	    return Relation_sparse->last();
	  }
      else 
	{
	  return Relation_dense->last(1);
	  /*
	    int h = Relation_dense->last(1);
	    if (h>=0) return h;
	    else return no_factor;
	  */
	}
    }

  int zweitgroesster_Faktor_in_Relation() const
    /* (Index) des zweitgrten Faktors in der Relation */
    {
      if (Relation_sparse)
        if (Relation_sparse->size()>1)
          return (*Relation_sparse)[Relation_sparse->size()-2];
        else
          return no_factor;
      else // Relation ist dense
        {
          signed int vorletzter_Faktor=Relation_dense->last(1);
          if (vorletzter_Faktor>=0)
           return Relation_dense->prev(vorletzter_Faktor,1);
          else
           return no_factor;
        }
    }
  

  inline unsigned int SizeOfRelation() const
    {
      if (Relation_sparse) return Relation_sparse->size();
      else return Relation_dense->count(1);
    }

  inline void optisize(void)
    {
      // Speicherbedarf der Relation optimieren (=minimieren)
      // 1. Falls sinnvoll, eine implizite Konvertierung dense<->sparse,
      // 2. Speicherbedarf innerhalb der Darstellung minimieren

      int size=SizeOfRelation();
      int largest=groesster_Faktor_in_Relation();
      int quotient = size ? largest/size : 0;

      if (Relation_sparse)
       {
         if (quotient<16-2) convert_Relation_to_dense();
         else Relation_sparse->optisize();
       }
      else // (Relation_dense)
       {
         if (quotient>16+2) convert_Relation_to_sparse();
         else Relation_dense->optisize();
       }
    }
  
  void convert_Relation_to_dense(); /* mache "dense" aus "sparse" */
  void convert_Relation_to_sparse(); /* mache "sparse" aus "dense" */
  inline bool Relation_ist_leer() /* gibt zurck, ob die Relation leer ist */
    const
    {
      return (relevanter_Faktor==no_factor);
      // if (Relation_sparse) return Relation_sparse->empty();
      // else return Relation_dense->last(1)==-1; 
    }	
  void kombinieren(const CRelation& GL2);
  /* die Vereinigungsmengen mit GL2 bilden und daraus die
     Schnittmengen entfernen. (-> also: symmetrische Differenz bilden)
     gibt zurck, ob die Relationsmenge leer ist, d.h. eine Kombination
     quadratischer Reste mit dem gleichen Wert gefunden wurde. */

  // Routinen, die ein mehrfaches Kombinieren beschleunigen:
 protected:
  static unsigned int Exponentenfeld[MAX_Primbasis_size]; // (Startwert mu {0} sein!)
  static unsigned int multi_kombinieren_Zaehler; // (Startwert mu 0 sein!)
 public:
  void multi_kombinieren_init();
  void multi_kombinieren_main(const CRelation& GL2);
  void multi_kombinieren_exit();
  
  void Quadratische_Kongruenz_ausrechnen() const;
  void Kongruenzcheck() const; // test, ob Kongruenz noch in Ordnung 
 private:
  inline void swap(CRelation &GL2)
    { /* Inhalte der beiden Relationen (referenzsemantisch) tauschen */
      std::swap(Relation_sparse,GL2.Relation_sparse);
      std::swap(Relation_dense,GL2.Relation_dense);
      std::swap(relevanter_Faktor,GL2.relevanter_Faktor);
      mpz_swap(Delta,GL2.Delta);
    }
 public:
  inline const bool operator< (const CRelation &GL2) const
    {	
      return relevanter_Faktor < GL2.relevanter_Faktor;
    }
  streampos speichern(ostream &out, const CmpqsFactor factor, const short int HitCount=0) const;
  inline streampos speichern(ostream &out, const int i=1, const short int HitCount=0) const
   {
     CmpqsFactor DLP;
     DLP=i;
     return speichern(out,DLP,HitCount);
   };
  CmpqsFactor kombinieren(istream &in, const streampos pos);
  CmpqsFactor multi_kombinieren_main(istream &in, const streampos pos);
  CmpqsFactor kombinieren(istream &in);
  CmpqsFactor multi_kombinieren_main(istream &in);
  friend ostream& operator<< (ostream& ostr, const CRelation& GL);
  friend void Relation_einfuegen(CRelation *GL, const bool do_multi_kombinieren_init=true);
  friend void Special_Factoren_durch_Primfaktor_splitten_LAZY(const int Primfaktor);
  friend bool SpecialFactorRelation_vermerken(const CmpqsFactor &DLP, CRelation *GL, const short int HitCount=0);
  friend bool SpecialFactorRelation_LAZY_vermerken(const CmpqsFactor &DLP, const string &GL_String);
  friend void SpecialFactors_Zyklensuche();
  friend class Cprocess_clients;
  void Relation_optimieren(int bitpos=1);
};


ostream& operator<< (ostream& ostr, const CRelation& GL);
// Stream-Ausgabefunktion fr Relation

void Special_Factoren_durch_Primfaktor_splitten(const int Primfaktor);
// Specialfaktoren liegen auerhalb der Primzahlbasis.
// Mit ihnen wird nicht gesiebt. Sie sind auerdem zusammengesetzt.
// Wenn es gelingt, von einem Specialfaktor einen Primfaktor aus der dynamischen FB
// abzusplitten, dann bestehen gute Chancen, dadurch einen neuen dynamischen Faktor
// zu erhalten oder gar einen Treffer im Gleichungssystem, wenn der entstandene
// dynamische Faktor bereits in der dynamischen Faktorbasis bekannt ist.


#endif
