// polynomial arithmetic
// written by Thorsten Reinecke, Bernd Edler & Frank Heyder, 1999-12-10
// some changes: 1999-12-30
// quick review: 2003-07-02 by Thorsten Reinecke
// last change: 2003-11-11

// man kann noch keinen Blumentopf damit gewinnen, aber es scheint
// wenigstens zu laufen ;)

/*
  TODO: DFFT is still at early alpha stage...

   A(x)=a[0]+a[1]*x^1+a[2]*x^2+...+a[k]*x^m  mod(N)
   B(x)=b[0]+b[1]*x^1+b[2]*x^2+...+b[k]*x^n  mod(N)
   -> C(x)=A(x)*B(x)=c[0]+c[1]*x+...+c[m+n]*x^(m+n)   mod(N)

   Problem: Quadratwurzeln knnen wir modulo N nicht ziehen,
   jedenfalls nicht in ertrglicher Zeit... (da N im allgemeinen eine
   zusammgesetzte Zahl ist, deren Teiler uns nicht bekannt sind)
   ... also knnen wir die diskrete FFT nicht durchfhren!

   Lsung:
   Beobachtung, dass die einzelnen Koeffizienten des Polynoms A(x)*B(x)
   nicht grer werden knnen als (m+n)*N^2, wenn die einzelnen
   Koeffizienten von A(x) und B(x) im Bereich { 0 ...N-1 } liegen.
   (Beweis: Worst-Case-berschlagsrechnung bei naiver Polynommultiplikation)

   [ Nebenbei: Das ist auch der Grund dafr, warum sich
     mpz_mod innerhalb der Rekursion bei der 
     Karatsuba-Polynommultiplikation nicht lohnt... ]

   Wir knnen uns also eine Primzahl P > (m+n)*N^2 suchen,
   fr die wir Quadratwurzeln ziehen knnen!
   Wir fhren die DFT dann modulo P durch.
   Das Ergebnis sollte immer mit dem Ergebnis ohne Modulorechnung
   bereinstimmen.
   -> Also  C(x) mod P = C(x) mod (m+n)*N^2
   -> (C(x) mod P) mod N ist dann unser gesuchtes Polynom!

   So viel zum Prinzipiellen...

   Leider wird die skalare Multiplikation dann unheimlich teuer
   und die Koeffizienten ziemlich gro, so dass man diese
   mod P reduzieren mu (allein schon, um Speicher zu sparen).
   Die Laufzeitgewinne schrumpfen also bei greren Zahlen
   schnell wieder ein...
    
   Alternativ knnen wir aber statt eines groen P viele kleine P's
   verwenden und die Teilergebnisse dann mit dem chinesichen Restsatz 
   zum Gesamtergebnis zusammensetzen.
   Da die Teilmultiplikationen nach dem SIMD-Prinzip (Single Instruction
   Multiple Data) durchgefhrt werden knnen, eignet sich das gut fr
   Vektorrechner und drfte auch gut fr eine sptere MMX/3dnow-Optimierung
   geeignet sein.

   32-bittige unsigned ints sind dafr allerdings nicht unbedingt zu empfehlen,
   da es im Intervall [2^31,2^32] nur 199 Primzahlen gibt mit (p-1)=2^20*rest,
   wie nachfolgende Prozedur zeigt:

     c:=0;
     for i from 2^31+1 to 2^32 by 2^20 do 
       if isprime(i) then c:=c+1; print(c,i); fi;
     od;

   Damit lieen sich dann grob berschlagen Dezimalzahlen der Stelligkeit
   lg(((2^31)^(199/2))/(2^20))=(31*99.5-20)*lg(2)=3064.5*0.30103=922.5
   beackern, da wir die 199 Primzahlen ber den chinesischen Restsatz
   kombinieren und die Gre der Polynome auf maximal 2^20 Koeffizienten
   begrenzen mssen. (Koeffizientengre von etwa 2^20*N^2 mu noch korrekt
   rekonstruierbar sein!)
   Hier heit es also auf 64bit umzusteigen oder in gleich Fliekomma
   zu rechnen.
*/


/*! @file
 * @brief
 * contains implementation of polynomial arithmetic for multiple precision numbers
 *
 * references and related literature:
 *
 *  [1] Alfred V. Aho, John E. Hopcroft, Jeffrey D. Ullman: 
 *      "The Design and Analysis of Computer Algorithms",
 *      Addison-Wesley, 1974
 *   
 *  [2] Robert Sedgewick: "Algorithmen",
 *      Addison-Wesley, 1991
 *
 *  [3] Peter L. Montgomery: 
 *      "An FFT Extension of the Elliptic Curve Method of Factorization",
 *      (dissertation, University of California), 1992,
 *      ftp.cwi.nl/pub/pmontgom/ucladissertation.psl.gz
 */
       
   
///////////////////////////////////////////////////////////////////////////
// Implementierung:
///////////////////////////////////////////////////////////////////////////

#ifndef __POLYNOMIAL__
#define __POLYNOMIAL__

#include <cstdlib>
#include <iostream>
#include <gmp.h>
#include <new>
#include "utils.H"


//! contains polynomial arithmetic concepts for mpz_t
namespace polynomial
{

//#define DEBUG    /* for explicite DEBUGGING-sessions */
//#define VERBOSE  /* be more verbose */

// this define should be triggered by Makefile
// #define USE_DFT /* enable use of early-alpha discrete fourier transform */

using std::cout;
using std::cerr;
using std::endl;
using std::cin;

typedef mpz_t* TPolynom;
typedef const mpz_t* TconstPolynom;
/* Ein Polynom ist ein Feld aus mpz_t-Koeffizienten:
   P(x)=a0*x^0+a1*x^1+...+a[k-1]*x^[k-1],
   gespeichert als (a0,a1,a2,...,a[k-1]).
*/

/* Hinweise zur Beachtung bei Funktionsaufrufen:

 TPolynom bezeichnet einen Zeiger auf ein Koeffizientenfeld
 (welches ber seine Koeffizienten das Polynom definiert).

 Wie in C++ blich, sind daher folgende Aufrufkonventionen
 zu beachten bzw. zu unterscheiden:

 Parameter		Restriktionen/Eigenschaften
 -------------------------------------------------------------
 mpz_t* p		 Zeiger vernderbar, Daten vernderbar
 const mpz_t* p		 Zeiger vernderbar, Daten konstant
 mpz_t* const p		 Zeiger konstant,    Daten vernderbar
 const mpz_t* const p    Zeiger konstant,    Daten konstant

 Nun sind aber "const TPolynom P" und "TPolynom const P"
 (leider) quivalent und bezeichnen beide
 einen konstanten Zeiger auf ein vernderbares Polynom!

 Deshalb habe ich zustzlich einen Datentyp fr
 konstante Polynome definiert: TconstPolynom
 Damit gilt:

 Parameter              Restriktionen/Eigenschaften
 ----------------------------------------------------------------
 TPolynom P              P vernderbar, Koeffizienten vernderbar
 const TPolynom P        P konstant,    Koeffizienten vernderbar
 TPolynom const P	 P konstant,    Koeffizienten vernderbar
 TconstPolynom P         P vernderbar, Koeffizienten konstant
 const TconstPolynom P	 P konstant,    Koeffizienten konstant
 TconstPolynom const P   P konstant,    Koeffizienten konstant

*/


// a tiny helper class for temporary arrays mpz_t[] in C++
// self initializing and self destructing
// (Einziger Sinn und Zweck: Ein komfortabler Ersatz fr die nach ISO-Standard
//  nicht untersttzten variablen Arrays)
class TTempPolynom
{
private:
  mpz_t* const data;
  const int _capacity;
public:
  explicit TTempPolynom(const int n) : data(new mpz_t[n]), _capacity(n)
   {
     for (int i=0; i<n; ++i) mpz_init(data[i]);
   }
  ~TTempPolynom()
   {
     for (int i=_capacity-1; i>=0; --i) mpz_clear(data[i]);
     delete [] data;
   }
  const int capacity() const { return _capacity; }

  //const mpz_t& operator[] (const unsigned int i) const { return data[i]; }
  //mpz_t& operator[] (const unsigned int i) { return data[i]; }

  // dangerous, but helpful:
  // remember: TPolynom will not live longer than TempPolynom!
  operator const TPolynom() const { return data; }
};



// this one is only for debugging
inline void evalp(const TconstPolynom P, const int k, const int pos)
{
  cout << "evalp: ";
  mpz_out_str(stdout,10,P[pos]);
  cout << endl;
}

void print (const TconstPolynom P, int k)
{
  //cout << "Polynomausgabe fr k=" << k << " :" << flush;
  while (--k>0) 
   {
     mpz_out_str(stdout,10,P[k]);
     cout << "*x^" << k << " + ";
   }
  if (k==0) mpz_out_str(stdout,10,P[0]);
  else cout << "0";
}

void eval(mpz_t res, const TconstPolynom P, const int k, const mpz_t x, const mpz_t m)
// berechnet mit dem Hornerschema res = P(x) mod m
{
  mpz_t y; // lokale Variable notwendig, falls &res=&x oder &res=&m
  mpz_init_set(y,P[k-1]);
  for (int i=k-2; i>=0; --i)
   {
     mpz_mul(y,y,x); mpz_add(y,y,P[i]); mpz_mod(y,y,m);
   }
  mpz_set(res,y);
  mpz_clear(y);
}


// mal ein kleines Template zwischendurch...
template<typename T> inline T ld(T n)
{
  // 2er-Logarithmus (truncated)
  T r = 0;
  while(n>>=1) ++r;
  return r;
}

} // namespace polynomial
// leave the namespace to include optional polynomial implementations
// possibly including header files in other namespaces...


#ifdef USE_DFT
 #include "dft.cc" // fast discrete fourier transformation
#endif


// enter the namespace again
namespace polynomial
{

int classic_mul(const TPolynom Pr, const int kr,
                const TconstPolynom P1, const int k1,
                const TconstPolynom P2, const int k2)
// Pres = P1*P2, &Pres mu von &P1,&P2 verschieden sein
// liefert Grad des resultierenden Polynoms zurck
// Komplexitt: O(k1*k2)
// (Klassische Methode, alternativ zu Karatsuba implementiert fr kleine Polynome)
{
  if (kr<k1+k2-1) cerr << "classic_mul: Nicht gengend Speicher fr Resultatpolynom" << endl;
  for (int i=0; i<kr; ++i) mpz_set_ui(Pr[i],0);

  for (int i=0; i<k1; ++i)
     for (int j=0; j<k2; ++j)
        mpz_addmul(Pr[i+j],P1[i],P2[j]);

  return k1+k2-1;
}

int classic_mul(const TPolynom Pr, const int kr,
                const TconstPolynom P1, const int k1,
                const TconstPolynom P2, const int k2, const mpz_t m)
// Pres = P1*P2, &Pres mu von &P1,&P2 verschieden sein
// liefert Grad des resultierenden Polynoms zurck
// Komplexitt: O(k1*k2)
// (Klassische Methode, alternativ zu Karatsuba implementiert fr kleine Polynome)
{
  if (kr<k1+k2-1) cerr << "classic_mul: Nicht gengend Speicher fr Resultatpolynom" << endl;
  for (int i=0; i<kr; ++i) mpz_set_ui(Pr[i],0);

  for (int i=0; i<k1; ++i)
     for (int j=0; j<k2; ++j)
        mpz_addmul(Pr[i+j],P1[i],P2[j]);

  // mod besser auslagern:
  for (int i=k1+k2-2; i>=0; --i)
    mpz_mod(Pr[i],Pr[i],m);
  return k1+k2-1;
}


int square_rek(const TPolynom R, const int kR,
               const TconstPolynom P, const int k,
               const TPolynom temp)
// R = P^2, &R mu von &P verschieden sein
// liefert Grad des resultierenden Polynoms zurck
// das Polynom temp mu gengend temporren Speicher zur Verfgung stellen!
// Komplexitt: O(k^1.59) -> Karatsuba
{
#if 1
  if (k==2) // handoptimierter Fall...
   {
     mpz_mul(R[0],P[0],P[0]);
     mpz_mul(R[1],P[0],P[1]); mpz_mul_2exp(R[1],R[1],1);
     mpz_mul(R[2],P[1],P[1]);
     return 3;
   }
#endif
  if (k==1) // eigentlicher Rekursionsanker
   {
     mpz_mul(R[0],P[0],P[0]);
     return 1;
   }


  const int Mitte=k/2; // Mitte trennt LO/HI fr divide & conquer

  // Pr=P^2
  // P gesplittet in PL=P[0]..P[Mitte-1], PH=P[Mitte]..P[k-1]

  const TconstPolynom PL=&P[0]; const int kPL=Mitte;
  const TconstPolynom PH=&P[kPL]; const int kPH=k-Mitte;

  const int kH1=MAX(kPL,kPH);
  const TPolynom H1=&R[0];
  for (int i=0; i<kPL; ++i) mpz_add(H1[i],PH[i],PL[i]);
  for (int i=kPL; i<kH1; ++i) mpz_set(H1[i], PH[i]);

  int kM=2*kH1-1;
  const TPolynom M=&R[Mitte]; // mittleres Polynom
  const TPolynom temp2 = &temp[2*kH1-1];
  kM=square_rek(temp,kM,H1,kH1,temp2);

  for (int i=kR-1; i>=0; --i) mpz_set_ui(R[i],0); // Resultatpolynom: Koeffizienten mit Nullen fllen.

  int kL=kPL+kPL-1;
  const TPolynom L=R;
  kL=square_rek(L,kL,PL,kPL,temp2);

  int kH=kPH+kPH-1;
  const TPolynom H=&R[Mitte*2];
  kH=square_rek(H,kH,PH,kPH,temp2);

  for (int i=0; i<kL; ++i) { mpz_sub(temp[i], temp[i], L[i]); }
  for (int i=0; i<kH; ++i) { mpz_sub(temp[i], temp[i], H[i]); }
  for (int i=0; i<kM; ++i) { mpz_add(M[i], M[i], temp[i]); }

  return 2*k-1;
}


int mul_rek(const TPolynom R, const int kR,
            const TconstPolynom P1, const int k1, const TconstPolynom P2, const int k2,
            const TPolynom temp)
// Pres = P1*P2, &R mu von &P1,&P2 verschieden sein
// liefert Grad des resultierenden Polynoms zurck
// das Polynom temp mu gengend temporren Speicher zur Verfgung stellen!
// Komplexitt: O(max(k1,k2)^1.59) -> Karatsuba
{
  if (k1==2)
   {
    if (k2==2) // handoptimierter Fall
     {
       // (a0+a1*x)(b0+b1*x)=a0*b0+((a0+a1)(b0+b1)-a0*b0-a1*b1)*x+(a1*b1)*x^2
       mpz_add(temp[0],P1[0],P1[1]); mpz_add(temp[1],P2[0],P2[1]);
       mpz_mul(R[0],P1[0],P2[0]); mpz_mul(R[2],P1[1],P2[1]);
       mpz_mul(R[1],temp[0],temp[1]);
       mpz_sub(R[1],R[1],R[0]); mpz_sub(R[1],R[1],R[2]);
       return 3;
     }
    if (k2==3)
     {
       // toom-cook-like (without recursion)
       // result should be: P1*P2=R=w0+w1*x+w2*x^2+w3*x^3

       mpz_add(temp[0],P1[0],P1[1]); mpz_add(temp[1],P2[0],P2[2]);
       mpz_add(temp[2],temp[1],P2[1]); 
       mpz_mul(R[1],temp[0],temp[2]); // R[1]=w0+w1+w2+w3
       mpz_sub(temp[1],temp[1],P2[1]); mpz_sub(temp[0],P1[0],P1[1]);
       mpz_mul(R[2],temp[0],temp[1]); // R[2]=w0-w1+w2-w3
       mpz_mul(R[0],P1[0],P2[0]); // R[0]=w0
       mpz_mul(R[3],P1[1],P2[2]); // R[3]=w3

       mpz_add(R[2],R[2],R[1]); // R[2]=2*(w0+w2);
       mpz_tdiv_q_2exp(R[2],R[2],1); // R[2]=w0+w2
       mpz_sub(R[1],R[1],R[2]); // R[1]=w1+w3
       mpz_sub(R[2],R[2],R[0]); // R[2]=w2
       mpz_sub(R[1],R[1],R[3]); // R[1]=w1

       return 4;
     }
#ifdef VERBOSE
    cout << __FUNCTION__ << ": kein handoptimierter Code fr " << k1 << "," << k2 << " vorhanden." << endl;
#endif
   } // k1==2

  if (k1==1 || k2==1)
   {
    if (k1==1 && k2==1)
     {
       mpz_mul(R[0],P1[0],P2[0]);
       return 1;
     }
    if (k1==1)
     {
       for (int i=0; i<k2; ++i) mpz_mul(R[i],P1[0],P2[i]);
       return k2;
     }
    if (k2==1)
     {
       for (int i=0; i<k1; ++i) mpz_mul(R[i],P2[0],P1[i]);
       return k1;
     }
   } // k1==1 || k2==1


#ifdef VERBOSE
  if (k1<4 || k2<4)
   {
     if (k1==3 && k2==3) 
      cout << __FUNCTION__ << " : 3x3!" << endl;
     else
      cout << __FUNCTION__  << ": " << k1 << ", " << k2 << endl;
   }
#endif

  const int Mitte=MIN(k1,k2)/2; // Mitte trennt LO/HI fr divide & conquer
  // wichtig: Mitte mu kleiner sein als minimum!

  // Pr=P1*P2
  // P1 gesplittet in P1L=P1[0]..P1[Mitte-1], P1H=P1[Mitte]..P1[k1-1]
  // P2 gesplittet in P2L=P2[0]..P2[Mitte-1], P2H=P2[Mitte]..P2[k2-1]

  //if (k1!=k2) cout << "k1=" << k1 << " , " << "k2=" << k2 << endl;

  const TconstPolynom P1L=&P1[0]; const int kP1L=Mitte;
  const TconstPolynom P1H=&P1[kP1L]; const int kP1H=k1-Mitte;
  const TconstPolynom P2L=&P2[0]; const int kP2L=Mitte;
  const TconstPolynom P2H=&P2[kP2L]; const int kP2H=k2-Mitte;

  const int kH1=MAX(kP1L,kP1H), kH2=MAX(kP2L,kP2H);
  const TPolynom H1=&R[0], H2=&R[kH1];
  for (int i=0; i<kP1L; ++i) mpz_add(H1[i],P1H[i],P1L[i]);
  for (int i=kP1L; i<kH1; ++i) mpz_set(H1[i], P1H[i]);
  for (int i=0; i<kP2L; ++i) mpz_add(H2[i],P2H[i],P2L[i]);
  for (int i=kP2L; i<kH2; ++i) mpz_set(H2[i], P2H[i]);

  int kM=kH1+kH2-1;
  const TPolynom M=&R[Mitte]; // mittleres Polynom
  const TPolynom temp2 = &temp[kH1+kH2-1];
  kM=mul_rek(temp,kM,H1,kH1,H2,kH2,temp2);

  for (int i=kR-1; i>=0; --i) mpz_set_ui(R[i],0); // Resultatpolynom: Koeffizienten mit Nullen fllen.

  int kL=kP1L+kP2L-1;
  const TPolynom L=R;
  kL=mul_rek(L,kL,P1L,kP1L,P2L,kP2L,temp2);

  int kH=kP1H+kP2H-1;
  const TPolynom H=&R[Mitte*2];
  kH=mul_rek(H,kH,P1H,kP1H,P2H,kP2H,temp2);

  for (int i=0; i<kL; ++i) { mpz_sub(temp[i], temp[i], L[i]); }
  for (int i=0; i<kH; ++i) { mpz_sub(temp[i], temp[i], H[i]); }
  for (int i=0; i<kM; ++i) { mpz_add(M[i], M[i], temp[i]); }

  return k1+k2-1;
}


// forward declarations...
int monic_mul(const TPolynom R, const int kR,
              const TconstPolynom P1, int k1,
              const TconstPolynom P2, int k2, const mpz_t m);
int monic_square(const TPolynom R, const int kR,
                 const TconstPolynom P, int k, const mpz_t m);


int square(const TPolynom R, const int kR, const TconstPolynom P, const int k, const mpz_t m)
// R = P^2, &R mu von &P verschieden sein
// liefert Grad des resultierenden Polynoms zurck
// Komplexitt: O(k^1.59) -> Karatsuba
{
  if (kR<2*k-1)
   {
     MARK;
     cerr << "square: Nicht gengend Speicher fr Resultatpolynom" << endl;
     cerr << "-> k=" << k << ", kR=" << kR << endl;
   }

  //if (((2*k-1)&1) && mpz_cmp_ui(P[k-1],1)==0) return monic_square(R,kR,P,k,m);
  //if (mpz_cmp_ui(P[k-1],1)==0) { MARK; return monic_square(R,kR,P,k,m); }

#ifdef USE_DFT
  if (dft_square_is_recommended(k))
   {
     return get_dft(2*k-1,m)->squaremod(R,kR,P,k);
   }
#endif

  const int estimated_operand_size = 2*mpz_sizeinbase(m,2)+5;
  const int tempsize = 2*2*k;
  const TPolynom temp = new mpz_t[tempsize];
  for (int i=tempsize-1; i>=0; --i) mpz_init2(temp[i],estimated_operand_size); // temporren Speicher allokieren
 
  const int ret=square_rek(R,kR,P,k,temp);

  for (int i=0; i<ret; ++i) 
    mpz_mod(R[i],R[i],m); // Resultatpolynom normieren
 
  for (int i=tempsize-1; i>=0; --i) mpz_clear(temp[i]); // temporren Speicher lschen
  delete [] temp;
  return ret;
}


int square(const TPolynom R, const int kR, const TconstPolynom P, const int k)
// R = P^2, &R mu von &P verschieden sein
// liefert Grad des resultierenden Polynoms zurck
// Komplexitt: O(k^1.59) -> Karatsuba
{
  if (kR<2*k-1)
   {
     MARK;
     cerr << "karasquare: Nicht gengend Speicher fr Resultatpolynom" << endl;
     cerr << "-> k=" << k << ", kR=" << kR << endl;
   }

#if 0 && defined(USE_DFT)
  // we need a valid "modulo m" value to do dft;
  // we could calculate m, but I think this takes too much time...
  // calculate m would be something like this:
  //  mpz_t m,h; mpz_init(m); mpz_init(h);
  //  for (int i=0; i<k; ++i)
  //    {  mpz_abs(h,P[i]); if (mpz_cmp(P[i],m)>0) mpz_swap(m,h); }
  //  ...
  //  mpz_clear(h); mpz_clear(m);
  if (dft_square_is_recommended(k))
   {
     if (mpz_cmp_ui(P[k-1],1)==0) { MARK; cout << k << " -> " << 2*k-1 << endl; }
     return get_dft(2*k-1,m)->square(R,kR,P,k);
   }
#endif

  const int tempsize = 2*2*k;
  const TPolynom temp = new mpz_t[tempsize];
  for (int i=tempsize-1; i>=0; --i) mpz_init(temp[i]); // temporren Speicher allokieren
 
  const int ret=square_rek(R,kR,P,k,temp);

  for (int i=tempsize-1; i>=0; --i) mpz_clear(temp[i]); // temporren Speicher lschen
  delete [] temp;
  return ret;
}


int mul(const TPolynom R, const int kR,
        TconstPolynom P1, int k1,
        TconstPolynom P2, int k2, const mpz_t m)
// Pres = P1*P2, &R mu von &P1,&P2 verschieden sein
// liefert Grad des resultierenden Polynoms zurck
// Komplexitt: O(max(k1,k2)^1.59) -> Karatsuba
// bzw. Komplexitt: O((k1+k2)*ld(k1+k2)) -> bei (optimaler) dft
{
  //cout << "mul_mod " << k1 << "x" << k2 << endl;

  if (kR<k1+k2-1)
   {
     cerr << "karamul: Nicht gengend Speicher fr Resultatpolynom" << endl;
     cerr << "-> k1=" << k1 << ", k2=" << k2 << ", kR=" << kR << endl;
   }

#if 1
  if ( ((k1+k2-1)&1) && mpz_cmp_ui(P1[k1-1],1)==0 && mpz_cmp_ui(P2[k2-1],1)==0 )
   {
     // we call monic mul for better performance ;-)
     //MARK; cout << "monic MUL!" << endl;
     return monic_mul(R,kR,P1,k1,P2,k2,m);
   }
#endif

#ifdef USE_DFT
  if (dft_mul_is_recommended(k1,k2))
   {
     return get_dft(k1+k2-1,m)->mulmod(R,kR,P1,k1,P2,k2);
   }
#endif

  if (k2<k1) { std::swap(k1,k2); std::swap(P1,P2); }
  // now k1<=k2

  if (2*k2>3*k1)
   {
#if defined(VERBOSE)
     MARK;
     cout << "using classic mul for " << k1 << ", " << k2 << endl;
#endif
     return classic_mul(R,kR,P1,k1,P2,k2,m);
   }
  //cout << "Hint: " << k1 << ", " << k2 << endl;


  const int estimated_operand_size = 2*mpz_sizeinbase(m,2)+5;
  const int tempsize = 4*k2;
  const TPolynom temp = new mpz_t[tempsize];
  for (int i=tempsize-1; i>=0; --i) mpz_init2(temp[i],estimated_operand_size); // temporren Speicher allokieren
 
  const int ret = mul_rek(R,kR,P1,k1,P2,k2,temp);
  for (int i=0; i<ret; ++i)
    mpz_mod(R[i],R[i],m); // Resultatpolynom normieren
 
  for (int i=tempsize-1; i>=0; --i) mpz_clear(temp[i]); // temporren Speicher lschen
  delete [] temp;
  return ret;
}

int mul(const TPolynom R, const int kR,
        TconstPolynom P1, int k1,
        TconstPolynom P2, int k2)
// Pres = P1*P2, &R mu von &P1,&P2 verschieden sein
// liefert Grad des resultierenden Polynoms zurck
// Komplexitt: O(max(k1,k2)^1.59) -> Karatsuba
{
  //cout << "mul " << k1 << "x" << k2 << endl;

  if (kR<k1+k2-1)
   {
     cerr << "karamul: Nicht gengend Speicher fr Resultatpolynom" << endl;
     cerr << "-> k1=" << k1 << ", k2=" << k2 << ", kR=" << kR << endl;
   }

  if (k2<k1) { std::swap(k1,k2); std::swap(P1,P2); }
  // now k1<=k2

  if (2*k2>3*k1)
   {
#if defined(VERBOSE)
     MARK;
     cout << "using classic mul for " << k1 << ", " << k2 << endl;
#endif
     return classic_mul(R,kR,P1,k1,P2,k2);
   }
  //cout << "Hint: " << k1 << ", " << k2 << endl;

  const int tempsize = 4*k2;
  const TPolynom temp = new mpz_t[tempsize];
  for (int i=tempsize-1; i>=0; --i) mpz_init(temp[i]); // temporren Speicher allokieren
 
  const int ret = mul_rek(R,kR,P1,k1,P2,k2,temp);
 
  for (int i=tempsize-1; i>=0; --i) mpz_clear(temp[i]); // temporren Speicher lschen
  delete [] temp;
  return ret;
}


int monic_mul(const TPolynom R, const int kR,
              const TconstPolynom P1, int k1,
              const TconstPolynom P2, int k2, const mpz_t m)
// multipliziert zwei Polynome, deren hchste Koeefizienten P1[k1-1]=P2[k2-1]=1 sind.
// monische Polynome, deren fhrende Koeefizienten Nullen sind, sind ebenso erlaubt.
// sollte etwas schneller sein als normales mul...
{
  if (kR<k1+k2-1)
   {
     cerr << "monic mul: Nicht gengend Speicher fr Resultatpolynom" << endl;
     cerr << "-> k1=" << k1 << ", k2=" << k2 << ", kR=" << kR << endl;
   }
  
  k1--; while (mpz_cmp_ui(P1[k1],0)==0) --k1;
  k2--; while (mpz_cmp_ui(P2[k2],0)==0) --k2;
  
  if (mpz_cmp_ui(P1[k1],1)!=0 || mpz_cmp_ui(P2[k2],1)!=0)
   {
     cerr << __FUNCTION__ << ": Polynome nicht monisch!" << endl;
     cout << "P1="; print(P1,k1+1); cout << endl;
     cout << "P2="; print(P2,k2+1); cout << endl;
     exit(1);
   }

#if 1
  // Spezialfall (geht besonders schnell & einfach)
  if (k1==1 && k2==1)
   {
     mpz_mul(R[0],P1[0],P2[0]); mpz_mod(R[0],R[0],m);
     mpz_add(R[1],P1[0],P2[0]); mpz_mod(R[1],R[1],m);
     mpz_set_ui(R[2],1); 
     return 3;
   }
#endif


  /*
    (x^k1 +PR1)*(x^k2 +PR2) = X^(k1+k2) + x^(k1) *PR2 + x^(k2) *PR1 + PR1*PR2
    Vorgehensweise:
     1. PR1*PR2 ausrechnen
  */ 

#ifdef USE_DFT
  int ret;
  if (dft_mul_is_recommended(k1,k2))
   {
     if (mpz_cmp_ui(P1[k1-1],1)==0 && mpz_cmp_ui(P2[k2-1],1)==0) { MARK; cout << k1+k2-1 << endl; }
     ret=get_dft(k1+k2-1,m)->mul(R,kR,P1,k1,P2,k2);
   }
  else
   ret=mul(R,kR,P1,k1,P2,k2);
#else
  int ret=mul(R,kR,P1,k1,P2,k2); // um ein Grad erniedrigte Polynome multiplizieren
#endif

  /*
     2. Resultat um x^(k1+k2) ergnzen (und ggf. Nullen einfgen)
  */
  while (ret<k1+k2) mpz_set_ui(R[ret++],0);
  mpz_set_ui(R[ret++],1);

  /*
     3. x^(k1) *PR2 hinzuaddieren
     4. x^(k2) *PR1 hinzuaddieren
  */
  for (int i=0; i<k1; ++i) mpz_add(R[i+k2],R[i+k2],P1[i]);
  for (int i=0; i<k2; ++i) mpz_add(R[i+k1],R[i+k1],P2[i]);
  for (int i=0; i<ret-1; ++i) mpz_mod(R[i],R[i],m);
  return ret;
}

int monic_square(const TPolynom R, const int kR,
                 const TconstPolynom P, int k, const mpz_t m)
// multipliziert zwei Polynome, deren hchste Koeefizienten P1[k1-1]=P2[k2-1]=1 sind.
// monische Polynome, deren fhrende Koeefizienten Nullen sind, sind ebenso erlaubt.
// sollte etwas schneller sein als normales mul...
{
  if (kR<2*k-1)
   {
     cerr << "monic square: Nicht gengend Speicher fr Resultatpolynom" << endl;
     cerr << "-> k=" << k << ", kR=" << kR << endl;
   }
  
  k--; while (k && mpz_cmp_ui(P[k],0)==0) --k;
  
  if (mpz_cmp_ui(P[k],1)!=0)
   {
     cerr << __FUNCTION__ << ": Polynom nicht monisch!" << endl;
     cout << "P="; print(P,k+1); cout << endl;
     exit(1);
   }

  // besonderer Spezialfall: P=x^0
  if (k==0) 
   {
     mpz_set_ui(R[0],1); return 1;
   }

#if 1
  // Spezialfall (geht besonders schnell & einfach)
  if (k==1)
   {
     mpz_mul(R[0],P[0],P[0]); mpz_mod(R[0],R[0],m);
     mpz_add(R[1],P[0],P[0]); mpz_mod(R[1],R[1],m);
     mpz_set_ui(R[2],1); 
     return 3;
   }
#endif


  /*
    (x^k1 +PR1)*(x^k2 +PR2) = X^(k1+k2) + x^(k1) *PR2 + x^(k2) *PR1 + PR1*PR2
    Vorgehensweise:
     1. PR1*PR2 ausrechnen
  */ 

#ifdef USE_DFT
  int ret;
  if (dft_square_is_recommended(k))
   {
     ret=get_dft(2*k-1,m)->square(R,kR,P,k);
   }
  else
   {
     ret=square(R,kR,P,k);
   }
#else
  int ret=square(R,kR,P,k); // um ein Grad erniedrigtes Polynom quadrieren
#endif
  /*
     2. Resultat um x^(k1+k2) ergnzen (und ggf. Nullen einfgen)
  */
  while (ret<2*k) mpz_set_ui(R[ret++],0);
  mpz_set_ui(R[ret++],1);

  /*
     3. x^(k1) *PR2 hinzuaddieren
     4. x^(k2) *PR1 hinzuaddieren
  */
  for (int i=0; i<k; ++i) mpz_addmul_ui(R[i+k],P[i],2);
  for (int i=0; i<ret-1; ++i) mpz_mod(R[i],R[i],m);
  return ret;
}


void reciprocal2p1(TPolynom R, int &kR, const TconstPolynom f, const int np1, const mpz_t m)
{
  const int n=np1-1;
  const TPolynom R1 = &R[1];
  TTempPolynom R2(2*np1), H(2*np1);
  mpz_t e, fni, x;
  mpz_init(e); mpz_init(fni); mpz_init(x);

  mpz_invert(fni,f[n],m); mpz_mod(fni,fni,m);
  kR=1; mpz_set(R1[0],fni);
  mpz_neg(e,f[n-1]); mpz_mul(e,e,fni); mpz_mod(e,e,m);

  for (int k=2; k<=n; k*=2)
   {
     const int kR2=square(R2,2*np1,R1,kR,m); // R2=R1^2
//     if (kR2<k) { MARK; cerr << kR2 << "," << k << endl; }
     mul(H,2*np1,R2,kR2,&f[n-(k-1)],k,m); // H=R1^2 * f[n-(k-1)],k
     for (int i=0; i<k/2; ++i) 
      {
        mpz_mul_2exp(R1[k/2+i],R1[i],1);
        mpz_set_ui(R1[i],0);
      }
     for (int i=0; i<k; ++i) mpz_sub(R1[i],R1[i],H[i+k-2]);

     mpz_mul(e,e,e); mpz_mod(e,e,m);
     if (k==2)
      mpz_set_ui(x,0);
     else
      {
        mpz_mul(x,H[k-3],f[n]); mpz_mod(x,x,m);
      }
     mpz_sub(e,e,x);
     mpz_mul(x,f[n-k],fni); mpz_mod(x,x,m);
     mpz_sub(e,e,x); mpz_mod(e,e,m);
     kR=k;
   }
  
  // man beachte &R1 = &R[1] !
  mpz_mul(x,e,fni); mpz_mod(R[0],x,m);
  kR++;

  mpz_clear(e); mpz_clear(fni); mpz_clear(x);
}


void reciprocal2(TPolynom R, int &kR, const TconstPolynom P, const int k, const mpz_t m)
{
  // berechnet das reziproke Polynom von P,
  // k mu eine Zweierpotenz sein!
  // R mu gengend Speicher bereitstellen!
  // P darf keine fhrenden Nullen besitzen!
  if (k==1)
   {
     if (mpz_invert(R[0],P[0],m)==0)
      {
        cerr << __FUNCTION__ << ": inverse does not exist!" << endl;
        cout << "P="; print(P,k); cout << endl;
        char ch; cin >> ch;
      }
     kR=1;
   }
  else
   {
     // Polynom Q bereitstellen
     const int mem_kQ=3*(k>>1); TTempPolynom Q(mem_kQ);
     int kQ=mem_kQ;
     reciprocal2(Q,kQ,&P[k>>1],k>>1,m); // und berechnen

     // Quadrat von Q berechnen     
     const int mem_kQQ=2*kQ; TTempPolynom QQ(mem_kQQ);
     int kQQ=square(QQ,mem_kQQ,Q,kQ,m);

     // B=QQ(x)*P(x)
     const int mem_kB=kQQ+k; TTempPolynom B(mem_kB);
     int kB=mul(B,mem_kB,QQ,kQQ,P,k,m);
          
     // Resultat berechnen
     for (int i=0; i<kR; ++i) mpz_set_ui(R[i],0); // Resultat mit Nullen fllen
     
     const int d = k>>1;
     for (int i=0; i<kQ; ++i) mpz_mul_ui(R[d+i],Q[i],2);
     for (int i=k-2; i<kB; ++i) mpz_sub(R[i-(k-2)],R[i-(k-2)],B[i]);
     kR = d+kQ >= kB-(k-2) ? d+kQ : kB-(k-2);
     kR--;
     while (kR>0 && (mpz_cmp_ui(R[kR],0)==0)) kR--; // normieren
     kR++;
     for (int i=0; i<kR; ++i) mpz_mod(R[i],R[i],m);
     //cout << "is "; print(R,kR); cout << endl;
   }
}

void reciprocal(TPolynom R, int &kR, const TconstPolynom P, const int k, const mpz_t m, const unsigned int scale=0)
{
  // berechnet das reziproke Polynom von P,
  // R mu gengend Speicher bereitstellen!
  // das bergebene Polynom wird zuvor mit x^scale multipliziert,
  // also die Koeffizienten um scale Stellen verschoben

  int z=1, h=k+scale;
  while (z<h) z<<=1;
  
  int d=z-h;
  if (d==0 && scale==0) 
   {
     // Zweierpotenz
#if defined(VERBOSE)
     cout << "calling recip with power of 2" << endl;
#endif
     reciprocal2(R,kR,P,k,m);
   }
  if (k==d+2 && scale==0)
   {
     // Zweierpotenz plus 1
#if defined(VERBOSE)
     cout << "calling recip with 1+power of 2" << endl;
#endif
     reciprocal2p1(R,kR,P,k,m);
   }
  else
   {
     // keine Zweierpotenz (oder scale!=0)
     d+=scale;
#if defined(VERBOSE)
     if (scale) cout << "Polynom wird skaliert!" << endl;
     else  cout << "keine Zweierpotenz " << k << ", " << z << ", " << d << endl;
#endif
     // Zwischenspeicher allokieren & reciprocal mit Displacement aufrufen
     const int kH=z; TTempPolynom H(kH);
     for (int i=0; i<k; ++i) mpz_set(H[i+d],P[i]);
     int kR2=2*kH; TTempPolynom R2(kR2);
#ifdef VERBOSE
     cout << __FUNCTION__ << ": calling recip" << endl;
#endif
     reciprocal2(R2,kR2,H,kH,m);
#ifdef VERBOSE
     cout << __FUNCTION__ << ": ...back (of calling recip)" << endl;
#endif
#ifdef DEBUG
     cout << "->"; print(R2,kR2); cout << endl;
#endif
     d-=scale;
     if (kR<kR2-d) cerr << __FUNCTION__ << ": Speichermangel im Zielpolynom!" << endl;
     kR=kR2-d;
     for (int i=0; i<kR; ++i) mpz_set_ui(R[i],0);
     for (int i=d; i<kR2; ++i) mpz_set(R[i-d],R2[i]);
   }
}


void olddiv(TPolynom Q, int &kQ, TPolynom R, int &kR,
         const TconstPolynom P1, int k1,
         const TconstPolynom P2, int k2, const mpz_t m)
// Polynomdivision, O(n^2)
// liefert P1/P2 in Q und P1 modulo P2 in R zurck
{
  cout << "olddiv: don't call me! i'm slow!" << endl; exit(1);

  for (int i=0; i<kQ; ++i) mpz_set_ui(Q[i],0);
  for (int i=0; i<kR; ++i) mpz_set(R[i],P1[i]);
  kR=k1; 
  mpz_t inv,x,y;
  mpz_init(inv); mpz_init(x); mpz_init(y);

  if (k2==0)
    {
      cout << endl;
      cerr << "Polynomdivision: Division durch Nullpolynom!" << endl;
      exit(1);
    }
  kR--; k2--; k1--;
  while (k2>=0  && (mpz_cmp_ui(P2[k2],0))==0) k2--;
  if (k2<0)
    {
      cout << endl;
      cerr << "Polynomdivision: Division durch Nullpolynom! (-1)" << endl;
      exit(1);
    }
  if (mpz_invert(inv,P2[k2],m)==0)
   {
     cout << "--->";
     print(P2,k2+1);
     cout << endl;
     cerr << "Polynomdivision: Inverses existiert nicht!" << endl;
     exit(1);
   }
  if (k1<0) // Division des Nullpolynoms -> 0
    {
      cout << "Blieblablub" << endl;
      kQ=kR=0;    
      goto fertig;
    }

  while (kR>=k2)
   {
    const int Graddifferenz=kR-k2;

    //Skalar=R[kR]/P2[k2] 
    mpz_mul(x,inv,R[kR]); mpz_mod(x,x,m);
    mpz_set(Q[Graddifferenz],x);
    mpz_set_ui(R[kR],0);
    kR--;
    for (int i=k2-1; i>=0; --i )
     {
       int j = kR+i-(k2-1);
       mpz_mul(y,P2[i],x);
       mpz_sub(y,R[j],y);
       mpz_mod(R[j],y,m);
     } 
   }

  while (kR>=0 && mpz_cmp_ui(R[kR],0)==0) kR--;

  kQ=k1-k2+1; kR++;
 fertig:
  mpz_clear(inv); mpz_clear(x); mpz_clear(y);
#if 1
  cout << "polynomdivision Result:" << endl;
  cout << "A(x) = "; print(P1,k1+1); cout << endl;
  cout << "B(x) = "; print(P2,k2+1); cout << endl;
  cout << "Q(x) = "; print(Q,kQ); cout << endl;
  cout << "R(x) = "; print(R,kR); cout << endl;
#endif
}


void oldmod(TPolynom R, int &kR,
            const TconstPolynom P1, int k1,
            const TconstPolynom P2, int k2, const mpz_t m)
// Polynomrest der Polynomdivision, O(n^2)
// liefert P1 modulo P2 in R zurck
{
#ifdef DEBUG
  cout << "POLMOD IN (old)" << endl;
#endif
  for (int i=0; i<kR; ++i) mpz_set(R[i],P1[i]);
  kR=k1; 
  /* mit minimalem Speicher innerhalb von R kme man aus, wenn
     man die Resultatgre vorbestimmt und R dann als
     Ringpuffer verwaltet (dabei darauf achtend, da die
     Koeffizienten des Resulatpolynoms zum Schlu in der richtigen
     Reihenfolge stehen).
     -> ist aber noch nicht implementiert.
  */

  mpz_t inv,x;
  mpz_init(inv); mpz_init(x);

  if (k2==0)
    {
      cout << endl;
      cerr << "Polynomdivision: Division durch Nullpolynom!" << endl;
      exit(1);
    }
  kR--; k2--; k1--;
  while (k2>=0 && (mpz_cmp_ui(P2[k2],0))==0) k2--;
  if (k2<0)
    {
      cout << endl;
      cerr << "Polynomdivision: Division durch Nullpolynom! (-1)" << endl;
      exit(1);
    }
  if (mpz_invert(inv,P2[k2],m)==0)
   {
     cout << "--->";
     print(P2,k2+1);
     cout << endl;
     cerr << "Polynomdivision: Inverses existiert nicht!" << endl;
     exit(1);
   }
  if (k1<0) // Division des Nullpolynoms -> 0
   {
     cout << "Division des Nullpolynoms!" << endl;
     kR=0;    
     goto fertig;
   }

  while (kR>=k2)
   {
    //Skalar=R[kR]/P2[k2] 
    mpz_mod(R[kR],R[kR],m); mpz_mul(x,inv,R[kR]); mpz_mod(x,x,m);
    mpz_set_ui(R[kR],0);
    for (int i=k2-1; i>=0; --i)
     {
       mpz_submul(R[kR-k2+i],P2[i],x);
     }
    --kR;
   }
  // nun noch Ergebnis normieren
  for (int i=kR; i>=0; --i)
    mpz_mod(R[i],R[i],m);
  while (kR>=0 && mpz_cmp_ui(R[kR],0)==0) kR--;
  kR++;
 fertig:
  mpz_clear(inv); mpz_clear(x);
#ifdef DEBUG
  cout << "Polynomdivision (MODULO) Result:" << endl;
  cout << "A(x) = "; print(P1,k1+1); cout << endl;
  cout << "B(x) = "; print(P2,k2+1); cout << endl;
  cout << "R(x) = "; print(R,kR); cout << endl;
  cout << "POLMOD OUT (old)" << endl;
#endif
}


void div(TPolynom Q, int &kQ,
         const TconstPolynom P1, const int k1,
         const TconstPolynom P2, const int k2, const mpz_t m)
{
  // (schnelle) Polynomdivision durch Multiplikation mit dem inversen Polynom
  /* Methode:
     "reziprokes" Polynom zu P2 ermitteln, dieses mit P1 multiplizieren und
     die (k1-k2) hchsten Koeffizenten des Produktpolynoms sind dann das Ergebnis.
     Asymptotisch sollte diese Methode schneller sein, da die Zeitkomplexitt
     der  Ermittlung des reziproken Polynoms und der anschlieende Multiplikation im
     wesentlichen von der Komplexitt des verwendeten Multiplikationsalgorithmus
     (Karatsuba, etc.) abhngen.
  */

#ifdef DEBUG
  cout << "div-in" << " k1=" << k1 << ", k2=" << k2 << endl;
  cout << "div:P1="; print(P1,k1); cout << endl;
  cout << "div:P2="; print(P2,k2); cout << endl;
#endif

  if ( kQ < k1-k2+1 ) cerr << "div: Zielpolynom zu klein!" << endl;
  if (k2<1) cerr << "Warnung: Null-Polynome bei Division!" << endl;
  if ( (k1>0 && mpz_cmp_ui(P1[k1-1],0)==0) ||
       (k2>0 && mpz_cmp_ui(P2[k2-1],0)==0) )
    {
      cerr << "div: Fhrende Nullen! (unerwnscht!)" << endl;
      exit(1);
    }

  if (k1==k2)
   {
     // Spezialfall: Division zweier Polynome mit gleichem Grad:
     cout << "div: Grad P1=P2 -> easy-div" << endl;
     // Q[0]=P1[k1-1]/P2[k2-1], kQ=1;
     if (kQ<1) cerr << "Speicherproblem!" << endl;
     mpz_invert(Q[0],P2[k2-1],m);
     mpz_mul(Q[0],Q[0],P1[k1-1]); mpz_mod(Q[0],Q[0],m);
     kQ=(mpz_cmp_ui(Q[0],0)==0) ? 0 : 1;
     return;
   }

  int scale=0; // hiermit wird das reziproke Polynom skaliert
  if (k1>=2*k2)
   {
#ifdef VERBOSE
      cout << __FUNCTION__ << ": Probleme mit reziprokem Polynom: k1>=2*k2!"
           << " k1,k2=" << k1 << "," << k2 << endl;
#endif
      scale=k1-k2-1;
   }

  // Hilfspolynom RP2 bereitstellen
  const int mem_kRP2=k2+scale; TTempPolynom RP2(mem_kRP2);
  int kRP2 = mem_kRP2;
  reciprocal(RP2,kRP2,P2,k2,m,scale); // und berechnen
  // -> RP2(x) = x^(2n-2) / P2(x)

  // da wir spter nur die k1-k2 hchsten Koeffizienten bentigen,
  // brauchen wir auch nicht sehr viel mehr auszurechnen...
  const int startP1=k1-(k1-k2+1);
  const int startRP2=kRP2-(k1-k2+1);
#ifdef VERBOSE
  cout << "div-shortcut: " << startP1 << ", " << startRP2 << endl;
#endif
  const int mem_kH=(k1-startP1)+(kRP2-startRP2)-1; TTempPolynom H(mem_kH);

  const int kH=mul(H,mem_kH,&P1[startP1],k1-startP1,&RP2[startRP2],kRP2-startRP2,m);
  // -> H(x)=P1(x)*RP2(x)

#if 0
  cout << "Divisionsergebnis: " << endl;
  cout << "full: "; print(H,kH); cout << endl;
  cout << "kRP2=" << kRP2 << ", k2=" << k2 << endl;
#endif

  // nun die k1-k2+1 hchsten Koeffizienten holen, sie sind das Ergebnis
  for (int i=0; i<=k1-k2; ++i) mpz_set(Q[i],H[kH-1-(k1-k2)+i]);
  kQ=k1-k2;
  while (kQ>=0 && mpz_cmp_ui(Q[kQ],0)==0) kQ--; // Polynom normieren
  kQ++; // size=Polynomgrad+1 !!

#ifdef DEBUG
  cout << "div-out" << endl;
#endif
}


void mod(TPolynom R, int &kR,
         const TconstPolynom P1, int k1,
         const TconstPolynom P2, int k2, const mpz_t m)
// Polynomrest der Polynomdivision, schnelle(re) Methode
// liefert P1 modulo P2 in R zurck
/* asymptotisch schneller als "naives modulo", weil schnelle Multiplikation
   verwendet werden kann. */
{

#if 0 || defined(DEBUG)
  cout << "POLMOD IN (new)" << endl;
  cout << "k1=" << k1 << ", k2=" << k2 << ", kR=" << kR << endl;
//  cout << "mod:P1="; print(P1,k1); cout << endl;
//  cout << "mod:P2="; print(P2,k2); cout << endl;
#endif

  if (k2<1) cerr << "Warnung: Null-Polynome bei Modulo!" << endl;

#ifdef VERBOSE
  if ( (k1>0 && mpz_cmp_ui(P1[k1-1],0)==0) ||
       (k2>0 && mpz_cmp_ui(P2[k2-1],0)==0) )
   {
     cerr << __FUNCTION__ << ": Fhrende Nullen! (unerwnscht!)" << "(fr " <<  k1 << "x" << k2 << ")" <<endl;
   }
#endif

  k1--; while (k1>=0 && mpz_cmp_ui(P1[k1],0)==0) k1--; k1++; // normieren
  k2--; while (k2>=0 && mpz_cmp_ui(P2[k2],0)==0) k2--; k2++; // normieren

#ifdef DEBUG
  cout << "k1=" << k1 << ", k2=" << k2 << ", kR=" << kR << endl;
  cout << "mod:P1="; print(P1,k1); cout << endl;
  cout << "mod:P2="; print(P2,k2); cout << endl;
#endif

  if (k2>k1)
   { 
     cout << "triviales mod:  P1 mod P2 -> P1  fr P2>P1" << endl;
     if (kR<k1) cerr << "Platz im Zielpolynom reicht nicht!" << endl;
     for (int i=0; i<k1; ++i) mpz_set(R[i],P1[i]);
     kR=k1;
     return;
   }

#if 1 
  /* speedup analog zu quicksort: nahe an den Blttern asymptotisch
     schlechtere, aber fr kleine Werte schnellere Methode verwenden. */
  if (k1<150 || k2<16)
   {
     //cout << "calling oldmod" << endl;
     oldmod(R,kR,P1,k1,P2,k2,m);
     return; 
   }
#endif

  // Quotientenpolynom Q ermitteln
  const int mem_kQ=k1-k2+1; TTempPolynom Q(mem_kQ); 
  int kQ=mem_kQ;
  div(Q,kQ,P1,k1,P2,k2,m); // wichtig: schnelle Division!
  
  // P1-Q*P2 berechnen
  const int mem_kM=kQ+k2-1; TTempPolynom M(mem_kM); 
  if (kQ>k2)
   { 
     cout << "shortcut mglich! kQ=" << kQ << ", k2=" << k2 << endl;
     kQ=k2;
     //cout << "kQ=k2=" << k2 << endl;
    }

 //cout << "FASTmod: " << kQ << "x" << k2 << endl; 
#if 1
  mul(M,mem_kM,Q,kQ,P2,k2,m); // wichtig: schnelle Multiplikation!
#else
  int kM=mul(M,mem_kM,Q,kQ,P2,k2,m); // wichtig: schnelle Multiplikation!
  if (kM<k2-1) cerr << "mod: Restprobleme" << endl;
  cout << "prod: "; print(M,kM); cout << endl;
#endif

  // shortcut: Polynomrest kann nur maximal k2-1 viele Koeffizienten umfassen
  {
    int i=k2-2; while (i>=0 && mpz_cmp(P1[i],M[i])==0) --i; // Ergebnis normieren
    if (i+1>kR) cerr << "mod: Speichermangel im Zielpolynom!" << endl;
    kR=i+1;
    while (i>=0) 
     {
       mpz_sub(R[i],P1[i],M[i]); mpz_mod(R[i],R[i],m);
       --i;
     }
  }

#ifdef DEBUG
  cout << "Polynomdivision (fast MODULO) Result:" << endl;
  cout << "A(x) = "; print(P1,k1); cout << endl;
  cout << "B(x) = "; print(P2,k2); cout << endl;
  cout << "R(x) = "; print(R,kR); cout << endl;
  cout << "POLMOD OUT (new)" << endl;
#endif
}


void multipoint_eval_rek(const TPolynom* R, const TconstPolynom P, const int k, TPolynom* A, const int h,
			 const mpz_t m, mpz_t* &res, int* const pos, const int* const step, const int* const stop)
// wird von multipoint_eval aufgerufen
{
#ifdef DEBUG
  cout << "rekursiver Aufruf fr h=" << h << endl;
#endif
  if (h==0) // Blatt erreicht -> Auswertung mglich
    {
      // direkte Auswertung des Blattes: (a*x+b) mod (x+c) = b-a*c
      if (k==1)
         mpz_mod(res[0],P[0],m);
      else
       {
         mpz_mul(res[0],P[1],A[h][pos[h]]);
         mpz_sub(res[0],P[0],res[0]);
         mpz_mod(res[0],res[0],m);
       }
      res++;
    }
  else
    {
      int kR=k;
      mod(&R[h][0],kR,P,k,&A[h][pos[h]],step[h],m);
      multipoint_eval_rek(R,R[h],kR, A,h-1, m,res,pos,step,stop);
      if (pos[h-1]<stop[h-1])
	multipoint_eval_rek(R,R[h],kR, A,h-1, m,res,pos,step,stop);
    }
  pos[h]+=step[h];
} 


void multipoint_eval(mpz_t* res,
                     const TconstPolynom P, const int k, 
                     const mpz_t* const Argumentenfeld, const int size,
                     const mpz_t m)
{
  if ( k<3 || size<3 || size>k-1 )
   {
     cerr << "multipoint_eval: invalid parameters!" << endl;
     exit(1);
   }

  const int MaxHoehe = ld(size)+1;

  TPolynom* const A = new TPolynom[MaxHoehe];
  int* const pos       = new int[MaxHoehe];
  int* const steparray = new int[MaxHoehe];
  int* const stop      = new int[MaxHoehe];

  for (int i=0; i<MaxHoehe; ++i) pos[i]=steparray[i]=stop[i]=0; // Initialisierung 

  // for optimizing mpz-memory-allocation & fragmentation we will use mpz_init2
  // therefore we need to estimate our operand size (wrong estimate cause no harm
  // except for triggering more unnecessary reallocations)
  const unsigned int estimated_operand_size = 2*mpz_sizeinbase(m,2)+5;

  /*
    A[h][s] ist das Feld der Polynome in der Hhe h (h=0 -> Bltter=monische
    Polynome vom Grad 1)
  */

  int h = 0, anz=size, step=2, s=anz*step;
  steparray[h]=step; stop[h]=s;

#ifdef VERBOSE
  cout << "Polynome fr Hhe " << h << " kalkulieren..." << endl;
#endif
  A[h] = new mpz_t[s]; // Platz fr Polynome in Hhe h allokieren
  for (int i=0, j=0; i<anz; ++i, j+=step)
    {
      // monische Polynome vom Grad 1 erzeugen
      mpz_init_set_ui(A[h][j+1],1); // 1*x
      mpz_init_set(A[h][j],Argumentenfeld[i]); mpz_neg(A[h][j],A[h][j]);
    }

                     // auch diesen Sonderfall optimal bearbeiten 
  while ( anz>2 || (anz==2 && k==2*size+1) )
    {
      ++h;
      if (h>=MaxHoehe)
       {
         cout << __FILE__ << ", " << __FUNCTION__ << ": line " <<  __LINE__ << endl;
         cerr << "too much iteration steps..." << endl;
	 cerr << "please increase MaxHoehe and recompile!" << endl;
	 exit(1);
       }

      const int oldstep = step;
      const int rest = anz % 2;
      steparray[h]=step=step*2-1; anz=(anz/2); stop[h]=s=(anz+rest)*step;
#ifdef VERBOSE
      cout << "Polynome fr Hhe " << h << " kalkulieren..." << endl;
#endif
      A[h] = new mpz_t[s]; // Platz fr Polynome in Hhe h allokieren
      for (int j=0; j<s; ++j) mpz_init2(A[h][j],estimated_operand_size); // mpz_t initialisieren

      for (int i=0, j=0; i<anz; ++i, j+=step)
	{
	  monic_mul(&A[h][j],step,
	      &A[h-1][2*i*oldstep],oldstep, &A[h-1][(2*i+1)*oldstep],oldstep,m);
	}
      if (rest) 
	{
	  for (int i=0; i<oldstep; i++)
	    mpz_set(A[h][s-step+i],A[h-1][stop[h-1]-oldstep+i]);
	}
      anz+=rest;
    }

  //cout << "READY for recursion..." << endl;

  // nun sind die einzelnen Polynome bereit fr Polynomdivision
  TPolynom* const R = new TPolynom[MaxHoehe]; // Polynom-Remainder
  for (int i=0; i<MaxHoehe; ++i)
    {
      R[i]=new mpz_t[k];
      for (int j=0; j<k; ++j) mpz_init2(R[i][j],estimated_operand_size);
    }

  for (int i=0; i<anz; i++) multipoint_eval_rek(R,P,k,A,h,m,res,pos,steparray,stop);

  for (int i=MaxHoehe-1; i>=0; --i)
    {
      for (int j=k-1; j>=0; --j) mpz_clear(R[i][j]);
      delete [] R[i];
    }
  delete [] R;

  while (h>=0)
    {
      for (int i=stop[h]-1; i>=0; --i) mpz_clear(A[h][i]);
      delete [] A[h]; // Speicher freigeben
      --h;
    }
  delete [] pos;
  delete [] steparray;
  delete [] stop;
  delete [] A;
}


int Polynom_aus_Nullstellen_konstruieren
      (TPolynom & res,
       const mpz_t* const Argumentenfeld, const int size,
       const mpz_t m)
{
  // Aufgabe:
  // Erzeugt ein neues Polynom mit den im Argumentenfeld gegebenen Nullstellen,
  // legt dieses Polynom in "res" ab und liefert die Gre als Funktionswert
  // zurck. Um sicherzustellen, dass durch versehentliche bergabe eines
  // vorhandenen Polynoms in "res" kein Memory-Leak entsteht, wird zustzlich
  // verlangt, da "res" beim Aufruf ein Pointer auf NULL ist.
  // additional remark:
  // Sure, it would be possible to return this pointer instead of using a reference
  // parameter, but what if you forget to delete it later? -- Using this technique,
  // the programmer is at least urged to provide a valid container fr our data!

  if (res!=NULL)
   {
     cout << __FILE__ << ", " << __FUNCTION__ << ": line " <<  __LINE__ << endl;
     cerr << "First parameter is a call by reference," << endl;
     cerr << "it should initilly point to NULL (to avoid memory-leaks)," << endl;
     cerr << "because a new pointer to new data will be generated and" << endl;
     cerr << "there is no need for initially data pointed by res!" << endl;
     exit(1);
   }

  if ( size < 3  )
   {
     cerr << "Polynom aus Nullstellen konstruieren: invalid parameters!" << endl;
     exit(1);
   }  

  const int MaxHoehe = ld(size)+1;
  //MARK; cout << "MaxHoehe: " << MaxHoehe << endl;

  TPolynom* const A = new TPolynom[MaxHoehe];
  int* const stop = new int[MaxHoehe];

  // for optimizing mpz-memory-allocation & fragmentation we will use mpz_init2
  // therefore we need to estimate our operand size (wrong estimate cause no harm
  // except for triggering more unnecessary reallocations)
  const unsigned int estimated_operand_size = 2*mpz_sizeinbase(m,2)+5;

  /*
    A[h][s] ist das Feld der Polynome in der Hhe h (h=0 -> Bltter=monische
    Polynome vom Grad 1)
  */

  int h = 0, anz=size, step=2, s=anz*step;
  stop[h]=s;
#ifdef VERBOSE
  cout << "Polynome fr Hhe " << h << " kalkulieren..." << endl;
#endif
  A[h] = new mpz_t[s]; // Platz fr Polynome in Hhe h allokieren
  for (int i=0, j=0; i<anz; ++i, j+=step)
    {
      // monische Polynome vom Grad 1 erzeugen
      mpz_init_set_ui(A[h][j+1],1); // 1*x
      mpz_init_set(A[h][j],Argumentenfeld[i]); mpz_neg(A[h][j],A[h][j]);
    }

  while (anz>1)
    {
      ++h;
      if (h>=MaxHoehe)
       {
         cout << __FILE__ << ", " << __FUNCTION__ << ": line " <<  __LINE__ << endl;
         cerr << "too much iteration steps..." << endl;
	 cerr << "please increase MaxHoehe and recompile!" << endl;
	 exit(1);
       }
      const int oldstep = step;
      const int rest = anz % 2;
      step=step*2-1; anz=(anz/2); stop[h]=s=(anz+rest)*step;
#ifdef VERBOSE
      cout << "Polynome fr Hhe " << h << " kalkulieren..." << endl;
#endif
      A[h] = new mpz_t[s]; // Platz fr Polynome in Hhe h allokieren
      for (int j=0; j<s; ++j) mpz_init2(A[h][j],estimated_operand_size); // mpz_t initialisieren
      for (int i=0, j=0; i<anz; ++i, j+=step)
	{
	  monic_mul(&A[h][j],step,
	      &A[h-1][2*i*oldstep],oldstep, &A[h-1][(2*i+1)*oldstep],oldstep,m);
	}
      if (rest) 
	{
	  for (int i=0; i<oldstep; i++)
	    mpz_set(A[h][s-step+i],A[h-1][stop[h-1]-oldstep+i]);
	}
      anz+=rest;
    }

  s--; // nun Ergebnispolynom normieren 
  while (s>=0 && mpz_cmp_ui(A[h][s],0)==0) { mpz_clear(A[h][s]); --s; }
   ++s;
  res=A[h]; // Ergebnispolynom

  while (--h>=0)
    {
      for (int i=0; i<stop[h]; ++i) mpz_clear(A[h][i]);
      delete [] A[h]; // Speicher freigeben
    }

  delete [] stop;
  delete [] A;
  return s; // Gre des Ergebnispolynoms
}


} // namespace polynomial



#if 0  /* only for testing the functions */

// to check the stuff do:  g++ -O2 -g -Wall polynomial.cc -lgmp
// and for  dft-check do:  g++ -O2 -g -Wall -DUSE_DFT polynomial.cc -lgmp

using namespace polynomial;


bool do_check(const int maxk, const int anzPunkte, const mpz_t m)
{
  cout << endl << endl << "====================================" << endl;
  cout << " MULTIPOINT POLYNOM-EVAL SANITY " << maxk << ", " << anzPunkte << endl; 

  const size_t limbs = mpz_size(m);
  int i;

  mpz_t Polynom [maxk];
  for (i=0; i<maxk; ++i) { mpz_init(Polynom[i]); mpz_random(Polynom[i],limbs); mpz_mod(Polynom[i],Polynom[i],m); }

  mpz_t Punkte [anzPunkte];
  for (i=0; i<anzPunkte; ++i) { mpz_init(Punkte[i]); mpz_random(Punkte[i],limbs); mpz_mod(Punkte[i],Punkte[i],m); }

  mpz_t Res_Horner [anzPunkte];
  for (i=0; i<anzPunkte; ++i) mpz_init(Res_Horner[i]);
 
  mpz_t Res_Multipoint [anzPunkte];
  for (i=0; i<anzPunkte; ++i) mpz_init(Res_Multipoint[i]);

  // testweise mit Hornerschema auswerten
  cout << "Starte Horner" << endl;
  time_t Start = time(NULL);
  for (i=0; i<anzPunkte; ++i) eval(Res_Horner[i],Polynom,maxk,Punkte[i],m);
  time_t Stop = time(NULL);
  cout << difftime(Stop,Start) << " Sekunden." << endl;

  // nun Multipoint-Auswertung
  cout << "Starte Multipoint" << endl;
  Start = time(NULL);
  multipoint_eval(Res_Multipoint,Polynom,maxk,Punkte,anzPunkte,m);
  Stop = time(NULL);
  cout << difftime(Stop,Start) << " Sekunden." << endl;

  cout << "berprfe Punkte fr Horner <-> Multipoint..." << endl;
  bool error = false;
  for (int i=0; i<anzPunkte; ++i)
   {
#ifdef DEBUG
     mpz_out_str(stdout,10,Punkte[i]); cout << " -> ";
     mpz_out_str(stdout,10,Res_Horner[i]); cout << " (Horner)  ";
     mpz_out_str(stdout,10,Res_Multipoint[i]); cout << " (Multipoint)" << endl;
#endif
     if (mpz_cmp(Res_Horner[i],Res_Multipoint[i]))
       {
	 error=true;
	 cerr << "Abweichung!! " << i << endl;
         mpz_out_str(stdout,10,Punkte[i]); cout << " -> ";
         mpz_out_str(stdout,10,Res_Horner[i]); cout << " (Horner)  ";
         mpz_out_str(stdout,10,Res_Multipoint[i]); cout << " (Multipoint)" << endl;
         break;
       }
   }
  if (error)
   {
     cout << "FAILED!" << endl;
     exit(1);
   }

  cout << "bereinstimmung festgestellt." << endl;
  cout << "PASSED." << endl;

  for (i=0; i<maxk; ++i) { mpz_clear(Polynom[i]); }
  for (i=0; i<anzPunkte; ++i) mpz_clear(Punkte[i]);
  for (i=0; i<anzPunkte; ++i) mpz_clear(Res_Horner[i]);
  for (i=0; i<anzPunkte; ++i) mpz_clear(Res_Multipoint[i]);
  return true;
}

bool performance_check(const int maxk, const int anzPunkte, const mpz_t m)
{
  cout << endl << endl << "====================================" << endl;
  cout << " MULTIPOINT POLYNOM-EVAL PERFORMANCE " << maxk << ", " << anzPunkte << endl; 

  const size_t limbs = mpz_size(m);
  int i;

  mpz_t Polynom [maxk];
  for (i=0; i<maxk; ++i) { mpz_init(Polynom[i]); mpz_random(Polynom[i],limbs); mpz_mod(Polynom[i],Polynom[i],m); }

  mpz_t Punkte [anzPunkte];
  for (i=0; i<anzPunkte; ++i) { mpz_init(Punkte[i]); mpz_random(Punkte[i],limbs); mpz_mod(Punkte[i],Punkte[i],m); }

  mpz_t Res_Multipoint [anzPunkte];
  for (i=0; i<anzPunkte; ++i) mpz_init(Res_Multipoint[i]);

  // nun Multipoint-Auswertung
  cout << "Starte Multipoint" << endl;
  time_t Start = time(NULL);
  multipoint_eval(Res_Multipoint,Polynom,maxk,Punkte,anzPunkte,m);
  time_t Stop = time(NULL);
  cout << difftime(Stop,Start) << " Sekunden." << endl;

  for (i=0; i<maxk; ++i) { mpz_clear(Polynom[i]); }
  for (i=0; i<anzPunkte; ++i) mpz_clear(Punkte[i]);
  for (i=0; i<anzPunkte; ++i) mpz_clear(Res_Multipoint[i]);
  return true;
}


int main()
{
  cout << endl << endl
       << "===============================================" << endl;
  cout << "POLYNOMIAL-MULTIPOINT-EVAL doing some checks..." << endl;
  cout << "===============================================" << endl;

  mpz_t m;
  mpz_init(m);

  // 120-stellige Zahl generieren
  mpz_set_ui(m,10); mpz_pow_ui(m,m,120); mpz_add_ui(m,m,3);

#if 0
  // torture check...
  for (int maxk=4; maxk<(1<<14); maxk*=2)
   {
     for (int i=3; i<maxk; ++i) do_check(maxk,i,m);
     for (int i=3; i<maxk+1; ++i) do_check(maxk+1,i,m);
   }
#endif

#if 1
  for (int maxk=4; maxk<(1<<19); maxk*=2)
   {
     performance_check(maxk+1,maxk,m);
   }
#endif

  // now a normal check

  for (int maxk=4; maxk<(1<<16); maxk*=2)
   {
     do_check(maxk,maxk-1,m);
     do_check(maxk+1,maxk,m);
   }

  mpz_set_str(m,"3923385745693995079670229419275984584311007321932374190635656246740175165573932140787529348954892963218868359081838772941945556717",10);
  for (int maxk=4; maxk<2048; maxk*=2)
   {
     do_check(maxk,maxk-1,m);
     do_check(maxk+1,maxk,m);
   }
}

#endif

#endif  // __POLYNOMIAL__
