#ifndef polphi_template_HEADER
#define polphi_template_HEADER

/*! @file
 * @brief
 * template definition for pollard-phi-like factoring algorithms
 */


template <class CRing, class CRingPhase2>
void polphi_template(const int phase1, const double phase2)
{
  // Pollard-Phi Methode modifiziert in drei Phasen:
  // Phase 0: alle kleine Zahlen bis konstanter Obergrenze
  // Phase 1: (im wesentlichen) Primzahlen bis phase1 als Obergrenze
  //          (Primfaktor-1) kann/darf in alle diese Primzahlen zerfallen
  // Phase 2: (im wesentlichen) Primzahlen bis phase2 als Obergrenze
  //          bis auf einen(!) aus Phase 2 mssen alle anderen
  //          Faktoren von (Primfaktor-1) bereits vorher entdeckt worden
  //          sein, um Primfaktor (effizient) entdecken zu knnen.
  //          (improved standard continuation with pairing & fft)

  using numtheory::is_prime;
  using numtheory::gcd;

  mpz_t x,y;
  mpz_init(x); mpz_init(y);

  CRing a,b;
  
  vector<int> KnownIndices;
  int VorigerIndex = 0;
  unsigned int polphi_seed = 0;
  const int i_Start = 2;
  int i;
  short int d;

restart_phi0_with_new_seed:
  KnownIndices.clear(); VorigerIndex=0;
  ++polphi_seed;

  // Phase 0: kleine Zahlen (mit Primzahlpotenzen)
  cout << "--------------------------------------------------" << endl;
  cout << CRing::name << "-phase 0" << endl;
  if (!a.set_startvalue(polphi_seed)) goto fertig; // seed wasn't accepted
  for (i=i_Start; i<10000; i++)
    {
      a.pow_mod(i,n);
retry:
      a.test_gcd(x,n);
      if (mpz_cmp_ui(x,1)!=0)  /* Teiler gefunden? */
        {
          cout << "factor found in phase 0: i=" << i << endl;
          if (mpz_cmp(x,n)==0) // gefundener Faktor = n ?
            {
              cout << "factor is trivial." << endl;

              int h=i; int j=2;
              while (h>j) // grten Primteiler ermitteln 
               {
                 while(h>j && h%j==0) h/=j;
                 ++j;
               }

              if (KnownIndices.empty()) KnownIndices.push_back(h);
              else // war dieser bereits im vorigen Durchlauf hinzugefgt
               if (VorigerIndex!=i || KnownIndices.back()!=h) KnownIndices.push_back(h);
               else
                {
                  // dann ggf. wieder entfernen
                  if (KnownIndices.back()!=i/h) KnownIndices.pop_back();
                  // und durch anderen Teiler ersetzen
                  KnownIndices.push_back(i/h);
                }
              VorigerIndex=i;

              cout << "KnownIndices:";
              for (vector<int>::iterator p=KnownIndices.begin(); p!=KnownIndices.end(); ++p) cout << " " << *p;
              cout << endl;

              if (KnownIndices.empty() || KnownIndices.back()==1)
               {
                 MARK; cout << "Gebe auf fr " << n << endl;
                 //{ char ch; cin >> ch; }
                 goto restart_phi0_with_new_seed; // trivialer Teiler -> Faktoren anders aufrollen oder Verfahren abbrechen
               }

              if (KnownIndices.size()<100)
               {
                 a.set_startvalue(polphi_seed);
                 for (vector<int>::iterator p=KnownIndices.begin(); p!=KnownIndices.end(); ++p) a.pow_mod(*p,n);
                 cout << "Versuche es erneut..." << endl;
                 i=i_Start; goto retry;
               }
              MARK; cout << "Breche ab fr " << n << endl;
              //{ char ch; cin >> ch; }

              goto restart_phi0_with_new_seed; // trivialer Teiler -> Faktoren anders aufrollen oder Verfahren abbrechen
            }

          if (mpz_probab_prime_p(x,probab_prime_checks))
            {
              const unsigned int exponent=mpz_remove(n,n,x);
              cout << x << " is factor." << endl;
              Factorization_to_file << MAL(x,exponent) << " [" << CRing::name << "0]" << flush;
            }
          else
            {
              cout << x << " is composite factor." << endl;
              mpz_divexact(n,n,x);
              if (mpz_probab_prime_p(n,probab_prime_checks))
               {
                 mpz_swap(n,x);
                 cout << x << " is factor." << endl;
                 Factorization_to_file << MAL() << x << " [" << CRing::name << "0/factorswap]" << flush;
                 //goto fertig;
               }
              else
               {
                mpz_t saved_n;
                mpz_init_set(saved_n,n);
                mpz_swap(n,x);
                cout << "Rufe " << CRing::name << " rekursiv auf..." << endl;
                polphi_template<CRing,CRingPhase2>(phase1,phase2);
                cout << "zurck aus Rekursion. Setze " << CRing::name << " fort..." << endl;
                const unsigned int exponent=mpz_remove(saved_n,saved_n,n)+1;
                if (mpz_probab_prime_p(n,probab_prime_checks))
                 Factorization_to_file << MAL(n,exponent) << " [" << CRing::name << "0]" << flush;
                else
                 Factorization_to_file << MAL(n,exponent) << " [" << CRing::name << "0] [composite]" << flush;
                mpz_swap(n,saved_n); mpz_clear(saved_n);
               }
              a.set_startvalue(polphi_seed); i=i_Start;
            }
          KnownIndices.clear(); VorigerIndex=0;
          if (mpz_probab_prime_p(n,probab_prime_checks) || mpz_cmp_ui(n,1)==0) goto fertig;
        }
    }
  a.pow_mod(2,n); // sicherheitshalber geraden Index garantieren
   
  /* normale phi-Phase 1 */
  {
   // Phase 1: Primzahlen ohne Potenzen (aber knnen in beliebiger Kombination p-1 teilen)
   const int D=CRing::SieveIntervalSize;
   i=i-(i%D); // i mit Eigenschaft i=5 (mod 6) fr schnellere Primzahlermittlung
   cout << CRing::name << "-phase 1" << endl;
   while (i<=phase1)
    {
#ifdef REACT_ON_SIGUSR
      if (USRSignalHandler.got_SIGUSR1()) goto fertig;
      if (USRSignalHandler.got_SIGUSR2()) break;
#endif
      b.set(a); // alten Wert fr evtl. Aufrollen merken
      // Primzahlen aussieben
      bool sieve[D];
      for (int j=1; j<=D; j+=2) sieve[j]=true; // Sieb initialisieren
      for (int p=5, dp=2; p*p<D+i; p+=dp, dp=6-dp)
       for (int j=p-(i%p); j<D; j+=p) sieve[j]=false;
       /* 
         Bemerkung:
          a) p*p<D+i ist als Primzahlbedingung fr das Intervall
             etwas zu streng, aber strt wegen Phase 0 nicht.
             (-> Primzahlen unter sqrt(D) werden nicht erkannt) 
          b) p-(i%p) ist zwar p, wenn p teilt i;
             aber: den Index 0 brauchen wir spter nicht!
       */
            
      for (int j=1, dj=4; j<D; j+=dj, dj=6-dj)
       {
         //if (sieve[j]!=is_prime(i+j))
         //  { cerr << "keine bereinstimmung: " << i+j << endl; }
         if (sieve[j]) a.fast_pow_mod(i+j,n);
       }
      /* Bemerkung:
         Nicht durch die Literatur verwirren lassen:
         Das akkumulierte Produkt der (a-1)-Werte mu natrlich
         nicht gebildet werden. Das wre eine total berflssige
         Aktion, da ein Teiler auch bei nachfolgenden a's erhalten
         bleiben mu (aufgrund des kleinen Fermatschen Satzes!).
         Falls sich andererseits mehrere Teiler akkumulieren, so kann
         auch das akkumulierte Produkt dies nicht verhindern...
         Keine Ahnung, warum ich das akkumulierte Produkt frher hier
         gebildet habe, offenbar habe ich mich verwirren lassen...
         (Ist ja auch wie mit den Elefanten: "Warum klatschst du denn
          dauernd in die Hnde?" - "Um die Elefanten zu verscheuchen!" -
          "Welche Elefanten?" - "Siehtst du! Alle verscheucht!")
      */
      // Schleife vermeidet hufige gcd-Berechnung
      // in seltenen Fllen knnen aber mehrere Faktoren zusammenfallen
      // -> Phi-Methode versagt dann (ggf. Schleifendurchlufe vermindern)

      int old_i=i; i+=D;
      cout << "Phase1: " << i << "/" << phase1 << "\r" << flush;

      a.test_gcd(x,n);
      if (mpz_cmp_ui(x,1)!=0)
        {
          cout << "factor found in phase 1: i=" << i << endl;
          if (mpz_cmp(x,n)==0) // gefundener Faktor = n ?
            {
trivialer_factor_found:
              cout << "Found factor is trivial, trying to isolate..." << endl;
              a.set(b); // Basis vom Intervallanfang
              i=old_i; // zurck zum Intervallanfang
              for (int j=1, dj=4; j<D; j+=dj, dj=6-dj) if (sieve[j])
               {
                 a.pow_mod(i+j,n);
                 a.test_gcd(x,n);
                 if (mpz_cmp_ui(x,1)!=0)
                  { 
                    if (mpz_cmp(x,n)==0) // gefundener Faktor = n ?
                     {
                       cout << "factor is trivial, aborting." << endl;
                       goto fertig; // trivialer Teiler -> Faktoren anders aufrollen oder Verfahren abbrechen
                     }
                    else
                     {
                       cout << "non-trivial factor isolated at i=" << i+j << endl;
                       break; // Faktor isoliert...
                     }
                  }
               }
            }
          if (mpz_probab_prime_p(x,probab_prime_checks))
            {
              const unsigned int exponent=mpz_remove(n,n,x);
              cout << x << " is factor." << endl;
              Factorization_to_file << MAL(x,exponent) << " [" << CRing::name << "1]" << flush;
            }
          else
            {
              mpz_divexact(n,n,x);
              cout << x << " is composite factor." << endl;
              if (mpz_probab_prime_p(n,probab_prime_checks))
               {
                 cout << n << " is factor." << endl;
                 Factorization_to_file << MAL() << n << " [" << CRing::name << "1/factorswap]" << flush;
                 mpz_set(n,x); goto trivialer_factor_found;
               }
              else
               {
                 cout << "Found factor is composite, trying to isolate..." << endl;
                 // The found factor is composite and has already been divided out of our number n.
                 // We could multiply him back again, but we can do much better:
                 // As we are only interested in splitting our composite factor, we can
                 // repeat the search in the prevoius interval, but this time modulo the compisite factor!
                 // For this reason we need local variables for the base "a" and our "n", because
                 // their original values are needed in the main loop, so we shouldn't clobber them.

                 mpz_t local_n;
                 CRing local_a;
                 mpz_init_set(local_n,x);
                 local_a.set(b); // Basis vom Intervallanfang
                 for (int j=1, dj=4; j<D; j+=dj, dj=6-dj) if (sieve[j])
                  {
                    local_a.pow_mod(old_i+j,local_n);
                    local_a.test_gcd(x,local_n);
                    if (mpz_cmp_ui(x,1)!=0)
                     {
                      mpz_divexact(local_n,local_n,x);
                      const unsigned int exponent=mpz_remove(n,n,x)+1;
                      cout << x << " is factor." << endl;
                      if (mpz_probab_prime_p(x,probab_prime_checks))
                       {
                        cout << "factor isolated at i=" << old_i+j << endl;
                        Factorization_to_file << MAL(x,exponent) << " [" << CRing::name << "1]" << flush;
                       }
                      else
                       {
                        cout << "factor still composite at i=" << old_i+j << endl;
                        Factorization_to_file << MAL(x,exponent) << " [" << CRing::name << "] [composite]" << flush;
                       }
                      if (mpz_cmp_ui(local_n,1)==0) break;
                     }
                  }
                 if (mpz_cmp_ui(local_n,1)!=0)
                  {
                    MARK; cerr << "Error: local_n should be 1 now!" << endl; exit(1);
                  }
                 mpz_clear(local_n);
               }
            }
          if (mpz_probab_prime_p(n,probab_prime_checks) || mpz_cmp_ui(n,1)==0) goto fertig;
        }
    }
   i--; d=4; while (!is_prime(i)) { i-=d; d=6-d; } // i auf vorige Primzahl adjustieren
   cout << CRing::name << "-phase 1, up to " << i << endl;
   if (i>=phase2) goto fertig;
  }


  /* improved standard continuation with pairing & fft */
  {
   double ii=i;
   //const double i = -1; // uncomment this just to be sure that i isn't used anymore...

   // Phase 2: isolierte (Prim-)zahl: p-1 mu bis auf diese Zahl vorher vollstndig zerlegbar gewesen sein.
   cout << CRing::name << "-phase 2 (improved with pairing & fft)" << endl;

   /* zur Bedeutung nachfolgender Werte siehe elliptische Kurven,
      fft-continuation */
   unsigned int pg=256, pms=2*1050;
   get_fft_parameter(pg,pms,phase2,n); // Parameter (pg,pms) fr "phase2" holen

   const int D = pms;
   const int Polynom_Index = pg;

   //cout << "size of polynomial: " << Polynom_Index << endl;
   //cout << "single interval size:" << D << endl;

   const polynomial::TPolynom sammel = new mpz_t[Polynom_Index];
   for (int j=0; j<Polynom_Index; ++j) mpz_init(sammel[j]);

   // in a steht die Start-Basis fr Phase 2
   CRingPhase2 a_phase2(a);

   // Precomputing most common used values:
   int Polynom_Index_i = 0;
   for (int j=1; j<D/2; j+=2)
    {
      if (gcd(j,D)==1)
       {
         a_phase2.get_polynomdef_point(x);
         mpz_set(sammel[Polynom_Index_i++],x);
       }
      a_phase2.calc_polynomdef_next_point(); 
    }

   cout << "original size of polynomial: " << Polynom_Index_i << endl;
   polynomial::TPolynom Polynom = NULL; // hier steht das anschlieend konstruierte Polynom drin
   // Rest bis zur nchsten Zweierpotenz mit Nullen auffllen
   /* Bemerkung: Besser wre es, Punkte auerhalb des Suchbereichs
                 zu whlen (Motto: doppeltes, aber auerhalb des
                 Suchbereichs lchriges Netz)
    */
   { 
     int anz_Punkte=Polynom_Index_i;    
     while (anz_Punkte<Polynom_Index) mpz_set_ui(sammel[anz_Punkte++],0);
     Polynom_Index_i=polynomial::Polynom_aus_Nullstellen_konstruieren(Polynom,sammel,anz_Punkte,n);
     cout << "polynomial created. size of polynomial = " << Polynom_Index_i << endl;
   }

   ii=floor(ii/D)*D; // i=i-(i%D); // Startpunkt ermitteln
   a_phase2.calc_EvalStartingPoint(D,ii);
   ii-=D/2; // Intervall von [i-D/2,i+D/2] auf [i,i+D] normieren
   while (ii<=phase2)
    {
      cout << "Phase2: " << ii << "/" << phase2 << endl << flush;
      cout << "Computing and collecting values..." << endl;

      //Werte sammeln
      int sammel_index=0;
      while (sammel_index<(Polynom_Index_i-1)/2)
       {
         a_phase2.get_point_and_calc_next_point(x);
         mpz_set(sammel[sammel_index++],x);
         // Werte fr nchstes Intervall bestimmen...
         ii+=D; // neues Intervall (bis i haben wir nun abgearbeitet)
       }

#ifdef REACT_ON_SIGUSR
      if (USRSignalHandler.got_SIGUSR1()) goto fertig_phase2;
      if (USRSignalHandler.got_SIGUSR2()) continue;
#endif

      // Polynomauswertung starten
      cout << "Phase2: " << ii << "/" << phase2 << endl << flush;
      cout << "starting multipoint polynomial evaluation" << endl;
      cout << "Parameter: polynomial size: " << Polynom_Index_i
           << ", points to evaluate: " << sammel_index << endl;
      polynomial::multipoint_eval(sammel,Polynom,Polynom_Index_i,sammel,sammel_index,n);
      cout << "polynomial evaluation finished, computing gcd." << endl;
      mpz_set(y,sammel[0]);
      for (int j=1; j<sammel_index; ++j)
        { mpz_mul(y,y,sammel[j]); mpz_mod(y,y,n); }
      // nun (erst) den ggT ermitteln
      mpz_gcd(x,y,n);
      if (mpz_cmp_ui(x,1)==0) continue;
      
      // now one or more factor have been found
      // possible problem: boundled factors which have to be separated
      // Da es aber nur log-viele Teiler geben kann, ist dieser
      // Abschnitt nicht zeitkritisch; also strt es wenig, alle
      // Werte nochmal durchzugehen (dies senkt die Wahrscheinlichkeit,
      // akkumulierte Teiler zu erhalten).
    
      for (int j=0; j<sammel_index; ++j)
       {
         mpz_gcd(x,sammel[j],n);
         if (mpz_cmp_ui(x,1)!=0)
          {
           cout << "factor found in phase 2: i=" << ii << endl;
           if (mpz_probab_prime_p(x,probab_prime_checks))
            {
              const unsigned int exponent=mpz_remove(n,n,x);
              cout << x << " is factor." << endl;
              Factorization_to_file << MAL(x,exponent) << " [" << CRing::name << "2]" << flush;
            }
           else
            {
              mpz_divexact(n,n,x);
              cout << x << " is composite factor." << endl;
              if (mpz_probab_prime_p(n,probab_prime_checks))
               {
                 mpz_swap(n,x);
                 cout << x << " is factor." << endl;
                 Factorization_to_file << MAL() << x << " [" << CRing::name << "2/factorswap]" << flush;
               }
              else
               {
                 if (mpz_cmp_ui(n,1)==0)
                  {
                    // this is really interesting!
                    mpz_swap(n,x);
                    mpz_set_d(y,ii); // short hack to get a well-formated ii
                    Factorization_to_file << MAL() << " [" << CRing::name << "2/trivial! i=" << y << "]" << flush;
                    goto fertig_phase2;
                  }
                 else
                  Factorization_to_file << MAL() << x << " [" << CRing::name << "2] [composite]" << flush;
               }
            }
           if (mpz_probab_prime_p(n,probab_prime_checks)) goto fertig_phase2;
           if (mpz_cmp_ui(n,1)==0) goto fertig_phase2;
           // anschlieend noch Polynom vereinfachen:
           for (int j=0; j<Polynom_Index_i; ++j) mpz_mod(Polynom[j],Polynom[j],n);
          }
       }

    }

 fertig_phase2:
   cout << CRing::name << "-phase 2, up to " << ii << endl;
   for (int j=0; j<Polynom_Index; ++j) mpz_clear(sammel[j]);
   delete [] sammel;
   for (int j=0; j<Polynom_Index_i; ++j) mpz_clear(Polynom[j]);
   delete [] Polynom;
  }

 fertig:
  mpz_clear(x); mpz_clear(y);
  cout << CRing::name << " method finished..." << endl;
}

#endif
