#include "numbpart.cc" /* for calculation of partition numbers */

/*! @file
 * @brief
 * contains routines for parsing a numerical term and evaluating its value
 */


//! term parsing and numeric evaluation concepts
namespace parse_term
{

short signed int gZ_priority (const char ch)
{
  switch (ch)
   {
     case '+': case '-': return 1;
     case '*': case '/': case ':': return 2;
     case '^': return 3;
     default: return 0;
   } 
}

bool gZ_priority_le (const char a, const char b)
{
  return gZ_priority(a)<=gZ_priority(b);
}

bool gZ_priority_gr (const char a, const char b)
{
  return gZ_priority(a)>gZ_priority(b);
}

bool get_Zahl (mpz_t n, char* &s, const char op='R', const char upto_op='\0')
{ 
  /* Zeichenkette in multiple-precision-Zahl umwandeln.
     Die Zeichenkette kann ein beliebiger Term mit Klammern,
     und den Operationszeichen +,-,*,:,/,^ sein, solange er numerisch
     auswertbar ist.
  */
  
  while (s[0]==' ') s++; // ggf. Leerzeichen berlesen
  
  bool term_structure_ok = true;
  char* s_anf = s; // Start-Position des aktuellen Subterms
  char h;
  mpz_t tmp;
  mpz_init(tmp);

  if (s[0]=='(')
    {
      s++; // ffnende Klammer berspringen
      term_structure_ok=get_Zahl(tmp,s); // geklammerten Term holen
      if (s[0]==')') s++; // schlieende Klammer berspringen
      else 
	{
	  term_structure_ok=false;
	  cerr << "error: ')' expected at '" << s << "'" << endl; 
	}
    }
  else
   if (s[0]=='F' || s[0]=='L' || s[0]=='P' ||
       s[0]=='f' || s[0]=='p' || s[0]=='q') // Fibonacci-/Lucas-/Partition-Index folgt...
    {
      /*
         F - Fibonnaccizahl-Index
         L - Lascaszahl-Index
         P - Partitionszahl-Index
         f - ermittelt die Fakultt des (nachfolgenden) Arguments
         p - ermittelt die nachfolgende Primzahl (falls Argument nicht prim ist)
         q - testet, ob Argument Primzahl ist, wenn nein: Programmende
             (dies ist ntzlich fr einige Schleifen)
      */
      s++; // Funktionssymbol F, L, P bzw. f,p,q berspringen
      term_structure_ok=get_Zahl(tmp,s,s_anf[0]); // Term als Funktionsargument auswerten
    }
   else
    { // ungeklammerten Term holen
      char* anf=s;
      while (s[0]>='0' && s[0]<='9') s++;
      h=s[0]; s[0]='\0';
      if (mpz_set_str(tmp, anf, 10))
	{
	  term_structure_ok=false;
	  s[0]=h;
	  cerr << "error: number expected at '" << anf << "'" << endl;
	}
      s[0]=h;
    }
  
  while (s[0]==' ') s++; // Leerzeichen berlesen
  h=s[0]; // nchstes Operationszeichen
  
  if (term_structure_ok) switch (op)
    {
    case '\0':
      break;
    case 'R' :
      //cout << "Reset" << endl;
      mpz_set(n,tmp);
      break;
    case 'F' : // Fibonacci-Nummer
      {
       if (mpz_cmp_ui(tmp,100000)>0)
        {
         cerr << "fibonacci: argument to big: " << tmp << endl;
         exit(1);
        }
       if (mpz_sgn(tmp)<0)
        {
         cerr << "fibonacci: argument negative: " << tmp << endl;
         exit(1);
        }
       cout << "computing fibonacci number" << endl;
       mpz_fib_ui(n,mpz_get_ui(tmp));
      }
      break;
    case 'L' : // Lucas-Nummer
      {
       if (mpz_cmp_ui(tmp,100000)>0)
        {
         cerr << "lucas: argument too big: " << tmp << endl;
         exit(1);
        }
       if (mpz_sgn(tmp)<0)
        {
         cerr << "lucas: argument negative: " << tmp << endl;
         exit(1);
        }
       cout << "computing lucas number" << endl;
       mpz_lucnum_ui(n,mpz_get_ui(tmp));
      }
      break;
    case 'P' : // Partitionszahl
      {
       if (mpz_cmp_ui(tmp,1000000)>0)
        {
         cerr << "partition number: argument to big: " << tmp << endl;
         exit(1);
        }
       if (mpz_sgn(tmp)<0)
        {
         cerr << "partition number: argument negative: " << tmp << endl;
         exit(1);
        }
       cout << "computing partition number" << endl;
       numbpart::numbpart(n,mpz_get_ui(tmp));
      }
      break;
    case 'f' : // Fakultt von tmp ermitteln
      {
       if (mpz_cmp_ui(tmp,100000)>0)
        {
         cerr << "factorial: argument to big: " << tmp << endl;
         exit(1);
        }
       if (mpz_sgn(tmp)<0)
        {
         cerr << "factorial: argument negative: " << tmp << endl;
         exit(1);
        }
       cout << "computing factorial" << endl;
       mpz_fac_ui(n,mpz_get_ui(tmp));
      }
      break;
    case 'p' : // nachfolgende Primzahl von tmp ermitteln (sofern tmp nicht prim ist)
      {
        mpz_set(n,tmp);
        while (!mpz_probab_prime_p(n,50)) mpz_add_ui(n,n,1);
      }
      break;
    case 'q' : // beendet Programm, wenn tmp prim ist
      {
	cout << "special command q, testing argument..." << endl;
        mpz_set(n,tmp);
        if (!mpz_probab_prime_p(n,50))
         {
	   cout << "argument is not prime, aborting program." << endl;
           exit(0);
         }
      }
      break;
    case '+' :
      if (gZ_priority_gr(h,op))
	{
	  //cout << "Punkt vor Strich" << endl; 
	  term_structure_ok=get_Zahl(tmp,++s,h,'+'); h=s[0];
	}
      //cout << "Addition" << endl;
      mpz_add(n,n,tmp);
      break;
    case '-' :
      if (gZ_priority_gr(h,op))
	{
	  //cout << "Punkt vor Strich" << endl; 
	  term_structure_ok=get_Zahl(tmp,++s,h,'-'); h=s[0];
	}
      //cout << "Subtraktion" << endl;
      mpz_sub(n,n,tmp);
      break;
    case '*' : 
      if (gZ_priority_gr(h,op))
	{
	  //cout << "Exponentiation vor Multiplikation" << endl; 
	  term_structure_ok=get_Zahl(tmp,++s,h,'*'); h=s[0];
	}
      //cout << "Multiplikation" << endl;
      mpz_mul(n,n,tmp);
      break;
    case '/' :
    case ':' :
      {
        if (gZ_priority_gr(h,op))
  	  {
	    //cout << "Exponentiation vor Division" << endl; 
	    term_structure_ok=get_Zahl(tmp,++s,h,'/'); h=s[0];
	  }
        //cout << "Division" << endl;
        mpz_t q;
        mpz_init(q);
        mpz_div(q,n,tmp);
        mpz_mul(tmp,q,tmp);
        if (mpz_cmp(tmp,n)) // bleibt Rest?
         {
           cout << "Warning: result of division is truncated!" << endl;
         }
        mpz_set(n,q);
        mpz_clear(q);
      }
      break;
    case '^' :
      if (h=='^') cout << "Warnung: a^b^c ambiguent. Setze (a^b)^c." << endl; 
      // cout << "Exponentiation" << endl;
      mpz_pow_ui(n,n,mpz_get_ui(tmp));
      break;
    default :
      s=--s_anf;
      cerr << "error: operation +-*/:^ expected at '" << s << "'" << endl;
      term_structure_ok=false;
      break;
    }
  mpz_clear(tmp);

  if (term_structure_ok && h!='\0' && h!=')' )
   {
     // Term ist bisher in Ordnung,
     // es liegt kein Klammerungsende vor und ein Operator folgt.
     if (upto_op=='\0') return get_Zahl(n,++s,h);
     if (gZ_priority_le(h,upto_op)) return term_structure_ok;
     else return get_Zahl(n,++s,h,upto_op);
   }
  else return term_structure_ok;
}

} // namespace parse_term
