// compute number of partitions of n
// written by Thorsten Reinecke, 1999-12-06
// last change: 2003-12-20

/*! @file
 * @brief
 * In this file the computation of partition numbers takes place.
 * 
 * p(n) = number of unodered partitions of n
 *      = #solutions of x[1]+x[2]+...+x[n]=n,
 *      with x[i]>=x[i+1], x[n+1]=0, i=1..n
 * 
 * Reference:
 *  - J.H. van Lint, R.M. Wilson: "A Course in Combinatorics",
 *    Cambridge University Press, 1992, p.132ff
 */

// implementation is now threadsafe   

#include <iostream>
#include <gmp.h>

//! partition number concepts
namespace numbpart
{

inline int omega(const int m)
{
  return (3*m*m-m)/2;
} 

class Cnumbpart
{
 // this class can be utilized to provide access to a bunch of
 // (possible cached) partition numbers
private:
 const int size;
 typedef mpz_t* Pmpz_t;
 Pmpz_t *p;
 void numbpart_recurse(const int n);

public:
 Cnumbpart(const int n) : size(n+1)
  {
    p = new Pmpz_t [size];
    for (int i=0; i<size; ++i) p[i]=NULL;
  }
 ~Cnumbpart()
  {
    for (int i=0; i<size; ++i) 
     if (p[i]) { mpz_clear(*p[i]); delete p[i]; p[i]=NULL; }
    delete [] p;
  }
 void get_numbpart(mpz_t res, const int n)
 {
   using std::cerr;
   using std::endl;
   if (n<0) { cerr << "negative argument in numbpart" << endl; exit(1); }
   if (n>=size) { cerr << "argument overflows provided range in numbpart" << endl; exit(1); }
   // if numbpart(n) has not yet been computed, then compute it...
   numbpart_recurse(n);
   mpz_set(res,*p[n]);
 } 
};

void Cnumbpart::numbpart_recurse(const int n)
{
  if (p[n]) return; // value is cached
  if (n==0) { p[n]=new mpz_t[1]; mpz_init_set_si(*p[n],1); return; }

  p[n]=new mpz_t[1];
  mpz_init(*p[n]);
  int m,i;
  m=1;
  while ((i=n-omega(m))>=0)
   {
     numbpart_recurse(i);
     if (m%2) mpz_add(*p[n],*p[n],*p[i]); else mpz_sub(*p[n],*p[n],*p[i]);
     m++;
   }
  m=1;
  while ((i=n-omega(-m))>=0)
   {
     numbpart_recurse(i);
     if (m%2) mpz_add(*p[n],*p[n],*p[i]); else mpz_sub(*p[n],*p[n],*p[i]);
     m++;
   }
}


/*!
 * Compute number of unordered partitions of @p n and stores the result
 * in @p res. The calculation is threadsafe.
 */
void numbpart(mpz_t res, const int n)
{
  Cnumbpart capsule(n+1);
  capsule.get_numbpart(res,n);
}

}

#if 0
int main()
{
  using namespace std;
  mpz_t res;
  mpz_init(res);

  cout << "Testing numbpart-function:" << endl;
  for (unsigned int i=10000; i<=11000; i+=101)
   {
     numbpart::numbpart(res,i);
     cout << "p(" << i << ") = ";
     mpz_out_str(stdout,10,res);
     cout << endl;
   }

  cout << "Testing cached access:" << endl;
  numbpart::Cnumbpart capsule(20000);
  for (unsigned int i=10000; i<=11000; i+=101)
   {
     capsule.get_numbpart(res,i);
     cout << "p(" << i << ") = ";
     mpz_out_str(stdout,10,res);
     cout << endl;
   }

  mpz_clear(res);
}
#endif
