/*! @file
 * @brief
 * implementation of MPQS polynomials
 */


#include "mpqsPolynom.H"

void CmpqsPolynom::erstes_Polynom_ermitteln(const mpz_t fuer_kN, const int M)
{
  
  mpz_set(kN,fuer_kN);
  mpz_div_ui(kN_div2,kN,2); // kN/2 fr Berechnung negativer Modulowerte
  
  //mpz_t rkN; mpz_init(rkN); mpz_sqrt(rkN,kN);
  //const double W1=sqrt(2)/2;
  //const double W2=-1/(2*sqrt(2));
  //double A0=W1*mpz_get_d(rkN)/M;
  //double B0=0;
  //double C0=W2*M*mpz_get_d(rkN);
  //mpz_clear(rkN);
  //mpz_set_d(D,sqrt(A0)); // Initialwert fr Polynome

  // exakter:
  mpz_mul_ui(D,kN,2); mpz_sqrt(D,D); mpz_div_ui(D,D,2); mpz_div_ui(D,D,M); mpz_sqrt(D,D);
  // D ist der Initalwert fr das "optimale" Polynom
  
  // Sonderfall:
  //  Falls D (bei kleinen Zahlen) in der statischen Faktorbasis
  //  liegt, gibt es Probleme. Ebenso, falls D dynamischer Faktor ist.
  if (mpz_cmp_ui(D,Specialfactor_Sieb_Threshold)<=0)
   {
     if (Factor_Threshold<=1.0)
      {
        collecting_phase_beendet = true; // keine dynamischen Faktoren sammeln!
        cout << "Dynamische Faktoren deaktiviert." << endl;
      }
     else
      cout << "Polynom kollidiert eventuell mit dynamischer Faktorbasis..." << endl;
   }
  if (mpz_cmp_ui(D,biggest_Prime_in_Factorbase)<=0)
   {
     cout << "Polynom kollidiert mit statischer Faktorbasis... korrigiere dies..." << endl;
     mpz_set_ui(D,biggest_Prime_in_Factorbase+1);
     collecting_phase_beendet = true; // keine dynamischen Faktoren sammeln!
   }

  // nun zur eigentlichen Polynomberechnung:
  // Let A ~ D^2, D (probab. prime), (D/kN)=1, D=3 (mod 4)
  // fr D=3 (mod 4) geht die Wurzelermittlung einfacher,
  // es reicht aber D=1 (mod 2)
  // meine Implementation lt alle Flle zu, da fr D=5 (mod 8)
  // die Wurzelberechnung ebenfalls recht einfach zu implementieren ist
  // und fr D=1 (mod 4 bzw. 8) die Ermittlung mit Lucassequenzen mglich ist...

  while (!mpz_congruent_ui_p(D,1,2)) mpz_add_ui(D,D,1);

  naechstes_Polynom_ermitteln(); // hier folgt die weitere Berechnung
}

void CmpqsPolynom::naechstes_Polynom_ermitteln(const int step /* =0 */)
{
  mpz_add_ui(D,D,step*4); // step dient nur dazu, die Polynomintervalle
                          // fr die clients verteilen zu knnen

  // Let A ~ D^2, D (probab. prime), (D/kN)=1, D=3 (mod 4)
  // fr D=3 (mod 4) geht die Wurzelermittlung einfacher,
  // es reicht aber D=1 (mod 2)
  // meine Implementation lt alle Flle zu, da fr D=5 (mod 8)
  // die Wurzelberechnung ebenfalls recht einfach zu implementieren ist
  // und fr D=1 (mod 4 bzw. 8) die Ermittlung mit Lucassequenzen mglich ist...

#if 0 /* 1=alte Methode, 0=neue Methode zur Bestimmung des nchsten D */
  do mpz_add_ui(D,D,2);
  while ( (!mpz_probab_prime_p(D,probab_prime_checks)) || (mpz_jacobi(D,kN)!=1) );
#else /* neue Methode zur Bestimmung des nchsten D */
  {
    unsigned long int h0=mpz_remainder_ui(D,3*5*7*11*13*17*19);
    do
     {
       register unsigned long int h=h0;
       do h+=2; while ( (h%3==0) || (h%5==0) || (h%7==0) || (h%11==0)
                                 || (h%13==0) || (h%17==0) || (h%19==0) );
       mpz_add_ui(D,D,h-h0); h0=h;
       // nun hat D keine kleinen Teiler mehr
     } while ( (!mpz_probab_prime_p(D,probab_prime_checks)) || (mpz_jacobi(D,kN)!=1) );
    // D ist nun (probable) prim und kein quadratischer Rest modulo kN
  }
#endif

  mpz_mul(A,D,D); mpz_mul_ui(A2,A,2);

  if (mpz_congruent_ui_p(D,1,4)) // D=1 (mod 4)
   {
     // h1 = sqrt(kN) (mod D)
     mpz_sqrtmod(h1,kN,D);

     // h0 = h1/kN (mod D)
     mpz_invert(h0,kN,D); mpz_mul(h0,h0,h1); mpz_mod(h0,h0,D);
   }
  else // D=3 (mod 4)
   {
     // Bemerkung: dieser Fall knnte durch obigen Fall ebenfalls
     // bernommen werden, aber die Berechnung gestaltet sich
     // fr D=3 (mod 4) noch einfacher (Invertieren nicht erforderlich) 

     // h0 = (kN)^((D-3)/4) mod D
     mpz_sub_ui(h0,D,3); mpz_div_ui(h0,h0,4); mpz_powm(h0,kN,h0,D);
  
     // h1 = kN*h0 mod D  (=sqrt(kN) (mod D)) 
     mpz_mul(h1,kN,h0); mpz_mod(h1,h1,D);
   }

  // h2 = (2*h1)^-1 * ((kN-h1^2) /D) mod D
  mpz_mul(h2,h1,h1); mpz_sub(h2,kN,h2);
  
  mpz_t x; mpz_init(x);

  if (!mpz_divisible_p(h2,D))
   {
     MARK;
     mpz_mod(x,h2,D);
     cerr << "Fehler: Division mit Rest " << x << " !" << endl;
     exit(1);
   }
      
  mpz_divexact(h2,h2,D);
  mpz_mul_ui(x,h1,2); mpz_invert(x,x,D); mpz_mul(h2,x,h2);
  mpz_mod(h2,h2,D);
  
  // B = h1 + h2*D mod A
  mpz_mul(B,h2,D); mpz_add(B,B,h1); mpz_mod(B,B,A);
  if (mpz_mod_ui(x,B,2)==0) mpz_sub(B,A,B); 
  
  // C = (B^2-kN) / (4*A)
  mpz_mul(C,B,B); mpz_sub(C,C,kN); mpz_div(C,C,A); mpz_div_ui(C,C,4);

  mpz_clear(x);

#if 0
  // vollstndige (aber wohl unntige Ausgabe)
  cout << "Die Parameter des Polynoms A*x^2+B*x+C lauten: " << endl;
  cout << "A=" << A << endl;
  cout << "B=" << B << endl;
  cout << "C=" << C << endl;
  cout << "D=" << D << " mit A=D^2" << endl;
#else

#ifdef VERBOSE
  // kurze Ausgabe der wesentlichen Infos:
  cout << "Neues Polynom mit D=" << D << " ermittelt." << endl;
#endif

#if 0 || defined(PROFILE)
  static unsigned int count = 100;
  if (--count==0) exit(0); //  fr Profiling-Testzwecke
#endif

#endif
  
  mpz_mul_ui(D2_inv_mod_kN,D,2); mpz_invert(D2_inv_mod_kN,D2_inv_mod_kN,kN);
  //testen();
}

void CmpqsPolynom::testen(const signed int Siebpos /* =0 */)
{
  mpz_t x,y;
  mpz_init(x); mpz_init(y);

  mpz_set_si(h2,Siebpos);
  
  // Variante 1: ( (2*A*wert+B) / (2*D) )^2 mod kN
  mpz_mul_ui(x,A,2); mpz_mul(x,x,h2); mpz_add(x,x,B);
  //mpz_div(x,x,D); mpz_div_ui(x,x,2);
  mpz_mul(x,x,D2_inv_mod_kN); mpz_mod(x,x,kN);
  mpz_mul(x,x,x); mpz_mod(x,x,kN);
  if (mpz_cmp(x,kN_div2)>0) { mpz_sub(x,kN,x); mpz_neg(x,x); } // ggf. negativen Wert berechnen!
  
  // Variante 2: A*wert^2 + B*wert + C mod kN
  
  mpz_mul(h0,A,h2); mpz_mul(h0,h0,h2);
  mpz_mul(h1,B,h2);
  mpz_add(y,h0,h1); mpz_add(y,y,C); mpz_mod(y,y,kN);
  if (mpz_cmp(y,kN_div2)>0) { mpz_sub(y,kN,y); mpz_neg(y,y); } // ggf. negativen Wert berechnen!
  
  cout << "Test: Q(" << Siebpos << ")" << endl;
  cout << " =" << x << endl;
  cout << " =" << y << endl;

  if (mpz_cmp(x,y)!=0)
    {
      cerr << "Fehler: Werte sind verschieden!" << endl;
      exit(1);
    }
  mpz_clear(x); mpz_clear(y);
}

void CmpqsPolynom::Werte_holen(const signed int Siebpos, mpz_t Wurzelwert, mpz_t Q) const
{  // Variante 1: ( (2*A*wert+B) / (2*D) )^2 mod kN
  mpz_set_si(Wurzelwert,Siebpos);
  mpz_mul(Wurzelwert,Wurzelwert,A2);
  mpz_add(Wurzelwert,Wurzelwert,B);
  mpz_mul(Wurzelwert,Wurzelwert,D2_inv_mod_kN); mpz_mod(Wurzelwert,Wurzelwert,kN);
  
  mpz_mul(Q,Wurzelwert,Wurzelwert); mpz_mod(Q,Q,kN);
  if (mpz_cmp(Q,kN_div2)>0) mpz_sub(Q,Q,kN); // ggf. negativen Wert berechnen!
  
  //cout << "Werte fr " << Siebpos << " ";
  //mpz_out_str(stdout,10,Wurzelwert); cout << " -> ";
  //mpz_out_str(stdout,10,Q); cout << endl;
}


void CmpqsPolynom::Daten_sichern(ostream &ostr) // fr Recovery etc.
{
  // Daten sichern, die fr die Ermittlung des aktuellen Polynoms wichtig sind
  // D ist der Wert, der das aktuelle Polynom aus der Funktionsschar bestimmt.
  ostr << D << endl;
}

void CmpqsPolynom::Daten_laden(istream &in) // fr Recovery etc.
{
  // Daten laden, die fr die Ermittlung des aktuellen Polynoms wichtig sind 
  in >> D; in.ignore(1,'\n');
  mpz_sub_ui(D,D,4); // um vier vermindern
  naechstes_Polynom_ermitteln(); // und neu aufsetzen
}

void CmpqsPolynom::Daten_laden_wenn_vorhanden(istream &in) // fr Recovery etc.
{
  while (isspace(in.peek())) in.get(); // ignore all whitespaces
  if (in.peek()==EOF)
   {
     cerr << "no mpqs polynomial coefficient found. using start value" << endl;
     naechstes_Polynom_ermitteln(); // und neu aufsetzen
   }
  else Daten_laden(in);
}
