#ifndef mpqsPolynom_header
#define mpqsPolynom_header

/*! @file
 * @brief
 * header file for declaration of MPQS polynomials
 *
 * refer "The Multiple Polynomial Quadratic Sieve",
 *       Robert D. Silverman,
 *       Mathematics Of Computation, Vol. 48, January 1987, p. 329-339
 *   for mathematical background.
 */


class CmpqsPolynom
{
  /* Implementierung der Polynomberechnungen fr MPQS.
     Variablennotation hlt sich (einigermaen) an die Beschreibung
     aus 
     "The Multiple Polynomial Quadratic Sieve",
     Robert D. Silverman,
     Mathematics Of Computation, Vol. 48, January 1987, p. 329-339
  */  
private:
  int M;
  mpz_t A,B,C,D,h0,h1,h2;
  mpz_t kN, kN_div2, D2_inv_mod_kN;
  mpz_t A2;
public:
  inline CmpqsPolynom() // Konstruktur
    {
      mpz_init(A); mpz_init(B); mpz_init(C); mpz_init(D);
      mpz_init(h0); mpz_init(h1); mpz_init(h2);
      mpz_init(kN); mpz_init(kN_div2); mpz_init(D2_inv_mod_kN);
      mpz_init(A2);
    }
  inline ~CmpqsPolynom() // Destruktor
    {
      mpz_clear(A); mpz_clear(B); mpz_clear(C); mpz_clear(D);
      mpz_clear(h0); mpz_clear(h1); mpz_clear(h2);
      mpz_clear(kN); mpz_clear(kN_div2); mpz_clear(D2_inv_mod_kN);
      mpz_clear(A2);
    }
  void erstes_Polynom_ermitteln(const mpz_t fuer_kN, const int M);
  void naechstes_Polynom_ermitteln(const int step=0);
  void Werte_holen(const signed int Siebpos, mpz_t Wurzelwert, mpz_t Q) const;
  void testen(const signed int Siebpos=0);
  void Daten_sichern(ostream &ostr); // fr Recovery etc.
  void Daten_laden(istream &in); // fr Recovery etc.
  void Daten_laden_wenn_vorhanden(istream &in); // fr Recovery etc.
  inline bool operator < (const mpz_t K) const 
    {
      return mpz_cmp(D, K)<0;
    }

  // some helper functions to provide read-only access to internal parameters
  inline const mpz_t& get_A2() const { return A2; }
  inline const mpz_t& get_B() const { return B; }
};

#endif
