#ifndef FAKEHEAP_HAEDER_
#define FAKEHEAP_HEADER_

/*! @file
 * @brief
 * contains faked priority queue implementation designed to speedup DLP-MPQS
 */


#include <vector>
#include <algorithm>

/*
 Der Standard-Vektor wird um die typischen PriorityQueue-Funktionen erweitert.
 Es handelt sich aber nicht wirklich um eine PriorityQueue; die Schnittstellen
 dienen lediglich dazu, in bestimmten Programmstellen langsamere echte
 Priorityqueues durch dieses Fake zu ersetzen, sofern bestimmte
 Nebenbedingungen erfllt werden:

 Zugriff ist nur blockweise gestattet.
 Zwischen push-Operationen einerseits und top/pop-Operationen andererseits
 ist "sort" aufzurufen.
*/


//#define FAKEHEAP_DEBUG

/*!
 *  This class provides an interface much like a priority queue, and it is
 *  intended to be used in this manner but with some restrictions and
 *  extensions.
 *
 *  You can access elements of the FakeHeap like elements of a vector,
 *  because it is publicly inherited by one.
 *
 *  - Use push() to push elements to the FakeHeap.
 *  - Use pop() to pop elements from the FakeHeap.
 *  - You must use sort() between blockwise usage of push() and pop()
 *    to guarantee that the elements are accessed by their priority.
 *  - Use clear() to clear the FakeHeap.
 *
 *  @remark Of course you could use a simple vector for these operations,too,
 *          but then the interface would be different; and FakeHeap was designed
 *          to replace priority queues in a manner that you can easily switch
 *          between these abstract data types.
 *          (The only reason for the existence of FakeHeap is that it produces
 *          a practical speedup compared to priority queues.)
 */
template <class T> class FakeHeap : public vector<T>
{
private:

#ifdef FAKEHEAP_DEBUG
 mutable bool push_locked;
#endif

public:

#ifdef FAKEHEAP_DEBUG
 inline FakeHeap() : vector<T>(), push_locked(false) { cout << "HELLO!" << endl; };
 inline ~FakeHeap() { };
#endif

#ifdef FAKEHEAP_DEBUG
 inline void clear()
  {
    cout << "FakeHeap:: CLEAR..." << endl;
    vector<T>::clear();
    push_locked=false;
  }
#endif

 inline void sort(void)
  {
//    std::cout << "FakeHeap: sorting " << size() << " elements... " << std::flush;
    std::sort(this->begin(),this->end());
//    std::cout << "done." << std::endl;
#ifdef FAKEHEAP_DEBUG
    push_locked=true;
#endif
  }
 inline void push(const T &x)
  {
#ifdef FAKEHEAP_DEBUG
    if (push_locked)
     { 
       std::cerr << "FakeHeap:: Hey?!!! push is locked!!!" << endl;
       push_back(x);
       sort();
       return;
     }
#endif
    push_back(x);
  }
 inline void pop(void)
  {
#ifdef FAKEHEAP_DEBUG
    if (!push_locked)
     {
       std::cerr << "FakeHeap:: Hey?!!! pop is locked!!!" << endl;
       //exit(1);
     }
    push_locked=true;
#endif
    this->pop_back();
  }
 inline const T& top(void) const
  {
#ifdef FAKEHEAP_DEBUG
    if (!push_locked)
     {
       std::cerr << "FakeHeap:: Hey?!!! top is locked!!!" << endl;
       //exit(1);
     }
    push_locked=true;
#endif
    return this->back();
  } 
};

#endif
