// Implementation of elliptic curves method (ecm) V1.27
// written by Thorsten Reinecke, 1999-06-23
// last change: 2004-01-05

/*! @file
 * @brief
 * contains implementation of elliptic curves
 */


// Bemerkung:
// Die ursprngliche Fassung der Weierstrass-Form y^2=x^2+a*x+b
// wurde in dieser Version zugunsten der Montgomery-Form B*y^2=x^3+A*x^2+x
// modifiziert.

// soweit erste Tests ergeben, scheint das Modul zu laufen:

// Phase 1 wird nun in der Form (x::z) durchlaufen
// erst fr Phase 2 wird in die Form (x:y:1) konvertiert.

// Phase 2 (in improved continuation) ist nun deutlich schneller.
// Hier ist die Form (x:y:1) ideal, da die Punkte mit mit nur
// einer Subtraktion+Multiplikation aufgesammelt werden knnen und
// nur eine aufwendige Berechnung (gcd zum Faktortest + Berechnung
// der nchsten Intervallgrenze) pro Intervall anfllt.
// Die Form (x::z) wre hier aufwendiger.

// Phase 2 (fft continuation) ist asymptotisch schneller
// als die improved continuation. Das liegt daran, da die innerhalb
// der Suchintervalle liegenden Punkte als Nullstellen eines Polynoms
// interpretiert werden knnen. Wenn nun viele Suchintervalle berprft
// werden mssen, knnen asymptotisch schnelle Verfahren angewandt werden,
// die Polynome in mehreren Punkten zugleich auswerten.
// Die aktuelle Implementierung der fft-continuation ist noch nicht ausgereift,
// zeigt sich aber bereits jetzt fr grere Suchrume als berlegen.
// Nachteil: hoher Speicherbedarf


//#define DEBUGMODE /* funktioniert nur im LANGSAM-Modus ohne NRESIDUE! */
//#define LANGSAM /* in allen Phasen in der Form (x:y:1) rechnen */


// you may choose one (or none) of the following defines:
/* 
   Hint: up to 300 digits you should disable both,
         above 300 digits MODULAR_ARITHMETIC should be enabled
         (on 32 bit processors).
*/
         

//#define NRESIDUE /* use "modular_mult.cc" instead of mpz_mod (if possible) */
/* Sollte in der aktuellen Implementierung asymptotisch schneller sein als
   mod mit Trialdivision. Da aber das Restklassensystem ein anderes ist,
   gibt es bei einigen Operationen Kompatibilittsprobleme.
   Wenn man die Operation "redc" durch eine (ggf.) asymptotisch schlechtere,
   aber besser (maschinennhere) Implementierung ersetzt, drfte dies
   fr kleine Zahlen die schnellste Variante sein und (bei asymptotisch guter
   Implementierung) fr grere Zahlen hnlich schnell wie
   "MODULAR_ARITHMETIC" ablaufen.
*/
   

//#define MODULAR_ARITHMETIC /* use modular_arithmetic.cc instead of mpz_mod (if possible) */
/* Verwendung von reciprocal, d.h. "mpz_mod(a,b,n)" wird durch
   a=b-floor(b*(2^k/n)/2^k) ersetzt, wobei 2^k/n konstant bleibt,
   und 1/2^k durch Shiften realisiert wird.
   -> 2 Multiplikationen ersetzen 1 Division.
   Dies ist (bei Verwendung von Karatsuba oder noch schnelleren
   Multiplikationen) asymptotisch schneller als die in gmp-2.0.2 verwendete
   trialdivision. Scheint sich ab ca. 300 Dezimalstellen zu rentieren.
*/
  

#if defined(NRESIDUE) && defined (MODULAR_ARITHMETIC)
 #error "You must not define both: NRESIDUE and MODULAR_ARITHMETIC"
#endif

#if ! ( defined(NRESIDUE) || defined(MODULAR_ARITHMETIC) )
 // we use mpz_mod
 #define mpz_mulmod(res,a,b,n) { mpz_mul(res,a,b); mpz_mod(res,res,n); }
 #define modulo(res,T,n) mpz_mod(res,T,n)
#endif 

#ifdef NRESIDUE
 #include "modular_mult.cc"
 /* WICHTIG: 
     A) Alle Werte, mit denen gerechnet wird, vorher nach NRESIDUE konvertieren!
     B) Das Resultat einer Multiplikation mu durch ein "redc" wieder normiert
        werden, bevor es verwendet werden kann!
     C) redc niemals (ohne genaue Analyse) ohne vorherige Multiplikation verwenden!
     -> daher: fr MULTIPLIKATION am besten immer das MAKRO mpz_mulmod benutzen!
 */ 
 #define mpz_mulmod(res,a,b,n) { mpz_mul(res,a,b); NResidue.redc(res,res); }
 #define modulo(res,T,n) NResidue.redc(res,T)
#endif

#ifdef MODULAR_ARITHMETIC
 #include "modular_arithmetic.cc"
 #define mpz_mulmod(res,a,b,n) { mpz_mul(res,a,b); NResidue.mod(res,res); }
 #define modulo(res,T,n) NResidue.mod(res,T)
#endif 


#ifdef ECM_FFT_CONTINUATION
 #include "polynomial.cc"
 #include "fft_param.cc"
#endif

#if defined(DEBUGMODE) && (!defined(LANGSAM) || defined(NRESIDUE))
 #error "DEBUGMODE works only in LANGSAM-Mode without NRESIDUE!"
#endif

#include "mpz_multi_invert.cc"
#include "utils.H"


class TmpzPunkt : private ForbidAssignment
{
 public:
  mpz_t x,z;
  inline TmpzPunkt() { mpz_init(x); mpz_init(z); } // Konstruktor
  inline ~TmpzPunkt() { mpz_clear(x); mpz_init(z); } // Destruktor
};
typedef TmpzPunkt* PmpzPunkt;
typedef const TmpzPunkt* PconstmpzPunkt;


class elliptic_curves : private ForbidAssignment
{
 public:
  typedef long long int NATNUM;
 private:
  mpz_t a,b; // auer in go nur lesend verwendbar!
  mpz_t h,k,m; // allgemein verwendbar (zwischen Aufrufen vernderbar)!
  mpz_t x3,y3; // nur in add, mul2 verwendbar!
  mpz_t xh_mul, yh_mul; // nur in mul verwendbar!
  PmpzPunkt A,B,C,T1,T2; // fr den PRAC-Algorithmus in XZ_multiply
  int sigma; // ber sigma wird die Kurve ausgewhlt/definiert
  int phase; // fr statistische Zwecke: in welcher Phase befinden wir uns?
#ifdef NRESIDUE
  CN_Residue NResidue;
#endif
#ifdef MODULAR_ARITHMETIC
  CN_Residue NResidue;
#endif
 public:
  inline elliptic_curves() /* Konstruktor */
   : A(new TmpzPunkt), B(new TmpzPunkt), C(new TmpzPunkt),
     T1(new TmpzPunkt), T2(new TmpzPunkt) 
#ifdef NRESIDUE
  , NResidue()
 /* Konstruktor aufrufen,
    NResidue wird spter mit n initialisiert.
    n ist DIE globale Variable (die faktorisiert werden soll)
    zugegeben: es wre besser, n trotzdem als Parameter allen Klassen
    zu bergeben, aber ich will's jetzt nicht mehr ndern, da
    dann auch die Rckgabe der Faktoren und die Division n/FAKTOR
    durchgefhrt werden mssen... 
 */
#endif   
#ifdef MODULAR_ARITHMETIC
  , NResidue()
 /* Konstruktor aufrufen,
    NResidue wird spter mit n initialisiert.
    n ist DIE globale Variable (die faktorisiert werden soll)
    zugegeben: es wre besser, n trotzdem als Parameter allen Klassen
    zu bergeben, aber ich will's jetzt nicht mehr ndern, da
    dann auch die Rckgabe der Faktoren und die Division n/FAKTOR
    durchgefhrt werden mssen... 
 */
#endif   
    {
      phase=-1;
      mpz_init(a); mpz_init(b);
      mpz_init(h); mpz_init(k); mpz_init(m);
      mpz_init(x3); mpz_init(y3);
      mpz_init(xh_mul); mpz_init(yh_mul);
#ifdef NRESIDUE
      NResidue.init(n);
#endif
#ifdef MODULAR_ARITHMETIC
      NResidue.init(n);
#endif
    }
  inline ~elliptic_curves() /* Destruktor */
    {
      mpz_clear(a); mpz_clear(b);
      mpz_clear(h); mpz_clear(k); mpz_clear(m);
      mpz_clear(x3); mpz_clear(y3); 
      mpz_clear(xh_mul); mpz_clear(yh_mul);
      delete A; delete B; delete C; delete T1; delete T2;
    }
 private:
  void check_curve(const mpz_t x, const mpz_t y) const;
  void factor_found(mpz_t k) const;

  bool sub(mpz_t xr, mpz_t yr,
           const mpz_t x1, const mpz_t y1,
           const mpz_t x2, const mpz_t y2); // for single point
  bool add(mpz_t xr, mpz_t yr,
           const mpz_t x1, const mpz_t y1,
           const mpz_t x2, const mpz_t y2); // for single point
  bool mul2(mpz_t xr, mpz_t yr, const mpz_t x1, const mpz_t y1);
  bool mul(mpz_t xr, mpz_t yr,
           const mpz_t x1, const mpz_t y1,
           NATNUM K);
  bool add(mpz_t xr, mpz_t yr,
           const mpz_t x1, const mpz_t y1,
           const mpz_t x2, const mpz_t y2,
           mpz_t k);
  bool init_arithmetic_progression(mpz_t* const x, mpz_t* const y,
	   const mpz_t startx, const mpz_t starty, const NATNUM startpos,
           const unsigned int delta, const unsigned int grad);
  bool arithmetic_progression(mpz_t* const x, mpz_t* const y, const int anz);

  void XZ_mul2(mpz_t xr, mpz_t zr,
               const mpz_t x1, const mpz_t z1);
  inline void XZ_mul2(const PmpzPunkt R, const PconstmpzPunkt A)
    { XZ_mul2(R->x,R->z,A->x,A->z); }
  void XZ_mul2plus1(mpz_t xr, mpz_t zr,
                    const mpz_t Xp0, const mpz_t Zp0,
                    const mpz_t Xp1, const mpz_t Zp1,
                    const mpz_t x1, const mpz_t z1);
  inline void XZ_mul2plus1(const PmpzPunkt R, const PconstmpzPunkt A, const PconstmpzPunkt B, const PconstmpzPunkt C)
    { XZ_mul2plus1(R->x,R->z,A->x,A->z,B->x,B->z,C->x,C->z); }

  static const unsigned int cost_of_evaluating_lucas_chain(const NATNUM K, const double alpha);
  void XZ_multiply(mpz_t xr, mpz_t zr, const mpz_t x1, const mpz_t z1, NATNUM K);

 public:
  void go(const int ecm_sigma, const NATNUM phase1, const NATNUM phase2);

};     


void elliptic_curves::check_curve(const mpz_t x, const mpz_t y) const
{
  // prft, ob der Punkt auf der Kurve liegt, sonst: Fehlermeldung & Abbruch
  mpz_t h,x1,y1;
  mpz_init(x1); mpz_init(y1); mpz_init(h);
  mpz_powm_ui(x1,x,3,n); mpz_mul(h,a,x); mpz_mul(h,h,x); mpz_add(x1,x1,h); mpz_add(x1,x1,x);
  mpz_mod(x1,x1,n);
  mpz_powm_ui(y1,y,2,n); mpz_mul(y1,y1,b); mpz_mod(y1,y1,n);
  const int fl = mpz_cmp(x1,y1);
  mpz_clear(x1); mpz_clear(y1); mpz_clear(h);
  if (fl!=0)
   {
     cerr << "Geprfter Punkt liegt nicht auf der Kurve!" << endl;
     cerr << "(x:y:1)=(" << x << ":" << y << ":1)" << endl;
     exit(1);
   }
}

#if defined(IS_SERVER) || !defined(USE_NETWORK)

void elliptic_curves::factor_found(mpz_t k) const
{
  cout << endl << "factor found in elliptic curve" << endl;
  if (mpz_cmp(k,n)==0) // gefundener Faktor = n ?
   {
     cout << "Factorization is trivial, aborting..." << endl;
     return; // trivialer Teiler -> Verfahren abbrechen
   }

#ifdef IS_CLIENT
  ostream& Factorization_to_file = cout;
#endif

  if (mpz_probab_prime_p(k,probab_prime_checks))
    {
      const unsigned int exponent = mpz_remove(n,n,k);
      cout << k << " is factor." << endl;
      if (mpz_sizeinbase(k,10)<25)
       Factorization_to_file << MAL(k,exponent) << " [ecm" << phase << "]" << flush;
      else
       Factorization_to_file << MAL(k,exponent) << " [ecm" << phase << ",sigma=" << sigma << "]" << flush;
    }
  else
    {
      mpz_t h;
      mpz_init(h); const unsigned int exponent = mpz_remove(h,n,k);
      cout << k << " is a composite factor." << endl;
      if (mpz_probab_prime_p(n,probab_prime_checks))
       {
	 mpz_swap(n,k);
	 cout << k << " is factor. (factorswap)" << endl;
         Factorization_to_file << MAL() << k << " [ecm" << phase << "/factorswap,sigma=" << sigma << "]" << flush;
       }
      else
       {
	 mpz_set(n,h);
         if (mpz_sizeinbase(k,10)<25)
          Factorization_to_file << MAL(k,exponent) << " [ecm" << phase << "] [composite]" << flush;
         else
          Factorization_to_file << MAL(k,exponent) << " [ecm" << phase << ",sigma=" << sigma << "] [composite]" << flush;
       }
      mpz_clear(h);
    }
}

#else
// net-client-mode -> Faktor an den server bertragen

void elliptic_curves::factor_found(mpz_t k) const
{
  cout << endl << "factor found in elliptic curve, phase " << phase << endl;
  if (mpz_cmp(k,n)==0) // gefundener Faktor = n ?
   {
     cout << "Factorization is trival, aborting..." << endl;
     return; // trivialer Teiler -> Verfahren abbrechen
   }
  if (mpz_probab_prime_p(k,probab_prime_checks))
    {
      cout << k << " is factor." << endl;
    }
  else
    {
      cout << k << " is composite factor." << endl;
    }
  mpz_remove(n,n,k);

  unix_io_client communication_stream(communication_name, server_port);
  communication_stream << "Faktor(ECM)! ";
  communication_stream << sigma << " " << k << endl;

  cout << "Faktor an Server bertragen..." << endl;
}

#endif


void elliptic_curves::XZ_mul2(mpz_t xr, mpz_t zr,
                              const mpz_t x1, const mpz_t z1)
{
  // zu berechnen:
  // xr=(x^2-z^2)^2
  // zr=4*x*z*(x^2+A*x*z+z^2)
#if 1 // schnelle Version
  // xr=(x^2-z^2)^2=((x+z)*(x-z))^2=(x+z)^2*(x-z)^2
  // 4*x*z=(x+z)^2-(x-z)^2
  // zr/(4*x*z)=x^2+A*x*z+z^2 = (x-z)^2 + 2*x*z + a*x*z = (x-z)^2+(a+2)*x*z
  // und wenn wir b=(a+2)/4 setzen, gilt:
  // zr=4*x*z * ( (x-z)^2 + b*4*x*z )
  // ALSO WICHTIG: b=(a+2)/4 mu gegeben sein !!!
  
  mpz_add(h,x1,z1); mpz_mulmod(h,h,h,n); // h=(x1+z1)^2
  mpz_sub(k,x1,z1); mpz_mulmod(k,k,k,n); // k=(x1-z1)^2
  mpz_mulmod(xr,h,k,n); // xr=h*k
 
  mpz_sub(m,h,k); // m=h-k=4*x1*z1
  mpz_mulmod(h,b,m,n); // h=b*m
  mpz_add(k,k,h); // k=(x1-z1)^2+b*m
  mpz_mulmod(zr,k,m,n); // zr=4*x1*z1*k

#else// langsamere Version (getestet und funktioniert)
  // hier werden die Werte ohne Umformung ausgerechnet
  #ifdef NRESIDUE
   #error "funktioniert (wegen mpz_mul_ui(h,h,4)) nicht mit NRESIDUE!"
  #endif
  mpz_mul(h,x1,x1); modulo(h,h,n); // h=x1^2
  mpz_mul(k,z1,z1); modulo(k,k,n); // k=z1^2
  mpz_sub(m,h,k); // m=x1^2-z1^2
  mpz_add(h,h,k); // h=x1^2+z1^2
  mpz_mul(k,x1,z1); modulo(k,k,n); // k=x1*z1
  mpz_mul(m,m,m); modulo(xr,m,n); // xr=m^2
  mpz_mul(m,k,a); modulo(m,m,n); mpz_add(h,h,m); // h=k*a+x1^2+z1^2
  mpz_mul(h,h,k); mpz_mul_ui(h,h,4); modulo(zr,h,n); // zr=4*x1*z1*h
#endif
}
const int COST_XZ_mul2 = 5; // Kosten der obigen Operation an mpz_mul's

void elliptic_curves::XZ_mul2plus1(mpz_t xr, mpz_t zr,
                                   const mpz_t Xp0, const mpz_t Zp0,
                                   const mpz_t Xp1, const mpz_t Zp1,
                                   const mpz_t x1, const mpz_t z1)
{
#if 1 // schnelle Version (getestet und funktioniert)
  mpz_sub(h,Xp1,Zp1); mpz_add(m,Xp0,Zp0);
  mpz_mulmod(h,h,m,n); // h=(Xp1-Zp1)*(Xp0+Zp0)
  mpz_add(k,Xp1,Zp1); mpz_sub(m,Xp0,Zp0);
  mpz_mulmod(k,k,m,n); // k=(Xp1+Zp1)*(Xp0-Zp0)

  // im folgenden: m=(h+k)^2, h=(h-k)^2
  //               xr=z1*m,   zr=x1*h
  mpz_add(m,h,k); mpz_sub(h,h,k);
  mpz_mulmod(m,m,m,n); mpz_mul(m,m,z1); // kein mulmod wg. Parameterkonsistenz!
  mpz_mulmod(h,h,h,n); mpz_mul(h,h,x1);
  modulo(xr,m,n); // xr=z1*m
  modulo(zr,h,n); // zr=x1*h

#else // langsame Version (getestet und funktioniert)
  mpz_mul(h,Xp0,Xp1); modulo(h,h,n);
  mpz_mul(k,Zp0,Zp1); modulo(k,k,n);
  mpz_sub(h,h,k); mpz_mul(h,h,h); modulo(h,h,n);
  mpz_mul(m,h,z1);

  mpz_mul(h,Zp0,Xp1); modulo(h,h,n);
  mpz_mul(k,Xp0,Zp1); modulo(k,k,n);
  mpz_sub(h,h,k); mpz_mul(h,h,h); modulo(h,h,n);
  mpz_mul(h,h,x1); modulo(zr,h,n);

  modulo(xr,m,n);
#endif
}
const int COST_XZ_mul2plus1 = 6; // Kosten der obigen Operation an mpz_mul's 


#if 0 /* 1->binrer Algorithmus zur Multiplikation */
      /* 0->Montgomery's PRAC(tical) Algorithmus (ist schneller!) */
void elliptic_curves::XZ_multiply(mpz_t xr, mpz_t zr,
                                  const mpz_t x1, const mpz_t z1,
                                  NATNUM K)
{
#ifdef DEBUGMODE
  cout << "XZ_multiply " << K << " -> " << endl;
#endif
  
  bool s[100];
  signed int i=0;

  K>>=1;
  while (K>1)
   {
     s[i++]=(K&1);
     K>>=1;
   }
#ifdef DEBUGMODE  
  int value=1, valuep1=2;
#endif
  mpz_t Xp0,Zp0,Xp1,Zp1;
  mpz_init_set(Xp0,x1); mpz_init_set(Zp0,z1);
  mpz_init(Xp1); mpz_init(Zp1);
  XZ_mul2(Xp1,Zp1,x1,z1);

  while (--i>=0)
   {
     if (s[i])
      {
#ifdef DEBUGMODE
        value=value+valuep1; valuep1=valuep1+valuep1;
#endif
        XZ_mul2plus1(Xp0,Zp0,Xp0,Zp0,Xp1,Zp1,x1,z1);
        XZ_mul2(Xp1,Zp1,Xp1,Zp1);
      }
     else
      {
#ifdef DEBUGMODE
        valuep1=value+valuep1; value=value+value;
#endif
        XZ_mul2plus1(Xp1,Zp1,Xp0,Zp0,Xp1,Zp1,x1,z1);
        XZ_mul2(Xp0,Zp0,Xp0,Zp0);
      }
     //cout << i << ". " << s[i] << " " << value << "," << valuep1 << endl;
   }
  
  // wenn wir davon ausgehen, da K ungerade war, gilt nun:
  // Ergebnis = value+valuep1
#ifdef DEBUGMODE
  cout << "value=" << value+valuep1 << endl;
#endif
  XZ_mul2plus1(xr,zr,Xp0,Zp0,Xp1,Zp1,x1,z1);

  mpz_clear(Xp0); mpz_clear(Zp0);
  mpz_clear(Xp1); mpz_clear(Zp1);
}

#else

// zunchst zwei Vertauschungsfunktionen
inline void swap(PmpzPunkt &A, PmpzPunkt &B)
{
  PmpzPunkt H;
  H=A; A=B; B=H;
}

inline void swap(PmpzPunkt &A, PmpzPunkt &B, PmpzPunkt &C)
{
  PmpzPunkt H;
  H=A; A=B; B=C; C=H;
}


const unsigned int elliptic_curves::cost_of_evaluating_lucas_chain(const NATNUM K, const double alpha)
{
  // like PRAC, but this one only counts the addition steps for a
  // particular alpha

  const NATNUM r=static_cast<NATNUM>(rint(K/alpha));
  register unsigned int COST=0;

  //cout << "PRAC-init (COST)" << endl;
     
     COST+=COST_XZ_mul2;
     NATNUM d=K-r;
     NATNUM e=2*r-K;

     while (d!=e)
      {
        //cout << "inner loop: " << d << "," << e << endl;
        if (d<e)
         { 
           std::swap(e,d);
         }
        // now: d>=e

        if ( ((d+e)%3==0) && (d*4<=e*5) )
         {
           // condition 1
           const NATNUM d_new=(2*d-e)/3;
           e=(2*e-d)/3; d=d_new;
           COST+=3*COST_XZ_mul2plus1;
           continue;
         }
        if ( ((d-e)%6==0) && (d*4<=e*5) )
         {
           // condition 2
           d=(d-e)>>1;
           COST+=COST_XZ_mul2plus1 + COST_XZ_mul2;
           continue;
         }
        if (d<=4*e)
         {
           // condition 3
           d=d-e;
           COST+=COST_XZ_mul2plus1;
           continue;
         }
        if ( ((d-e)&1)==0 )
         {
           // condition 4
           d=(d-e)>>1;
           COST+=COST_XZ_mul2plus1 + COST_XZ_mul2;
           continue;
         }
        if ((d&1)==0)
         {
           // condition 5
           d>>=1;
           COST+=COST_XZ_mul2plus1 + COST_XZ_mul2;
           continue;
         }
        if (d%3==0)
         {
           // condition 6
           d=(d/3)-e;
           COST+=COST_XZ_mul2 + 3*COST_XZ_mul2plus1;
           continue;
         }
        if ((d+e)%3==0)
         {
           // condition 7
           d=(d-2*e)/3;
           COST+=COST_XZ_mul2 + 3*COST_XZ_mul2plus1;
           continue;
         }
        if ((d-e)%3==0)
         {
           // condition 8
           d=(d-e)/3;
           COST+=COST_XZ_mul2 + 3*COST_XZ_mul2plus1;
           continue;
         }
        if ((e&1)==0)
         {
           // condition 9
           e>>=1;
           COST+=COST_XZ_mul2plus1 + COST_XZ_mul2;
           continue;
         }
        cerr << "Error in PRAC-Algorithm" << endl;
        exit(1);
      }
     COST+=COST_XZ_mul2plus1;

  //cout << "PRAC-exit (COST)" << endl;
  return COST;
}
 
/* Peter L. Montgomery's PRAC-Algorithmus */
void elliptic_curves::XZ_multiply(mpz_t xr, mpz_t zr,
                                  const mpz_t x1, const mpz_t z1,
                                  NATNUM K)
{
#ifdef DEBUGMODE
  cout << "XZ_multiply (PRAC)" << K << " -> " << endl;
#endif
  
  static const double alpha_continued_fraction [25] = 
   {
     /* decimal value          ,   continued fraction form                  */
     1.618033988749894848204587, /* 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,... */
     1.723606797749978969640917, /* 1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,... */
     1.381966011250105151795413, /* 1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,... */
     1.580178728295464104708674, /* 1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,... */
     1.632839806088706285425954, /* 1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,... */
     1.612429949509495001921461, /* 1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,... */
     1.617214616534403862663845, /* 1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,... */
     1.620181980807415764823945, /* 1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,... */
     1.618347119656228057976350, /* 1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,... */
     1.617914406528817893862475, /* 1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,... */
     1.618079668469895811954673, /* 1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,... */
     1.618016541142025285987741, /* 1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,... */
     1.618040653214942998209496, /* 1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,... */
     1.618031443161248228921464, /* 1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,... */
     1.618034961079766208915596, /* 1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,... */
     1.618033617353155449748868, /* 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,... */
     1.618034130610858549698459, /* 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,... */
     1.618033934563833141806412, /* 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,... */
     1.618034009447129396675689, /* 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,... */
     1.618033980844254824942950, /* 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,... */
     1.618033991769580649023070, /* 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,... */
     1.618033987596477509789958, /* 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,... */
     1.618033989190461068579593, /* 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,... */
     1.618033988581613526362231, /* 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,... */
     1.618033988814172593483292, /* 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,... */
   };

  static unsigned int first_best [25] =
   {          1,        11,        23,       103,       173,
            367,       491,       673,      1433,      4871,
          11311,     25873,     67057,    173669,    401393,
        1039387,   2774557,   7121347,  18732257,  48650851,
      126814879, 333390199, 867971879,         0,         0 };

  unsigned int best_i = 0; // (initial) Index of alpha (with minimum cost)

//----------------------------------------------------------------------------
#if 0 /* only needed to get first_best values during implementation phase */
  {
    // searching for an alpha giving a short lucas chain...
    unsigned int best_cost=cost_of_evaluating_lucas_chain(K,alpha_continued_fraction[0]);
    for (unsigned int i=1; i<25; ++i)
     {
       const unsigned int cost=cost_of_evaluating_lucas_chain(K,alpha_continued_fraction[i]);
       if (cost<best_cost) { best_cost=cost; best_i=i; }
     }
   
    static unsigned int index = 0;
    if (best_i>index)
     {
       index=best_i;
       cout << endl << "K[" << best_i << "]=" << K << endl;
     }
    return;
    /* note:
      if you find a better order or better values for
      "alpha_continued_fraction" or "first_best" let me know...
      I have not spend much time to find these ones ;)
    */  
  }
#endif
//----------------------------------------------------------------------------


#if 1 /* following loop tries to optimize lucas chain, (may be left out) */
  {
    // searching for an alpha giving a short lucas chain...
    unsigned int best_cost=cost_of_evaluating_lucas_chain(K,alpha_continued_fraction[0]);
    for (unsigned int i=1; K>=first_best[i] && i<23; ++i)
     {
       const unsigned int cost=cost_of_evaluating_lucas_chain(K,alpha_continued_fraction[i]);
       if (cost<best_cost) { best_cost=cost; best_i=i; }
     }
  }
#endif

 
  mpz_set(A->x,x1); mpz_set(A->z,z1);
  //cout << "PRAC-init" << endl;

  const NATNUM r=static_cast<NATNUM>(rint(K/alpha_continued_fraction[best_i]));
  // if a better value for r(p) is known, use it instead
     
     mpz_set(B->x,A->x); mpz_set(B->z,A->z);
     mpz_set(C->x,A->x); mpz_set(C->z,A->z);
     XZ_mul2(A,A);
     NATNUM d=K-r;
     NATNUM e=2*r-K;

     while (d!=e)
      {
        //cout << "inner loop: " << d << "," << e << endl;
        if (d<e)
         { 
           std::swap(e,d);
           swap(A,B);
         }
        // now: d>=e

        if ( ((d+e)%3==0) && (d*4<=e*5) )
         {
           // condition 1
           const NATNUM d_new=(2*d-e)/3;
           e=(2*e-d)/3; d=d_new;
           XZ_mul2plus1(T1,A,B,C);
           XZ_mul2plus1(T2,T1,A,B);
           XZ_mul2plus1(B,B,T1,A);
           swap(A,T2);
           continue;
         }
        if ( ((d-e)%6==0) && (d*4<=e*5) )
         {
           // condition 2
           d=(d-e)>>1;
           XZ_mul2plus1(B,A,B,C);
           XZ_mul2(A,A);
           continue;
         }
        if (d<=4*e)
         {
           // condition 3
           d=d-e;
           XZ_mul2plus1(T1,B,A,C);
           swap(B,T1,C);
           continue;
         }
        if ( ((d-e)&1)==0 )
         {
           // condition 4
           d=(d-e)>>1;
           XZ_mul2plus1(B,B,A,C);
           XZ_mul2(A,A);
           continue;
         }
        if ((d&1)==0)
         {
           // condition 5
           d>>=1;
           XZ_mul2plus1(C,C,A,B);
           XZ_mul2(A,A);
           continue;
         }
        if (d%3==0)
         {
           // condition 6
           d=(d/3)-e;
           XZ_mul2(T1,A);
           XZ_mul2plus1(T2,A,B,C);
           XZ_mul2plus1(A,T1,A,A);
           XZ_mul2plus1(T1,T1,T2,C);
           swap(C,B,T1);
           continue;
         }
        if ((d+e)%3==0)
         {
           // condition 7
           d=(d-2*e)/3;
           XZ_mul2plus1(T1,A,B,C);
           XZ_mul2plus1(B,T1,A,B);
           XZ_mul2(T1,A); XZ_mul2plus1(A,A,T1,A);
           continue;
         }
        if ((d-e)%3==0)
         {
           // condition 8
           d=(d-e)/3;
           XZ_mul2plus1(T1,A,B,C);
           XZ_mul2plus1(C,C,A,B);
           swap(B,T1);
           XZ_mul2(T1,A); XZ_mul2plus1(A,A,T1,A);
           continue;
         }
        if ((e&1)==0)
         {
           // condition 9
           e>>=1;
           XZ_mul2plus1(C,C,B,A);
           XZ_mul2(B,B);
           continue;
         }
        cerr << "Error in PRAC-Algorithm" << endl;
        exit(1);
      }
     XZ_mul2plus1(A,A,B,C);

  //cout << "PRAC-exit" << endl;
  mpz_set(xr,A->x); mpz_set(zr,A->z);
}
#endif


bool elliptic_curves::mul2(mpz_t xr, mpz_t yr, const mpz_t x1, const mpz_t y1)
{
  // !!! IMPORTANT: Don't use NResidues in this function !!!
  mpz_mul_ui(k,y1,2); mpz_mul(k,k,b); mpz_mod(k,k,n);
  if (!mpz_invert(k,k,n))
    {
      mpz_mul_ui(k,y1,2); mpz_mul(k,k,b); mpz_mod(k,k,n); mpz_gcd(k,k,n);
      factor_found(k);
      return true; // mit dieser Kurve sind wir fertig
    }
  mpz_mul(m,x1,x1); mpz_mod(m,m,n); mpz_mul_ui(m,m,3);
  mpz_mul(h,a,x1); mpz_mul_ui(h,h,2); mpz_add(m,m,h); mpz_add_ui(m,m,1);
  mpz_mul(m,m,k); mpz_mod(m,m,n);
  mpz_mul(x3,m,m); mpz_mul(x3,x3,b); mpz_sub(x3,x3,a); mpz_sub(x3,x3,x1); mpz_sub(x3,x3,x1); mpz_mod(x3,x3,n);
  mpz_sub(y3,x1,x3); mpz_mul(y3,y3,m); mpz_sub(y3,y3,y1); mpz_mod(yr,y3,n);
  mpz_set(xr,x3);
#ifdef DEBUGMODE
  cout << "mul2 " << x << "," << y << endl;
  check_curve(xr,yr);
#endif
  return false; // kein Faktor gefunden
}

bool elliptic_curves::sub(mpz_t xr, mpz_t yr,
                          const mpz_t x1, const mpz_t y1,
                          const mpz_t x2, const mpz_t y2)
{
  bool flag;
  mpz_t y2_inv;

  mpz_init(y2_inv);
  mpz_neg(y2_inv, y2);
  mpz_mod(y2_inv, y2_inv, n);
  flag = add(xr,yr,x1,y1,x2,y2_inv);
  mpz_clear(y2_inv);
  return flag;
}


bool elliptic_curves::add(mpz_t xr, mpz_t yr,
                          const mpz_t x1, const mpz_t y1,
                          const mpz_t x2, const mpz_t y2)
{
  // !!! IMPORTANT: Don't use NResidues in this function !!!
  if (mpz_cmp(x1,x2)) // x1!=x2
   {
     //cout << "'";
     mpz_sub(k,x2,x1); mpz_mod(k,k,n);
     if (!mpz_invert(k,k,n))
      {
        mpz_sub(k,x2,x1); mpz_mod(k,k,n); mpz_gcd(k,k,n);
        factor_found(k);
        return true; // mit dieser Kurve sind wir fertig
      }
     mpz_sub(m,y2,y1); mpz_mul(m,m,k); mpz_mod(m,m,n);
     mpz_mul(x3,m,m); mpz_mul(x3,x3,b); mpz_sub(x3,x3,a); mpz_sub(x3,x3,x1); mpz_sub(x3,x3,x2); mpz_mod(x3,x3,n);
     mpz_sub(y3,x1,x3); mpz_mul(y3,y3,m); mpz_sub(y3,y3,y1); mpz_mod(y3,y3,n);
   }
  else // x1==x2
   {
     if (mpz_cmp(y1,y2)==0) // y1==y2
       {
         //cout << ".";
         mpz_mul_ui(k,y1,2); mpz_mul(k,k,b); mpz_mod(k,k,n);
         if (!mpz_invert(k,k,n))
          {
            mpz_mul_ui(k,y1,2); mpz_mul(k,k,b); mpz_mod(k,k,n); mpz_gcd(k,k,n);
            factor_found(k);
            return true; // mit dieser Kurve sind wir fertig
          }
	 mpz_mul(m,x1,x1); mpz_mod(m,m,n); mpz_mul_ui(m,m,3);
         mpz_mul(h,a,x1); mpz_mul_ui(h,h,2); mpz_add(m,m,h);
         mpz_add_ui(m,m,1);
         mpz_mul(m,m,k); mpz_mod(m,m,n);
         mpz_mul(x3,m,m); mpz_mul(x3,x3,b); mpz_sub(x3,x3,a); mpz_sub(x3,x3,x1); mpz_sub(x3,x3,x2); mpz_mod(x3,x3,n);
         mpz_sub(y3,x1,x3); mpz_mul(y3,y3,m); mpz_sub(y3,y3,y1); mpz_mod(y3,y3,n);
       }
      else // y1!=y2
       {
         cout << "Elliptic curves: Nicht implementierter Sonderfall" << endl;
         return true;
       } 
   }

  mpz_set(xr,x3); mpz_set(yr,y3);
#ifdef DEBUGMODE
  cout << "add" << endl;
  check_curve(xr,yr);
#endif
  return false; // kein Faktor gefunden
}

bool elliptic_curves::add(mpz_t xr, mpz_t yr,
                          const mpz_t x1, const mpz_t y1,
                          const mpz_t x2, const mpz_t y2,
                          mpz_t k)
{
  // Hinweis: dieses add ist analog zu der ursprnglichen Funktion,
  // aber in k wird das bereits invertierte Element bergeben, braucht
  // also nicht mehr berechnet zu werden.

  // !!! IMPORTANT: Don't use NResidues in this function !!!
  if (mpz_cmp(x1,x2)) // x1!=x2
   {
     //cout << "'";
     mpz_sub(m,y2,y1); mpz_mul(m,m,k); mpz_mod(m,m,n);
     mpz_mul(x3,m,m); mpz_mul(x3,x3,b); mpz_sub(x3,x3,a); mpz_sub(x3,x3,x1); mpz_sub(x3,x3,x2); mpz_mod(x3,x3,n);
     mpz_sub(y3,x1,x3); mpz_mul(y3,y3,m); mpz_sub(y3,y3,y1); mpz_mod(y3,y3,n);
   }
  else // x1==x2
   {
     if (mpz_cmp(y1,y2)==0) // y1==y2
       {
         //cout << ".";
         mpz_mul_ui(k,y1,2); mpz_mul(k,k,b); mpz_mod(k,k,n);
         if (!mpz_invert(k,k,n))
          {
            mpz_mul_ui(k,y1,2); mpz_mul(k,k,b); mpz_mod(k,k,n); mpz_gcd(k,k,n);
            factor_found(k);
            return true; // mit dieser Kurve sind wir fertig
          }
	 mpz_mul(m,x1,x1); mpz_mod(m,m,n); mpz_mul_ui(m,m,3);
         mpz_mul(h,a,x1); mpz_mul_ui(h,h,2); mpz_add(m,m,h);
         mpz_add_ui(m,m,1);
         mpz_mul(m,m,k); mpz_mod(m,m,n);
         mpz_mul(x3,m,m); mpz_mul(x3,x3,b); mpz_sub(x3,x3,a); mpz_sub(x3,x3,x1); mpz_sub(x3,x3,x2); mpz_mod(x3,x3,n);
         mpz_sub(y3,x1,x3); mpz_mul(y3,y3,m); mpz_sub(y3,y3,y1); mpz_mod(y3,y3,n);
       }
      else // y1!=y2
       {
         cout << "Elliptic curves: Nicht implementierter Sonderfall" << endl;
         return true;
       } 
   }

  mpz_set(xr,x3); mpz_set(yr,y3);
#ifdef DEBUGMODE
  cout << "add" << endl;
  check_curve(xr,yr);
#endif
  return false; // kein Faktor gefunden
}

bool elliptic_curves::init_arithmetic_progression(
  mpz_t* const x, mpz_t* const y, const mpz_t startx, const mpz_t starty,
  const NATNUM startpos, const unsigned int delta, const unsigned int grad)
{
  for (unsigned int i=0; i<=grad; i++)
   {
     NATNUM e = startpos + (i*delta);
     mpz_set(x[i], startx); mpz_set(y[i], starty);
     for (unsigned int j=0; j<grad; j++)
      if (mul(x[i],y[i],x[i],y[i],e)) return true;
   }
  for (unsigned int i=1; i<=grad; i++)
   {
     for (unsigned int j=grad; j>=i; j--)
      if (sub(x[j],y[j],x[j],y[j],x[j-1],y[j-1])) return true;
   }
  return false;
}

bool elliptic_curves::arithmetic_progression(mpz_t* const x, mpz_t* const y, const int anz)
{
 // addiert einen Vektor von Punkten nach folgendem Schema
 // for i=0 to anz-1 do (x[i],y[i]:1):=(x[i],y[i]:1)+(x[i+1]:y[i+1]:1); od
 // wird bentigt, um die Punkte fr Polynome in arithmetischer Progression
 // sukzessive auszuwerten.

#if 0
 // slow version
 bool flag=false;
 for (int i=0; i<anz && !flag; ++i)
   flag=add(x[i],y[i],x[i],y[i],x[i+1],y[i+1]);
 return flag;
#else
 // fast version
 // Trick: multi-invert benutzen
 mpz_t* const h = new mpz_t[anz];
 mpz_t* const r = new mpz_t[anz];
 bool flag=false;
 for (int i=0; i<anz; ++i)
  {
    mpz_init(h[i]); mpz_init(r[i]);
    mpz_sub(h[i],x[i+1],x[i]);
    if (mpz_cmp_ui(h[i],0)==0)
     {
       mpz_mul_ui(h[i],y[i],2); mpz_mul(h[i],h[i],b); mpz_mod(h[i],h[i],n);
     }
  }
 // nun die zu invertierenden Werte effizient invertieren
 if (mpz_multi_invert(r,h,anz,n))
  {
    // Inverse existieren -> der add-Routine mit bergeben
    for (int i=0; i<anz && !flag; ++i)
      flag=add(x[i],y[i],x[i],y[i],x[i+1],y[i+1],r[i]);
  }
 else
  {
    // mindestens ein Inverses existiert nicht -> normale add-Routine aufrufen
    for (int i=0; i<anz && !flag; ++i)
      flag=add(x[i],y[i],x[i],y[i],x[i+1],y[i+1]);
  }
 for (int i=0; i<anz; ++i)
  {
    mpz_clear(h[i]); mpz_clear(r[i]);
  }
 delete [] h; delete [] r;
 return flag;
#endif 
}

bool elliptic_curves::mul(mpz_t xr, mpz_t yr,
                          const mpz_t x1, const mpz_t y1,
                          NATNUM K)
{
  // !!! IMPORTANT: Don't use NResidues in this function !!!

  mpz_set(xh_mul,x1); mpz_set(yh_mul,y1);

  //kleines Problem: Addition mit neutralem Element mit add nicht mglich!
  //also wird folgendermaen verfahren:
#ifdef DEBUGMODE
  if (K==0) { cerr << "Multiplikation mit Null nicht erlaubt!"; exit(1); }
#endif
  while ((K&1)==0)
   {
     K>>=1;
     if (mul2(xh_mul,yh_mul,xh_mul,yh_mul)) return true; // Faktor gefunden
   }
  // nun gilt (K&1)==1
  mpz_set(xr,xh_mul); mpz_set(yr,yh_mul);
  while (K>>=1)
   {
     if (mul2(xh_mul,yh_mul,xh_mul,yh_mul)) return true; // Faktor gefunden
     if (K&1) if (add(xr,yr,xr,yr,xh_mul,yh_mul)) return true; // Faktor gefunden
   }
#ifdef DEBUGMODE
  cout << "mul" << endl;
  check_curve(xr,yr);
#endif
  return false; // kein Faktor gefunden
}


void elliptic_curves::go(const int ecm_sigma, const NATNUM phase1, const NATNUM phase2)
{
  using numtheory::is_prime;
  using numtheory::gcd;

  mpz_t x,y,z, saved_x,saved_y, xh,yh;
  mpz_init(x); mpz_init(y); mpz_init(z);
  mpz_init(saved_x); mpz_init(saved_y);
  mpz_init(xh); mpz_init(yh);
  sigma=ecm_sigma;

  cout << "--------------------------------------------------" << endl;

#if 1

  /*
     Nachfolgende Zeile aktivieren, falls das Sigma manuell
     eingegeben werden soll. (Kann eigentlich nur dann
     sinnvoll sein, wenn man zu Debuggingzwecken oder
     fr eine bekannte Zahl das Timing bestimmen will.)
  */
  //cout << "Sigma= "; cin >> sigma;

  mpz_set_ui(x,sigma); mpz_mul(x,x,x); mpz_sub_ui(x,x,5); mpz_mod(x,x,n);
  mpz_set_ui(y,sigma); mpz_mul_ui(y,y,4); mpz_mod(y,y,n);
  mpz_powm_ui(b,x,3,n); mpz_mul(b,b,y); mpz_mul_ui(b,b,4); mpz_mod(b,b,n);
  mpz_invert(a,b,n);
  mpz_sub(xh,y,x); mpz_powm_ui(xh,xh,3,n);
  mpz_mul_ui(yh,x,3); mpz_add(yh,yh,y); mpz_mul(xh,xh,yh);
  mpz_mul(a,a,xh); mpz_sub_ui(a,a,2); mpz_mod(a,a,n);

  mpz_powm_ui(x,x,3,n); mpz_powm_ui(z,y,3,n);

//#ifdef LANGSAM
  // konvertieren (nur erforderlich, falls in der Form (x:y:1)
  // gearbeitet werden soll, schadet aber auch sonst nicht viel)
  mpz_invert(z,z,n); mpz_mul(x,x,z); mpz_mod(x,x,n);
  mpz_set_ui(y,1);
  mpz_powm_ui(xh,x,3,n); mpz_powm_ui(yh,x,2,n); mpz_mod(yh,yh,a);
  mpz_add(xh,xh,yh); mpz_add(b,xh,x); mpz_mod(b,b,n);
  mpz_set_ui(z,1); // (x:y:z)=(x:1:1)
//#endif

#else
  cout << "Alternative Sigma-Methode" << endl;
  //cout << "Sigma= "; cin >> sigma;
  if (sigma==2) { cerr << "wrong sigma (must be <>2)!" << endl; return; }
  mpz_set_ui(a,sigma); mpz_set_ui(y,1);
  mpz_set_ui(x,2); 
  mpz_mul_ui(b,a,4); mpz_add_ui(b,b,10);
#endif

  cout << "Trying Elliptic Curve Method with sigma=" << sigma << endl;
#ifdef VERBOSE
  cout << "x=" << x << endl;
  cout << "y=" << y << endl;
  cout << "a=" << a << endl;
  cout << "b=" << b << endl;
#endif

#ifdef DEBUGMODE
  check_curve(x,y);
#endif

#ifndef LANGSAM
  // fr die XZ_mal2-Prozedur wird (in der schnellen Version)
  // b=(a+2)/4 erwartet!
  mpz_set_ui(h,4); mpz_invert(h,h,n);
  mpz_add_ui(b,a,2); mpz_mul(b,b,h); mpz_mod(b,b,n);
#endif

#if defined(NRESIDUE) && !defined(LANGSAM)
 // Die wesentlichen Parameter nach NResidue konvertieren
 NResidue.convert(a); NResidue.convert(b);
 NResidue.convert(x); NResidue.convert(y); NResidue.convert(z);
#endif

  NATNUM i;
  NATNUM j,k,d;
  // Phase 0: kleine Primzahlen (mit hohen Primzahlpotenzen)
  phase=0;
  cout << "Elliptic Curve Phase 0" << endl;
#ifdef LANGSAM
  for (j=1; j<=50; j++) if (mul2(x,y,x,y)) goto fertig; // 2er-Potenzen
  for (j=1; j<=25; j++) if (mul(x,y,x,y,3)) goto fertig; // 3er-Potenzen
#else
  for (j=1; j<=50; j++) XZ_mul2(x,z,x,z); // 2er-Potenzen
  for (j=1; j<=25; j++) 
    { 
      XZ_mul2(xh,yh,x,z);
      XZ_mul2plus1(x,z,xh,yh,x,z,x,z); // 3er-Potenzen
    }
#endif
  d=4; i=1; // +4+2 Folge nutzen, um 2,3 Teilbarkeit auszuschlieen
  while (i<500) // 750
   {
     do { i+=d; d=6-d; } while (!is_prime(i));
     j=phase2; k=1;
     do { k*=i; j/=i; } while (j>=i);
#ifdef LANGSAM
     if (mul(x,y,x,y,k)) goto fertig;
     if (mul(x,y,x,y,k)) goto fertig; // ruhig zweimal...
#else
     XZ_multiply(x,z,x,z,k);
     XZ_multiply(x,z,x,z,k); // ruhig zweimal...
#endif
   }

#ifndef LANGSAM
  mpz_gcd(h,z,n); if (mpz_cmp_ui(h,1)) { factor_found(h); goto fertig; }
#endif


#ifdef DEBUGMODE
  check_curve(x,y);
#endif

  {
   // Phase 1: Primzahlen (ohne/mit kleinen) Potenzen (aber knnen in beliebiger Kombination p-1 teilen)
   phase=1;
   const unsigned int D=2*3*5*7*11*13*2; // *17
   i=i-(i%D); // i mit Eigenschaft i=5 (mod 6) fr schnellere Primzahlermittlung
   cout << "Elliptic Curve Phase 1" << endl;
   while (i<=phase1)
    {
#ifdef REACT_ON_SIGUSR
      if (USRSignalHandler.got_SIGUSR1()) goto fertig;
      if (USRSignalHandler.got_SIGUSR2()) break;
#endif
      // Primzahlen aussieben
      bool sieve[D];
      sieve[1]=(i!=0); // dummer Spezialfall: 1 keine Primzahl!
      for (unsigned int j=3; j<=D; j+=2) sieve[j]=true; // Sieb initialisieren
      for (unsigned int p=5, dp=2; p*p<i+D; p+=dp, dp=6-dp)
       for (unsigned int j=p-(i%p); j<D; j+=p) sieve[j]=false;
       /* 
         Bemerkung:
          a) p*p<D+i ist als Primzahlbedingung fr das Intervall
             etwas zu streng, aber strt wegen Phase 0 nicht.
             (-> Primzahlen unter sqrt(D) werden nicht erkannt) 
          b) p-(i%p) ist zwar p, wenn p teilt i;
             aber: den Index 0 brauchen wir spter nicht!
       */
            
      for (unsigned int j=1, dj=4; j<D; j+=dj, dj=6-dj)
       {
         //if (sieve[j]!=is_prime(i+j))
         //  { cerr << "keine bereinstimmung: " << i+j << endl; }
         if (sieve[j])
           {
             NATNUM i2 = i+j;
             NATNUM j2=phase2, k=1;
             do { k*=i2; j2/=i2; } while (j2>=i2);
#ifdef LANGSAM
             if (mul(x,y,x,y,k)) goto fertig;
#else
             XZ_multiply(x,z,x,z,k);
#endif
             if (i2>=phase1) { i=i2; d=dj; break; }
           }
       }
      // Schleife vermeidet hufige gcd-Berechnung
      // in seltenen Fllen knnen aber mehrere Faktoren zusammenfallen
      // -> ECM versagt dann (ggf. Schleifendurchlufe vermindern)

#ifndef LANGSAM
      mpz_gcd(h,z,n); if (mpz_cmp_ui(h,1)) { factor_found(h); goto fertig; }
#endif

      if (i<=phase1) i+=D;
      cout << "Phase1: " << i << "/" << phase1 << "\r" << flush;
    }

   if (i%D==0)
    {
      i--; d=4; while (!is_prime(i)) { i-=d; d=6-d; } // i auf vorige Primzahl adjustieren
    }
   cout << "Elliptic Curve Phase 1, up to " << i << endl;
   if (i>=phase2) goto fertig;
  }


#ifndef LANGSAM
  // noch einmal auf Teiler testen...
  mpz_gcd(h,z,n); if (mpz_cmp_ui(h,1)) { factor_found(h); goto fertig; }
  mpz_gcd(h,x,n); if (mpz_cmp_ui(h,1)) { factor_found(h); goto fertig; }

#if defined(NRESIDUE) && !defined(LANGSAM)
 // Die wesentlichen Parameter von NResidue in die Normalform konvertieren
 NResidue.convert_back(a); NResidue.convert_back(b);
 NResidue.convert_back(x); NResidue.convert_back(y); NResidue.convert_back(z);
#endif

  // und nun Rckkonvertierung in die (x:y:1)-Form
  mpz_set_ui(y,1);
  mpz_invert(h,z,n); mpz_mul(x,x,h); mpz_mod(x,x,n); // x=x/z

  mpz_powm_ui(xh,x,3,n); mpz_powm_ui(yh,x,2,n); mpz_mul(yh,yh,a);
  mpz_add(xh,xh,yh); mpz_add(b,xh,x); mpz_mod(b,b,n);
  mpz_set_ui(z,1); // (x:y:z)=(x:1:1)

  check_curve(x,y);
#endif

  phase=2; // for statistical reasons: we are now in continuation phase (phase 2)

#ifndef ECM_FFT_CONTINUATION

/* IMPORTANT NOTE:
   In phase 2 never use NResidues for calculating points on the curve.
   But it should be worthwhile - if NRESIDUE is defined - to work with
   "bfact" (and a transformed copy of "x" named "saved_x") as NResidues
   to speed up the modular multiplications with "sammel".
*/

  {
   // Phase 2: isolierte (Prim-)zahl: Kurve mu bis auf diese Zahl vorher vollstndig zerlegbar gewesen sein.
   cout << "Elliptic Curve Phase 2 (improved)" << endl;
   mpz_set(saved_x,x); mpz_set(saved_y,y); // Basis fr Phase2

   // precomputing most common used differences:
   // pre_mul_size ist die Intervallgre, in der auch gesiebt wird
   unsigned int pre_mul_size = 2*3*5; // 2*3 ist mindestens erforderlich!
   {
     // Je mehr verschiedene, kleine Primfaktoren enthalten sind, desto
     // weniger teilerfremde Zahlen gibt es im Intervall -> desto geringer
     // sind also die Speicherkosten.
     // Ideale Intervallgre wre etwa sqrt(phase2), wenn man die Kosten
     // fr die Ermittlung der Primzahlen, usw. vernachlssigt.
     unsigned int p=5;
     do
      { 
        do p+=2; while (!is_prime(p));
        pre_mul_size*=p;
      } while (pre_mul_size<=phase2/pre_mul_size);

     if (pre_mul_size>3*sqrt(phase2)) // Schwellwert-Heuristik
      {
        pre_mul_size/=p; // grter Primfaktor war zu gro -> abdividieren
        // stattdessen noch bis zur optimalen Gre mit Zusatzfaktor auffllen:
        pre_mul_size*=static_cast<unsigned int>(ceil(sqrt(phase2)/pre_mul_size));
      }
     cout << "pre_mul_size=" << pre_mul_size << endl;
   }

   mpz_t* const bfact = new mpz_t[pre_mul_size];
    // hier wird der zu einer Differenz gehrige x-Kurvenwert abgelegt

   /* Im folgenden wird die Tabelle bfact aufgebaut, die von dem erreichten
      Startpunkt P=(saved_x:saved_y) die Punkte k*P fr 0<=k<pre_mul_size
      enthlt. Genauer: bfact[k]=X-Wert von k*P (den Y-Wert bentigen wir
      fr sptere Zugriffe nicht mehr).
      Die Tabelle wir spter bentigt, um in pre_mul_size-Schritten auf der
      elliptischen Kurve voranspringen zu knnen. Einen Punkt m*P knnen
      wir dann erreichen, in dem wir m1=floor(m/pre_mul_size), m2=m%premulsize
      setzen und m*P=(m1*premulsize)*P+m2*P berechnen.
      Fr einzelne Wertberechnungen bringt uns diese Form nichts, aber fr
      viele sukzessive Werte reicht es dann, den Punkt Q=premulsize*P
      zu berechnen, womit m*P=m1*Q+m2*P gilt.
      Wir haben nun erreicht, da
        1. m2*P =:R[m2] fr 0<=m2<pre_mul_size ber Tabellenzugriffe geholt
           werden kann. -> "kostenlos"
        2. m1*premulsize*P=m1*Q =:S ist fr Werte innerhalb eines Intervalls
           der Gre premulsize ebenfalls konstant, solange m1 sich nicht
           ndert!
       -> Die Kosten reduzieren sich innerhalb eines Intervalls auf
          die Kosten der Addition von S+R[m2].
     
      Da wir in der improved-standard-continuation fr k lediglich alle
      aufeinanderfolgenden Primzahlen (die grer sind als der letzte Wert
      aus Phase 1) durchlaufen wollen, knnen wir fr premulsize das
      Produkt kleiner Primzahlen whlen und brauchen fr m2 nur die zu
      premulsize teilerfremden Zahlen zu nehmen.
      Um diese Punkte R[m2] an diesen Punkten zu berechnen, empfiehlt es sich
      fr groes premulsize, die Punkte hnlich der standart-continuation
      zu ermitteln; d.h. in Differenzen voranzuspringen und die Punkte
      ber Additionen der (vorberechneten) "Differenzpunkte" zu bestimmen.

      Der vorangegangene Kommentar beschreibt zwar nicht exakt unsere
      tatschliche Vorgehensweise, aber er umschreibt die dahinterliegende
      "Idee", die umgesetzt wird...
   */
      
   const int table_dxy_size = 23;
   mpz_t table_dxy[table_dxy_size][2];

   if (mul2(x,y,saved_x,saved_y)) goto fertig; // Primzahldifferenzen sind gerade (fr p>3)
   mpz_set(xh,x); mpz_set(yh,y); // (x,y) nach (xh,yh) sichern
 
   mpz_init_set(table_dxy[2][0],x);
   mpz_init_set(table_dxy[2][1],y);
   for (int j=4; j<table_dxy_size; j+=2) 
    {
      if (add(x,y,x,y,xh,yh)) goto fertig;
      mpz_init_set(table_dxy[j][0],x);
      mpz_init_set(table_dxy[j][1],y);
    }

   mpz_set(x,xh); mpz_set(y,yh); // (x,y) zurckholen
   for (int j=pre_mul_size-1, jlast=pre_mul_size-1; j>=0; j-=2)
    {
      if (gcd(j,pre_mul_size)==1) // j teilerfremd?
       {
         /* 
            aktuellen Punkt (premulsize-1-j)*(xh,yh) berechnen und
            die X-Ordinate in bfact[j] ablegen
         */
         int dj=jlast-j;
         if (dj>=table_dxy_size)
          { 
            cerr << "increase table_dxy_size to " << dj+1 << "!" << endl;
            exit(1);
          }
         if (dj>0) // Sonderfall: fr dj=0 nicht addieren
          if (add(x,y,x,y,table_dxy[dj][0],table_dxy[dj][1])) goto fertig;
         jlast=j;
         mpz_init_set(bfact[j],x);
#ifdef NRESIDUE
         NResidue.convert(bfact[j]);
#endif
       }
    }

   for (int j=2; j<table_dxy_size; j+=2) // Tabelle der Differenzpunkte lschen
    {
      mpz_clear(table_dxy[j][0]);
      mpz_clear(table_dxy[j][1]);
    }


   NATNUM next_i;
   mpz_t sammel;
   mpz_init(sammel); mpz_set_ui(sammel,1);

   // vielleicht etwas umstndlich, aber dafr verstndlich:
   if (mul(xh,yh,saved_x,saved_y,pre_mul_size)) goto fertig; // Intervall-Summand
   // (xh,yh)=premulsize*(saved_x,saved_y) ist nun der Intervallsummand

   if (i<pre_mul_size) 
     { 
       /* phase1<pre_mul_size,
          d.h. in der nachfolgenden Routine treten negative Werte auf,
          und das ergibt keinen Sinn.
          Normalerweise sollte ein kleiner Teil des berlappenden
          Restes von Phase 1 und dem Start von Phase 2 nicht stren.
          Wie auch immer, durch Erzeugung der bfact-Tabelle sollten
          alle Primzahlen von 1 bis pre_mul_size bereits ebenfalls
          abgetestet sein...
          Nun wird jedenfalls der Start der Phase 2 ein Intervall hher
          angesetzt.
        */
       cout << "Phase 1 ziemlich niedrig..." << endl;
       cout << "setze Start fr Phase 2 hher an..." << endl;
       i+=pre_mul_size;
     }
   i=((i/pre_mul_size)*pre_mul_size)-1; d=2; // i mit Eigenschaft i=5 (mod 6) fr schnellere Primzahlermittlung
   if (mul(x,y,saved_x,saved_y,i)) goto fertig; // Startwert

   next_i = i+1+pre_mul_size; // Intervallgrenze setzen

   bool* const sieve = new bool[pre_mul_size];
   for (int i1=pre_mul_size-1; i1>=0; i1-=2) sieve[i1]=false;

   for (unsigned int p=5, dp=2; p<pre_mul_size && p*p<next_i; p+=dp, dp=6-dp )
    for (unsigned int i1=p-((next_i-pre_mul_size)%p); i1<pre_mul_size; i1+=p) sieve[i1]=true;

   while (i<=phase2) 
    { 
      int runden=25000; // alle "runden" Runden mit gcd auf Faktor testen
      while (runden>0)
	{
          // nchste Primzahl ermitteln...
          do { i+=d; d=6-d; } while (i<next_i && sieve[i%pre_mul_size]);

         /* falls pre_mul_size gro genug gewhlt wurde, wre statt
            "while"-Schleife auch eine "if"-Bedingung ausreichend... */
         while (i>next_i) // neues Intervall berechnen?
          { 
            next_i += pre_mul_size;
            if (add(x,y,x,y,xh,yh)) goto fertig_phase2;
#ifdef NRESIDUE
            mpz_set(saved_x,x);
            NResidue.convert(saved_x);
#endif
            for (int i1=pre_mul_size-1; i1>=0; i1-=2) sieve[i1]=false;
            for (unsigned int p=5, dp=2; p<pre_mul_size && p*p<next_i; p+=dp, dp=6-dp)
             for (unsigned int i1=p-((next_i-pre_mul_size)%p); i1<pre_mul_size; i1+=p) sieve[i1]=true;
          }

          // nun in "sammel" den mglicherweise gefundenen Faktor sammeln... 
#ifdef NRESIDUE
          mpz_sub(h,saved_x,bfact[next_i-i]);
#else
          mpz_sub(h,x,bfact[next_i-i]);
#endif        
          mpz_mul(sammel,sammel,h); modulo(sammel,sammel,n);
	  --runden;
	}

      /* Bemerkung:
         Wenn wir das in "sammel" gebildete Produkt als Auswertung eines
         Polynoms (x-c1)*(x-c2)*...*(x-ck) auffassen, dann bietet es sich
         an, eine Polynomauswertung zu implementieren, die schneller
         arbeitet als die k-fache Multiplikation.
         (-> Siehe nachfolgend implementierte fft-continuation,
             die alternativ aktiviert werden kann.)
      */

      // und erst nach "runden"-vielen Durchlufen die aufgesammelten Zahlen
      // auf grten gemeinsamen Teiler testen...
      mpz_gcd(sammel,sammel,n);
      if (mpz_cmp_ui(sammel,1)) { factor_found(sammel); goto fertig_phase2; }
      mpz_set_ui(sammel,1); // sptere Multiplikation erleichtern...
      cout << "Phase2: " << i << "/" << phase2 << "\r" << flush;
    }

 fertig_phase2:
  mpz_clear(sammel);
  cout << "Elliptic Curve Phase 2, up to " << i << endl;
   for (int j=pre_mul_size-1; j>=0; j-=2)
    {
      // freeing memory for precomputed differences
      // !! siehe auch die Allokierung!
      if (gcd(j,pre_mul_size)==1) mpz_clear(bfact[j]);
    }
   delete [] sieve;
   delete [] bfact;
  }

// end of ecm improved standard continuation

#else
 //---------------------------------------------------------------
 // ecm fft-continuation
 //---------------------------------------------------------------

/* Dies ist eine fr fft-continuation umgemodelte Version der vorangegangenen
   improved standard continuation. fft bedeutet in diesem Zusammhang, da
   multipoint polynomial evaluation verwendet wird. Fr die konkrete
   Implementierung siehe das Modul "polynomial.cc".  Die eigentliche
   Optimierungsarbeit mu in dem oben genannten Modul vorgenommen werden.
*/


/* IMPORTANT NOTE:
   In phase 2 (fft) never use NResidues for calculating points on the curve.
*/

  {
   // Phase 2: isolierte (Prim-)zahl: Kurve mu bis auf diese Zahl vorher vollstndig zerlegbar gewesen sein.
   cout << "Elliptic Curve Phase 2 (improved with pairing & fft)" << endl;
   mpz_set(saved_x,x); mpz_set(saved_y,y); // Basis fr Phase2

   // precomputing most common used differences:
   // pre_mul_size ist die Intervallgre mit der pro Polynom gesucht wird
   // -> siehe fft_param.cc
   unsigned int pg=256, pms=2*1050;
   get_fft_parameter(pg,pms,phase2,n); // Parameter (pg,pms) fr "phase2" holen

   /* Am Rande:
      Die Schleifen wurden von "int" auf "NATNUM" umgestellt, so dass
      der Datentyp jetzt in Grenzen whlbar ist.
      Mit "long long" kann man dann auch bei 32-Bit-Rechnern bis zur
      Speichergrenze gehen. Bei 64-Bit-Rechnern drfte "int" statt "long long"
      dann wieder gengend gro sein.

      Bei 1 GB main memory und pg=262144 geht dem Rechner bereits der
      Speicher aus. (Der Speicherbedarf hngt natrlich auch von der Gre
      der zu faktorisierenden Zahl ab.) Wenn der Rechner nur im Hintergrund
      faktorisieren soll und Memory geswapt wird, ist das kontraproduktiv.
      -> Die Werte sind also individuell zu tunen.
   */

   const unsigned int pre_mul_size = pms;
   const unsigned int Polynom_Index = pg;

   //cout << "size of polynomial: " << Polynom_Index << endl;
   //cout << "single interval size:" << pre_mul_size << endl;

   mpz_t* const  sammel = new mpz_t[Polynom_Index];
   for (unsigned int j=0; j<Polynom_Index; ++j) mpz_init(sammel[j]);

   /*
      Fr den Aufbau des Polynoms:
      Nullstellen des Polynoms entsprechen in etwa der bfact-Tabelle der
      improved standard continuation, siehe auch dortige Kommentare.

      (k+d)^2 = k^2 + r mit r=d^2+2*k*d 

      setze d=2 und starte mit k=1, d=2, r=4+2*1*2=8
      wir erhalten:
       ... (k+2)^2=k^2+r=1+8=9, r=r+2*d^2=8+8=16,
       ... (k+2)^2=k^2+r=9+16=25, r=r+2*d^2=16+8=24,
       ... (k+2)^2=k^2+r=25+24=49, r=r+2*d^2=24+8=32,
       usw.

       Es wird nun eine Tabelle von Punkten der Form (k+d)^2 * P gebildet.
       Die Punkte werden in arithmetischer Progression sukzessive ermittelt.
       (xd[0]:yd[0]:0) ist dabei der jeweils gewnschte Punkt,
       (xd[i]:yd[i]:0), i=1..2 sind Hilfspunkte fr die arithmetische
       Progression zu berechnen.
       (vgl. Kapitel 5.9 in der Dissertation von Peter L. Montgomery)
   */

   const int d_exp = 12; // geeignete Werte sind 2, 6, 12, 24, empfohlen: 12
   mpz_t xd[d_exp+1], yd[d_exp+1];
   for (int j=0; j<=d_exp; ++j) { mpz_init(xd[j]); mpz_init(yd[j]); }
   if (init_arithmetic_progression(xd, yd, saved_x, saved_y, 1, 2, d_exp)) goto fertig;

   unsigned int Polynom_Index_i = 0;
   for (unsigned int j=1; j<pre_mul_size/2; j+=2)
    {
      if (gcd(j,pre_mul_size)==1) // j teilerfremd?
       {
         /* 
            aktueller Punkt x,y=j^2*(saved_x,saved_y),
            die X-Ordinate fr Polynomnullstelle ablegen,
         */
         //cout << j << "/" << pre_mul_size/2 << endl;
         mpz_set(sammel[Polynom_Index_i++],xd[0]);
       }
      if (arithmetic_progression(xd,yd,d_exp)) goto fertig;
    }

   cout << "original size of polynomial: " << Polynom_Index_i << endl;

   polynomial::TPolynom Polynom = NULL; // hier steht das anschlieend konstruierte Polynom drin
 
   // Rest bis zur nchsten Zweierpotenz mit Nullen auffllen
   /* Bemerkung: Besser wre es, Punkte auerhalb des Suchbereichs
                 zu whlen (Motto: doppeltes, aber auerhalb des Suchbereichs
                 lchriges Netz)
    */
   { 
     unsigned int anz_Punkte=Polynom_Index_i; 
     while (anz_Punkte<Polynom_Index) mpz_set_ui(sammel[anz_Punkte++],0);
     Polynom_Index_i=polynomial::Polynom_aus_Nullstellen_konstruieren(Polynom,sammel,anz_Punkte,n);
     cout << "polynomial created. size of polynomial = " << Polynom_Index_i << endl;
   }
   

   i=i-(i%pre_mul_size); // i ist nun Vielfaches von pre_mul_size
   // folgende Konvention:
   //  i zeigt (auer in der inneren Berechnungsschleife) immer auf den Start
   // des Intervalls [i,i+pre_mul_size].
   // In der inneren Berechnungsschleife zeigt i hingegen auf die Mitte
   // des Intervalls. 

   // Intervallhochzhler beim Pairing initialisieren:
   if (init_arithmetic_progression(xd, yd, saved_x, saved_y, i+pre_mul_size/2, pre_mul_size, d_exp)) goto fertig_phase2;

   while (i<=phase2)
    { 

      cout << "Phase2: " << i << "/" << phase2 << endl << flush;
      cout << "Computing and collecting values..." << endl;

      i+=pre_mul_size/2; // i auf Intervallmitte setzen
      //Werte sammeln
      unsigned int sammel_index=0;
      while (sammel_index<(Polynom_Index_i-1)/2)
       {
         // nun (xd[0]:yd[0]:1)=i^2*(saved_x:saved_y:1)
         mpz_set(sammel[sammel_index++],xd[0]);
         // jetzt schon den nchsten Punkt bestimmen...
         if (arithmetic_progression(xd,yd,d_exp)) goto fertig_phase2;
         i += pre_mul_size; // und neues Intervall setzen
       }
      i-=pre_mul_size/2; // i auf Intervallstart setzen

#ifdef REACT_ON_SIGUSR
      if (USRSignalHandler.got_SIGUSR1()) goto fertig_phase2;
      if (USRSignalHandler.got_SIGUSR2()) continue;
#endif

      // Polynomauswertung starten
      cout << "Phase2: " << i << "/" << phase2 << endl << flush;
      cout << "starting multipoint polynomial evaluation" << endl;
      cout << "Parameter: polynomaial size: " << Polynom_Index_i
           << ", points to evaluate: " << sammel_index << endl;
      polynomial::multipoint_eval(sammel,Polynom,Polynom_Index_i,sammel,sammel_index,n);
      cout << "polynomial evaluation finished, computing gcd." << endl;
      mpz_set(h,sammel[0]);
      for (unsigned int j=1; j<sammel_index; ++j) { mpz_mul(h,h,sammel[j]); mpz_mod(h,h,n); }
      mpz_gcd(h,h,n);
      /* Hinweis: Backtracking ist mglich, falls
                  Mehrfachfaktor entdeckt werden sollte:
                  a) gcd(sammel[j],n) getrennt berechnen (geschieht nunmehr)
                  b) Polynomnullstellen einzeln untersuchen, falls auch dies scheitert.
                     (dies geschieht nicht)
      */
      if (mpz_cmp_ui(h,1)) // Faktor gefunden?
        {
          if (mpz_probab_prime_p(h,probab_prime_checks))
            { factor_found(h); goto fertig_phase2; }
          else // versuchen, zusammengesetzte Faktoren zu trennen
            {
              for (unsigned int j=0; j<sammel_index; ++j)
               {
                 mpz_gcd(h,sammel[j],n);
                 if (mpz_cmp_ui(h,1)) factor_found(h);
               }
              goto fertig_phase2; // diese Kurve ist ausgelutscht...
            }
        }

      /* Bemerkung:
         Wenn wir das in "sammel" gebildete Produkt als Auswertung
         eines Polynoms (x-c1)*(x-c2)*...*(x-ck) auffassen, dann
         bietet es sich an, eine Polynomauswertung zu implementieren, die
         schneller arbeitet als die k-fache Multiplikation.

         Folgende Vorgehensweise ist hier (in etwa) implementiert
          1. Nullstellenform des Polynoms ermitteln
             -> P(x)=(x-c1)*(x-c2)*...*(x-ck)
          2. P(x) ausmultiplizieren
             -> P(x)=a0 + a1*x + a2*x^2 + ... + a[k-1]*x^(k-1) + x^k
          3. k Werte (x1 ... x[k]) aufsammeln, fr die P(x) ausgewertet
             werden soll (-> k Intervallwerte von x bestimmen) 
          4. r[i]=P(x[i]) fr i=1..k bestimmen,
             z.B. mit schneller Auswertung ber
             r[i]=P(x) modulo (x-x[i]) fr i=1...k simultan auswerten.
             (-> divide and conquer)
          5. Produkt R=r1*r2*...r[k] modulo N berechnen
             und gcd(R,N) bestimmen. 
          6. Falls Schritt 5 keinen Teiler geliefert hat und die
             Obergrenze nicht erreicht ist, ab Schritt 3 fortfahren.

         Bemerkung: Um gengend viele gleiche Polynome auswerten zu knnen,
           sollte man hierbei auf Primzahltests verzichten und immer
           alle Nullstellen auswerten. (-> Nur ein Polynom bestimmen.)
           Mit asymptotisch schnellen Algorithmen sollte dies dann in etwa
           der FFT-continuation entsprechen.
      */

    }

 fertig_phase2:
  for (int j=0; j<=d_exp; ++j) { mpz_clear(xd[j]); mpz_clear(yd[j]); }
  for (unsigned int j=0; j<Polynom_Index; ++j) mpz_clear(sammel[j]);
  delete [] sammel;
  for (unsigned int j=0; j<Polynom_Index_i; ++j) mpz_clear(Polynom[j]);
  delete [] Polynom;
  cout << "Elliptic Curve Phase 2, up to " << i << endl;
  }

// end of ecm fft continuation



#endif


fertig:
  mpz_clear(x); mpz_clear(y); mpz_clear(z);
  mpz_clear(saved_x); mpz_clear(saved_y);
  mpz_clear(xh); mpz_clear(yh);
}
