#ifndef tinyvector_header
#define tinyvector_header

// Bereitstellung eines simplen Containers, der eine Datenfolge
// im dynamischen Array verwaltet.
// Hauptaufgabe: sparse-Darstellung der Faktorenindices der
// statischen Faktorbasis (fr qsieve).
// (C) 1999 by Thorsten Reinecke
// major rewrite 2003-10 by Thorsten Reinecke
// last change: 2003-10-28 by Thorsten Reinecke

/*! @file
 * @brief
 * provides some templates for constructing C++-Arrays and a tiny implementation of them 
 */


#include <limits>

//
// Basistypen fr C-Arrays, die in einem "struct" gekapselt sind
//

template <typename _Datatype, typename _Sizetype = int> class _StructArray
{
public:
 typedef _Datatype Datatype;
 typedef _Sizetype Sizetype;

 Datatype* v;		// Eigenschaft: v kann/darf durch anderen Zeiger ersetzt werden 
 Sizetype Capacity;     // Anzahl allokierter Elemente
 Sizetype akt_size;


 _StructArray(Datatype* const _v, const Sizetype _Capacity, const Sizetype _size=0)
  : v(_v), Capacity(_Capacity), akt_size(_size) { }

 // max_size gibt die maximal mgliche Anzahl an Elementen an.
 // (Die Allokierung eines greren Arrays fr "v" ist fr diese Klasse erlaubt!) 
 inline Sizetype max_size(void) const { return std::numeric_limits<Sizetype>::max(); }
};

template <typename _Datatype, typename _Sizetype = int> class _FixedStructArray
{
public:
 typedef _Datatype Datatype;
 typedef _Sizetype Sizetype;

 Datatype* const v;	   // Eigenschaft: v darf nicht mehr verndert werden
 const Sizetype Capacity;  // Anzahl allokierter Elemente (konstant!)
 Sizetype akt_size;


 _FixedStructArray(Datatype* const _v, const Sizetype _Capacity, const Sizetype _size=0)
  : v(_v), Capacity(_Capacity), akt_size(_size) { }

 // max_size gibt die maximal mgliche Anzahl an Elementen an.
 // (Die Allokierung eines greren Arrays fr "v" ist fr diese Klasse verboten!) 
 inline Sizetype max_size(void) const { return Capacity; }
};


//
// Anreicherung der Basistypen mit ntzlichen Objektfunktionen
//

//#define RANGE_CHECKS

// einige ntzliche Objektfunktionen
template <typename StructArraytype> class EnrichedArray
: protected StructArraytype
{
public:
 typedef typename StructArraytype::Datatype Datatype;
 typedef typename StructArraytype::Sizetype Sizetype;

 EnrichedArray(Datatype* const _v, const Sizetype _Capacity, const Sizetype _size=0)
  : StructArraytype(_v,_Capacity,_size) { }

#ifndef RANGE_CHECKS
 inline void set_size(const Sizetype new_size) { StructArraytype::akt_size=new_size; }
#else
 inline void set_size(const Sizetype new_size)
  {
    if (new_size>StructArraytype::Capacity || new_size<0)
     {
       cerr << "set_size: Range Overflow! " << new_size << " > " << StructArraytype::Capacity << endl;
       exit(1);
     } 
    StructArraytype::akt_size=new_size;
  }
#endif


 inline Sizetype size(void) const { return StructArraytype::akt_size; }
 inline bool empty(void) const { return (StructArraytype::akt_size==0); }
 inline Sizetype capacity(void) const { return StructArraytype::Capacity; }
 // max_size wird wegen protected-Vererbung noch einmal definiert
 inline Sizetype max_size(void) const { return StructArraytype::max_size(); }

#ifndef RANGE_CHECKS
 inline const Datatype& operator[] (const Sizetype pos) const { return StructArraytype::v[pos]; }
 inline Datatype& operator[] (const Sizetype pos) { return StructArraytype::v[pos]; }
#else
 inline const Datatype& operator[] (const Sizetype pos) const
  {
    if ( pos>capacity() || pos<0 || capacity()<1 )
     {
       cerr << "r/operator[]: Range Overflow! " << pos << " > " << StructArraytype::Capacity << endl;
       exit(1);
     } 
    return StructArraytype::v[pos];
  }
 inline Datatype& operator[] (const Sizetype pos)
  {
    if ( pos>capacity() || pos<0 || capacity()<1 )
     {
       cerr << "rw/operator[]: Range Overflow! " << pos << " > " << StructArraytype::Capacity << endl;
       exit(1);
     } 
    return StructArraytype::v[pos];
  }
#endif
};


// Konstruktor-/Destruktor, der die Arrayelemente auch wirklich allokiert/destruiert
template <typename StructArraytype> class AutofiedStructArray
: public StructArraytype
{
private:
  typedef typename StructArraytype::Datatype Datatype;
  typedef typename StructArraytype::Sizetype Sizetype;
public:
 explicit AutofiedStructArray(const Sizetype _Capacity)
  : StructArraytype(new Datatype[_Capacity],_Capacity,0) { }

 AutofiedStructArray(Datatype* const _v, const Sizetype _Capacity, const Sizetype _size=0)
  : StructArraytype(_v,_Capacity,_size) { }

 ~AutofiedStructArray() { delete [] AutofiedStructArray::v; }
};



// macro for template composition analogous to composition of mathematical functions: g(f(X))=(gof)(X)
// "Composed_"-template (f) is composed by the "Composer_"-template (g);
// the composition of these templates results in the "Compositum_" (gof).
// example: B<A<T1,T2> >  --TemplateComposition--> C<T1,T2> (accessible as C<T1,T2>::Type)
#define TemplateComposition(Composer_, Composed_, Compositum_) \
template <typename T1, typename T2 = int> struct Compositum_ \
{ \
  typedef Composer_ < Composed_ <T1,T2> >  Type; \
};


TemplateComposition(EnrichedArray,_StructArray,StructArray)
TemplateComposition(EnrichedArray,_FixedStructArray,FixedStructArray)


#define TemplateComposition2(Composer_, Composed1_, Composed2_, Compositum_) \
template <typename T1, typename T2 = int> struct Compositum_ \
{ \
  typedef Composer_ < Composed1_ < Composed2_ <T1,T2> > >  Type; \
};

// zweistufige Komposition: h(g(f(X)))=(hogof)(X)
TemplateComposition2(AutofiedStructArray,EnrichedArray,_StructArray,AutoStructArray)
TemplateComposition2(AutofiedStructArray,EnrichedArray,_FixedStructArray,AutoFixedStructArray)



template <typename Datatype, typename Sizetype=int> class CTinyVector
: public AutoStructArray<Datatype,Sizetype>::Type
{
  typedef typename AutoStructArray<Datatype,Sizetype>::Type Ancestor;
public:
  inline explicit CTinyVector(const Sizetype MaxSize = 20) // Konstruktor
   : Ancestor(new Datatype [MaxSize],MaxSize) { };

  inline CTinyVector(const Datatype* Feld, const Sizetype Size)
   : Ancestor(Size)
    // Konstruktor: mit vorgegebenem Feld initialisieren
    {
      for(Sizetype i=0; i<Size; i++) Ancestor::v[i]=Feld[i];
      Ancestor::akt_size=Size;
    }
  inline CTinyVector(const CTinyVector<Datatype>& tv)
    // Konstruktor: mit anderem TinyVector initialisieren
   : Ancestor(tv.size())
    {
      Ancestor::akt_size=tv.akt_size;
      for(Sizetype i=0; i<Ancestor::akt_size; i++) Ancestor::v[i]=tv[i];
    }
  inline void reset(void)
    {
      Ancestor::akt_size=0;
    }
  inline void resize(const Sizetype new_size)
    {
      if (new_size<Ancestor::akt_size)
	{
          cerr << "Tiny-Vector: Resizing not possible!" << endl;
          exit(1);
        }
      if (new_size==Ancestor::capacity()) return; // nothing to do...
      Datatype* vnew = new Datatype [new_size];
      for (Sizetype i=0; i<Ancestor::akt_size; i++) vnew[i]=Ancestor::v[i];
      delete [] Ancestor::v; Ancestor::v=vnew; Ancestor::Capacity=new_size;
    }
  inline void optisize(void) // Gre des dyn. Array optimieren
    {
      resize(Ancestor::akt_size);
    }
  inline void append(const Datatype value)
    {
      if (Ancestor::akt_size>=Ancestor::Capacity) 
	{
	  cerr << "TinyVector: Index " << Ancestor::akt_size
	       << " is out Of Range!"
	       << " Auto-Resizing by 5" << endl;
	  resize(Ancestor::akt_size+5);
	}
      if (!Ancestor::empty() && value<=Ancestor::v[Ancestor::akt_size-1])
	{ cerr << "TinyVector: unsortiertes append!" << endl; } 
      Ancestor::v[Ancestor::akt_size++]=value; 
    }
  inline void fast_append(const Datatype value)
    {
      Ancestor::v[Ancestor::akt_size++]=value;
    }
  inline Datatype last(void) const
    {
      if (!Ancestor::empty()) return Ancestor::v[Ancestor::akt_size-1];
      else
	{
	  cerr << "TinyVector: no last Element!" << endl;
	  return -1;
	}
    }

  inline void copy_from(const Datatype* Feld, const Sizetype Size)
   {
     if (Size>Ancestor::capacity()) { delete [] Ancestor::v; Ancestor::v = new Datatype [Size]; Ancestor::Capacity=Size; }
     Ancestor::akt_size=Size;
     for(Sizetype i=0; i<Size; i++) Ancestor::v[i]=Feld[i];
   }
  inline void copy_from(const CTinyVector<Datatype> &tv)
   {
     copy_from(tv.v,tv.size());
   }
  inline void optisized_copy_from(const Datatype* Feld, const Sizetype Size)
   {
     if (Size!=Ancestor::capacity()) { delete [] Ancestor::v; Ancestor::v = new Datatype [Size]; }
     Ancestor::akt_size=Ancestor::Capacity=Size;
     for(Sizetype i=0; i<Size; i++) Ancestor::v[i]=Feld[i];
   }
  inline void optisized_copy_from(const CTinyVector<Datatype> &tv)
   {
     if (this==&tv) { optisize(); return; } // optimize copy from myself!
     optisized_copy_from(tv.v,tv.size());
   }
  inline const CTinyVector<Datatype>& operator= (const CTinyVector<Datatype> &tv)
   {
     copy_from(tv.v,tv.size()); return *this;
   }

};

#endif
