// mpz-Zahl quadrieren
// written by Thorsten Reinecke 1999
// last change: 13.11.1999


// Quadrierung nach folgendem Prinzip:
// 1. x wird in lo/hi gesplittet: x=a+2^h+b
// 2. x^2=(a*2^h+b)^2=2^(2*h)*a+2*2^h*a*b+b^2)
//    und wegen (a+b)^2=a^2+2ab+b^2 gilt 2ab=(a+b)^2-a^2-b^2
//    also x^2 = 2^(2*h)*a^2 + 2^h*((a+b)^2-a^2-b^2) - b^2
//    alternativ: (a-b)^2=a^2-2ab+b^2 -> 2ab=a^2+b^2-(a-b)^2,
//    also x^2 = 2^(2*h)*a^2 - b^2 + 2^h*(a^2+b^2-(a-b)^2)
// 3. rekursives Auswerten der Terme a^2, b^2 sowie (a+b)^2 bzw. (a-b)^2,
//    anschlieend Term nach obiger Formel zusammensetzen.

// Kosten bzgl. O-Notation (vereinfacht kalkuliert):
// Die Quadrierung einer Zahl mit n Bits kostet 3 Quadrierungen von
// Zahlen der Grenordnung n/2 Bits. Das Verfahren terminiert (sptestens)
// bei einer 1-Bit Zahl, da wir dann 0^2=0 bzw. 1^2=1 direkt setzen knnen.
// Wir bentigen zum Abstieg also insgesamt 3^ld(n) Funktionsaufrufe.
// 3^ld(n)=(2^ld(3))^(ld(n))=2^(ld(3)*ld(n))=(2^ld(n))^ld(3)=n^ld(3).
// Die Kosten betragen also etwa O(n^1.59). [Man beachte, da der Exponent
// in der O-Notation wegen der Teilmengeninklusion der O-Klassen immer auf
// die nchsthhere letzte gltige Dezimalstelle aufgerundet werden mu!]
// (Fr eine korrekte Analyse mssen wir die Kosten fr den Zusammenbau des
// Terms bercksichtigen (denn eine Rekursionsfunktion kann ja kaum in einer
// niedrigeren O-Klasse liegen als z.B. eine Teilfunktion, die sie im Laufe
// der Rekursion aufruft); glcklicherweise besteht jeder Schritt aus einer
// konstanten Anzahl von Operationen, die maximal lineare Laufzeit besitzen.
// Fr einen formalen Beweis reichen freilich auch diese Argumente nicht
// aus.)

// -> siehe auch Multiplikationsverfahren nach Karatsuba

// Bzgl. O-Notation mssen Quadrierung und Multiplikation in den selben
// Klassen liegen, denn fr die Multiplikation gilt a*b=((a+b)^2-(a-b)^2)/4,
// und die doppelte Auswertung geht in der O-Notation als Konstante unter.

// Nebenbei: Fr die O-Notation uninteressant, aber bei Speicherplatzmangel
// und "kleinen" Werten kann die Multiplikation durch eine Tabelle mit 
// Quadraten anstelle einer "quadratischen" Tabelle realisiert werden,
// wenn z.B. hardwaremig keine Multiplikation realisiert ist.


///////////////////////////////////////////////////////////////////////////

/* Bemerkung:
   In einer effizienten Implementierung wrde man anstatt shift-Operationen
   zu verwenden, die Bitpositionen der Zahl direkt adressieren.
   Auerdem wrde man anstelle einer Binrbasis eine 2-er-Potenz als Basis
   verwenden (also Maschinenwrter).
*/

// Dies hier ist KEINE schnelle Implementierung!
/*
   Sie sollte auch nicht verwendet werden, da der Karatsuba-Algorithmus
   in der GMP-Library bereits effizienter umgesetzt ist.
   --------------------------------------------------------------------
   This implementation of "mpz_square" is not an efficient one!
   Do not use it except for educational purpose!
   "mpz_mul" (GMP-Library) is much faster!
*/ 
   

void mpz_square(mpz_t &res, mpz_t &x)
{
  mpz_abs(x,x);

  int h = mpz_sizeinbase(x,2);
  if (h<24)
   {
     // wir knnen in double rechnen...
     double s = mpz_get_d(x);
     mpz_set_d(res,s*s);
     return;
   }
  else
   {
     mpz_t a,b;
     mpz_init(a); mpz_init(b);
     h/=2;

     // splitte x, so da a*2^h+b=x
     mpz_fdiv_q_2exp(a,x,h);
     mpz_fdiv_r_2exp(b,x,h);

     mpz_add(res,a,b); // res=a+b
     mpz_square(a,a); mpz_square(res,res); 
     mpz_sub(res,res,a); mpz_mul_2exp(a,a,h); mpz_add(res,res,a);
     mpz_clear(a); mpz_square(b,b); 
     mpz_sub(res,res,b); mpz_mul_2exp(res,res,h); mpz_add(res,res,b);
     mpz_clear(b);
   }
}
